/* queue_valid.c -- Queues requests to server

   Copyright (c) Dave Airlie 2000
   airlied@samba.org

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "config.h"
#ifdef PAM_SMB_DAEMON
#include <sys/types.h>
#include <sys/ipc.h>
#if !defined(__APPLE__)
#include <sys/msg.h>
#else
#define EIDRM EINVAL
#define ENOMSG EAGAIN
#endif
#include <unistd.h>
#include <syslog.h>
#include <time.h>
#include <string.h>
#include <sys/time.h>
#include "smblib.h"
#include "constants.h"
#include "pam_smb_queue.h"

#include <errno.h>

#define safestrcpy(s1, s2, n) strncpy(s1, s2, n); ((char *)s1)[n-1] = 0

/*
 * Queues a request to the server
 */
int queue_user(char *username,char *password, char *domain, int cachetime, int failcachetime, int debug) { 

	struct timeval basetime;
	key_t key=PAMSMBKEY;
	pid_t cli_pid;
	int mid, n;
	MESSAGE msg;
	time_t start_time, now_time;

	cli_pid = getpid();

	if ((mid=msgget(key, 0)) == -1) {
      syslog(LOG_AUTHPRIV | LOG_ERR, "pamsmbd is not running, no authentication possible");
      return 2;
    }

	msg.msg_to=SERVER_ID;
	msg.msg_fm=cli_pid;
	msg.return_code=2;

	safestrcpy(msg.username, username, MAX_NT_LEN);
	safestrcpy(msg.password, password, MAX_PASS_LEN);
	safestrcpy(msg.domain, domain, MAX_NTDOM_LEN);
	msg.cachetime = cachetime;
	msg.failcachetime = failcachetime;
	msg.debug = debug;

	gettimeofday(&basetime,NULL);
	memcpy(&msg.time, &basetime, sizeof(basetime));

	if(msgsnd(mid, (struct msgbuf *)&msg, sizeof(msg),  0) == -1) { 
		syslog(LOG_AUTHPRIV | LOG_ERR, "pamsmbd : msg_snd problem");
		return 2;
	}

	start_time=time(NULL);
  
	for(;;) {
 	        alarm(PAMSMB_SERVERTIMEOUT);
   	        n=msgrcv(mid,(struct msgbuf *)&msg, sizeof(msg), cli_pid, 0); 
		alarm(0);
		if(n!=-1) {  
	 		if(memcmp(&msg.time, &basetime, sizeof(basetime)) == 0) {
				if (debug)
					syslog(LOG_AUTHPRIV | LOG_DEBUG, "pamsmbd: Got something back... %d", msg.return_code); 
				safestrcpy(username, msg.username, MAX_NT_LEN);
				return(msg.return_code);
			} else {
				/* In the event that the PID is the same but the request  * is not...*/ 
				if ((long)msg.time.tv_sec < ((long)basetime.tv_sec +
					(long)(PAMSMB_SERVERTIMEOUT + 1))) {
					/* we need to put it back on the queue, just in case  * another 
					* thread is listening for this */
					syslog(LOG_AUTHPRIV | LOG_ERR, "pamsmbd: Not my message, re-enqueing");   
					if(msgsnd(mid, (struct msgbuf *)&msg, sizeof(msg),  0) == -1) { 
						syslog(LOG_AUTHPRIV | LOG_ERR, "pamsmbd : msg_snd problem in resend"); 
						return 2;
					}
				}
				syslog(LOG_AUTHPRIV | LOG_ERR, "pamsmbd: Stale Message on queue - destroying"); 

				/* else ... it is a stale/timed out msg ... thus we drop
				* it by not placing it back on the queue */ 
			}
		} else { 
			switch(errno) {
				case EIDRM:
					syslog(LOG_AUTHPRIV | LOG_ERR, "pamsmbm: server finished..\n");
					return 2;
					break;
				case EINTR:
					syslog(LOG_AUTHPRIV | LOG_ERR, "pamsmbm: caught signal..\n");
				case ENOENT:
				case ENOMSG:
					now_time=time(NULL);
					if ((now_time-start_time)>PAMSMB_SERVERTIMEOUT) {
						syslog(LOG_AUTHPRIV | LOG_ERR, "pamsmbm: no server...\n");
						return 2;
		  			}
					break;
				default:
					syslog(LOG_AUTHPRIV | LOG_ERR,
						"pamsmbm: msgrcv failed with errno %d...\n",
						errno);
					return 2;
			       break;
			}
		}
	}
}

#else
void dummy_function_queue_valid(void) { } /* stops some compilers complaining */
#endif
