/****************************************************************
Copyright (C) Lucent Technologies 1997
All Rights Reserved

Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby
granted, provided that the above copyright notice appear in all
copies and that both that the copyright notice and this
permission notice and warranty disclaimer appear in supporting
documentation, and that the name Lucent Technologies or any of
its entities not be used in advertising or publicity pertaining
to distribution of the software without specific, written prior
permission.

LUCENT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
IN NO EVENT SHALL LUCENT OR ANY OF ITS ENTITIES BE LIABLE FOR ANY
SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
THIS SOFTWARE.
****************************************************************/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "awklib.h"
#include "awkgram.h"
#include "proto.h"

typedef struct Keyword {
	const char *word;
	int	sub;
	int	type;
} Keyword;

static int peek(awk_t *);
static int gettok(awk_t *, char **, int *);
static int binsearch(const char *, const Keyword *, int);

const Keyword keywords[] ={	/* keep sorted: binary searched */
	{ "BEGIN",	XBEGIN,		XBEGIN },
	{ "END",	XEND,		XEND },
	{ "NF",		VARNF,		VARNF },
	{ "atan2",	FATAN,		BLTIN },
	{ "break",	BREAK,		BREAK },
	{ "close",	CLOSE,		CLOSE },
	{ "continue",	CONTINUE,	CONTINUE },
	{ "cos",	FCOS,		BLTIN },
	{ "delete",	DELETE,		DELETE },
	{ "dlalloc",	DLALLOC,	DLALLOC },
	{ "dlcall",	DLCALL,		DLCALL },
	{ "dlclose",	DLCLOSE,	DLCLOSE },
	{ "dlfield",	DLFIELD,	DLFIELD },
	{ "dlfree",	DLFREE,		DLFREE },
	{ "dlopen",	DLOPEN,		DLOPEN },
	{ "dlproto",	DLPROTO,	DLPROTO },
	{ "do",		DO,		DO },
	{ "else",	ELSE,		ELSE },
	{ "exit",	EXIT,		EXIT },
	{ "exp",	FEXP,		BLTIN },
	{ "fflush",	FFLUSH,		BLTIN },
	{ "for",	FOR,		FOR },
	{ "func",	FUNC,		FUNC },
	{ "function",	FUNC,		FUNC },
	{ "gensub",	GENSUB,		GENSUB },
	{ "getline",	GETLINE,	GETLINE },
	{ "gsub",	GSUB,		GSUB },
	{ "if",		IF,		IF },
	{ "in",		IN,		IN },
	{ "index",	INDEX,		INDEX },
	{ "int",	FINT,		BLTIN },
	{ "length",	FLENGTH,	BLTIN },
	{ "log",	FLOG,		BLTIN },
	{ "match",	MATCHFCN,	MATCHFCN },
	{ "next",	NEXT,		NEXT },
	{ "nextfile",	NEXTFILE,	NEXTFILE },
	{ "print",	PRINT,		PRINT },
	{ "printf",	PRINTF,		PRINTF },
	{ "rand",	FRAND,		BLTIN },
	{ "return",	RETURN,		RETURN },
	{ "shift",	SHIFTARGS,	SHIFTARGS },
	{ "sin",	FSIN,		BLTIN },
	{ "split",	SPLIT,		SPLIT },
	{ "sprintf",	SPRINTF,	SPRINTF },
	{ "sqrt",	FSQRT,		BLTIN },
	{ "srand",	FSRAND,		BLTIN },
	{ "strftime",	FSTRFTIME,	BLTIN },
	{ "sub",	SUB,		SUB },
	{ "substr",	SUBSTR,		SUBSTR },
	{ "system",	FSYSTEM,	BLTIN },
	{ "systime",	FSYSTIME,	BLTIN },
	{ "tolower",	FTOLOWER,	BLTIN },
	{ "toupper",	FTOUPPER,	BLTIN },
	{ "while",	WHILE,		WHILE },
};

#define	RET(x)	do {							\
	if(awkp->dbg) {							\
		printf("lex %s\n", awklib_tokname(x));			\
	}								\
	return(x);							\
} while (/* CONSTCOND */ 0)

static int
peek(awk_t *awkp)
{
	int c = input(awkp);

	unput(awkp, c);
	return c;
}

/* get next input token */
static int
gettok(awk_t *awkp, char **pbuf, int *psz)
{
	int c, retc;
	uint8_t *buf = (uint8_t *) *pbuf;
	int sz = *psz;
	uint8_t *bp = buf;

	c = input(awkp);
	if (c == 0)
		return 0;
	buf[0] = c;
	buf[1] = 0;
	if (!isalnum(c) && c != '.' && c != '_')
		return c;

	*bp++ = c;
	if (isalpha(c) || c == '_') {	/* it's a varname */
		for ( ; (c = input(awkp)) != 0; ) {
			if (bp-buf >= sz)
				if (!awklib_adjbuf(awkp, &buf, &sz, bp-buf+2, 100, &bp, "gettok"))
					FATAL(awkp,  "out of space for name %.10s...", buf );
			if (isalnum(c) || c == '_')
				*bp++ = c;
			else {
				*bp = 0;
				unput(awkp, c);
				break;
			}
		}
		*bp = 0;
		retc = 'a';	/* alphanumeric */
	} else {	/* maybe it's a number, but could be . */
		char *rem;
		/* read input until can't be a number */
		for ( ; (c = input(awkp)) != 0; ) {
			if (bp-buf >= sz)
				if (!awklib_adjbuf(awkp, &buf, &sz, bp-buf+2, 100, &bp, "gettok"))
					FATAL(awkp,  "out of space for number %.10s...", buf );
			if (isdigit(c) || c == 'e' || c == 'E' 
			  || c == '.' || c == '+' || c == '-')
				*bp++ = c;
			else {
				unput(awkp, c);
				break;
			}
		}
		*bp = 0;
		strtod((char *)buf, &rem);	/* parse the number */
		if (rem == (char *)buf) {	/* it wasn't a valid number at all */
			buf[1] = 0;	/* return one character as token */
			retc = buf[0];	/* character is its own type */
			unputstr(awkp, rem+1); /* put rest back for later */
		} else {	/* some prefix was a number */
			unputstr(awkp, rem);	/* put rest back for later */
			rem[0] = 0;	/* truncate buf after number part */
			retc = '0';	/* type is number */
		}
	}
	*pbuf = (char *)buf;
	*psz = sz;
	return retc;
}

static int	word(char *, YYSTYPE *, awk_t *);
static int	string(YYSTYPE *, awk_t *);
static int	regexpr(YYSTYPE *, awk_t *);

static int
string(YYSTYPE *lvalp, awk_t *awkp)
{
	int c, n;
	uint8_t *s, *bp;

	if (awkp->stringbuf == NULL && (awkp->stringbuf = malloc(awkp->stringbufsz)) == NULL)
		FATAL(awkp, "out of space for strings");
	for (bp = awkp->stringbuf; (c = input(awkp)) != '"'; ) {
		if (!awklib_adjbuf(awkp, &awkp->stringbuf, &awkp->stringbufsz, (int)(bp - awkp->stringbuf+2), 500, &bp, "string"))
			FATAL(awkp, "out of space for string %.10s...", awkp->stringbuf);
		switch (c) {
		case '\n':
		case '\r':
		case 0:
			SYNTAX(awkp,  "non-terminated string %.10s...", awkp->stringbuf );
			awkp->lineno++;
			if (c == 0)	/* hopeless */
				FATAL(awkp,  "giving up" );
			break;
		case '\\':
			c = input(awkp);
			switch (c) {
			case '\n': break;
			case '"': *bp++ = '"'; break;
			case 'n': *bp++ = '\n'; break;	
			case 't': *bp++ = '\t'; break;
			case 'f': *bp++ = '\f'; break;
			case 'r': *bp++ = '\r'; break;
			case 'b': *bp++ = '\b'; break;
			case 'v': *bp++ = '\v'; break;
			case 'a': *bp++ = '\007'; break;
			case '\\': *bp++ = '\\'; break;

			case '0': case '1': case '2': /* octal: \d \dd \ddd */
			case '3': case '4': case '5': case '6': case '7':
				n = c - '0';
				if ((c = peek(awkp)) >= '0' && c < '8') {
					n = 8 * n + input(awkp) - '0';
					if ((c = peek(awkp)) >= '0' && c < '8')
						n = 8 * n + input(awkp) - '0';
				}
				*bp++ = n;
				break;

			case 'x':	/* hex  \x0-9a-fA-F + */
			    {	char xbuf[100], *px;
				for (px = xbuf; (c = input(awkp)) != 0 && px-xbuf < 100-2; ) {
					if (isdigit(c)
					 || (c >= 'a' && c <= 'f')
					 || (c >= 'A' && c <= 'F'))
						*px++ = c;
					else
						break;
				}
				*px = 0;
				unput(awkp, c);
	  			sscanf(xbuf, "%x", &n);
				*bp++ = n;
				break;
			    }

			default:
				WARNING(awkp, "warning: escape sequence `\\%c' "
				    "treated as plain `%c'", c, c);
				*bp++ = c;
				break;
			}
			break;
		default:
			*bp++ = c;
			break;
		}
	}
	*bp = 0; 
	s = (uint8_t *)awklib_tostring(awkp, (char *)awkp->stringbuf);
	*bp++ = ' '; *bp++ = 0;
	(*lvalp).cp = setawkvar(awkp, (char *)awkp->stringbuf, (char *)s, 0.0, CON|STR|DONTFREE, awkp->symtab);
	RET(STRING);
}


static int
binsearch(const char *w, const Keyword *kp, int n)
{
	int cond, low, mid, high;

	low = 0;
	high = n - 1;
	while (low <= high) {
		mid = (low + high) / 2;
		if ((cond = strcmp(w, kp[mid].word)) < 0)
			high = mid - 1;
		else if (cond > 0)
			low = mid + 1;
		else
			return mid;
	}
	return -1;
}

static int
word(char *w, YYSTYPE *lvalp, awk_t *awkp) 
{
	const Keyword *kp;
	int c, n;

	n = binsearch(w, keywords, sizeof(keywords)/sizeof(keywords[0]));
/* BUG: this ought to be inside the if; in theory could fault (daniel barrett) */
	kp = keywords + n;
	if (n != -1) {	/* found in table */
		(*lvalp).i = kp->sub;
		switch (kp->type) {	/* special handling */
		case BLTIN:
			if (kp->sub == FSYSTEM && awkp->safe)
				SYNTAX(awkp,  "system is unsafe" );
			RET(kp->type);
		case FUNC:
			if (awkp->infunc)
				SYNTAX(awkp,  "illegal nested function" );
			RET(kp->type);
		case RETURN:
			if (!awkp->infunc)
				SYNTAX(awkp,  "return not in function" );
			RET(kp->type);
		case VARNF:
			(*lvalp).cp = setawkvar(awkp, "NF", "", 0.0, NUM, awkp->symtab);
			RET(VARNF);
		default:
			RET(kp->type);
		}
	}
	c = peek(awkp);	/* look for '(' */
	if (c != '(' && awkp->infunc && (n=awklib_isarg(awkp, w)) >= 0) {
		(*lvalp).i = n;
		RET(ARG);
	} else {
		(*lvalp).cp = setawkvar(awkp, w, "", 0.0, STR|NUM|DONTFREE, awkp->symtab);
		if (c == '(') {
			RET(CALL);
		} else {
			RET(VAR);
		}
	}
}

/* next call to awklib_lex will return a regular expression */
void
startreg(awk_t *awkp)
{
	awkp->reg = 1;
}

static int
regexpr(YYSTYPE *lvalp, awk_t *awkp)
{
	int c;
	uint8_t *bp;

	if (awkp->regexprbuf == 0 && (awkp->regexprbuf = malloc(awkp->regexprbufsz)) == NULL)
		FATAL(awkp, "out of space for rex expr");
	bp = awkp->regexprbuf;
	for ( ; (c = input(awkp)) != '/' && c != 0; ) {
		if (!awklib_adjbuf(awkp, &awkp->regexprbuf, &awkp->regexprbufsz, (int)(bp - awkp->regexprbuf+3), 500, &bp, "regexpr"))
			FATAL(awkp, "out of space for reg expr %.10s...", awkp->regexprbuf);
		if (c == '\n') {
			SYNTAX(awkp,  "newline in regular expression %.10s...", awkp->regexprbuf ); 
			unput(awkp, '\n');
			break;
		} else if (c == '\\') {
			*bp++ = '\\'; 
			*bp++ = input(awkp);
		} else {
			*bp++ = c;
		}
	}
	*bp = 0;
	if (c == 0)
		SYNTAX(awkp, "non-terminated regular expression %.10s...", awkp->regexprbuf);
	(*lvalp).s = awklib_tostring(awkp, (char *)awkp->regexprbuf);
	unput(awkp, '/');
	RET(REGEXPR);
}

/* low-level lexical stuff, sort of inherited from lex */

static int
pgetc(awk_t *awkp)		/* get 1 character from awk program */
{
	int c;

	for (;;) {
		if (awkp->yyin == NULL) {
			if (awkp->curscript >= awkp->scriptc)
				return EOF;
			if (strcmp(awkp->scripts[awkp->curscript], "-") == 0)
				awkp->yyin = stdin;
			else if ((awkp->yyin = fopen(awkp->scripts[awkp->curscript], "r")) == NULL)
				FATAL(awkp, "can't open file %s", awkp->scripts[awkp->curscript]);
			awkp->lineno = 1;
		}
		if ((c = getc(awkp->yyin)) != EOF)
			return c;
		if (awkp->yyin != stdin)
			fclose(awkp->yyin);
		awkp->yyin = NULL;
		awkp->curscript++;
	}
}

int
input(awk_t *awkp)	/* get next lexical input character */
{
	int c;

	if (awkp->yysptr > awkp->yysbuf)
		c = (uint8_t)*--awkp->yysptr;
	else if (awkp->lexprog != NULL) {	/* awk '...' */
		if ((c = (uint8_t)*awkp->lexprog) != 0)
			awkp->lexprog++;
	} else				/* awk -f ... */
		c = pgetc(awkp);
	if (c == '\n')
		awkp->lineno++;
	else if (c == EOF)
		c = 0;
	if (awkp->ep >= awkp->ebuf + sizeof(awkp->ebuf))
		awkp->ep = awkp->ebuf;
	return *awkp->ep++ = c;
}

void
unput(awk_t *awkp, int c)	/* put lexical character back on input */
{
	if (c == '\n')
		awkp->lineno--;
	if (awkp->yysptr >= awkp->yysbuf + sizeof(awkp->yysbuf))
		FATAL(awkp, "pushed back too much: %.20s...", awkp->yysbuf);
	*awkp->yysptr++ = c;
	if (--awkp->ep < awkp->ebuf)
		awkp->ep = awkp->ebuf + sizeof(awkp->ebuf) - 1;
}

void
unputstr(awk_t *awkp, const char *s)	/* put a string back on input */
{
	int i;

	for (i = strlen(s)-1; i >= 0; i--)
		unput(awkp, s[i]);
}

int
awklib_lex(void *lval, void *vp)
{
	YYSTYPE	*lvalp = (YYSTYPE *)lval;
	awk_t	*awkp = (awk_t *)vp;
	int c;

	if (awkp->lexbuf == NULL && (awkp->lexbuf = malloc(awkp->lexbufsize)) == NULL)
		FATAL(awkp,  "out of space in awklib_lex" );
	if (awkp->sc) {
		awkp->sc = 0;
		RET('}');
	}
	if (awkp->reg) {
		awkp->reg = 0;
		return regexpr(lvalp, awkp);
	}
	for (;;) {
		c = gettok(awkp, &awkp->lexbuf, &awkp->lexbufsize);
		if (c == 0)
			return 0;
		if (isalpha(c) || c == '_')
			return word(awkp->lexbuf, lvalp, awkp);
		if (isdigit(c)) {
			(*lvalp).cp = setawkvar(awkp, awkp->lexbuf, awklib_tostring(awkp, awkp->lexbuf), atof(awkp->lexbuf), CON|NUM, awkp->symtab);
			/* should this also have STR set? */
			RET(NUMBER);
		}
	
		(*lvalp).i = c;
		switch (c) {
		case '\n':	/* {EOL} */
			RET(NL);
		case '\r':	/* assume \n is coming */
		case ' ':	/* {WS}+ */
		case '\t':
			break;
		case '#':	/* #.* strip comments */
			while ((c = input(awkp)) != '\n' && c != 0)
				;
			unput(awkp, c);
			break;
		case ';':
			RET(';');
		case '\\':
			if (peek(awkp) == '\n') {
				input(awkp);
			} else if (peek(awkp) == '\r') {
				input(awkp); input(awkp);	/* \n */
				awkp->lineno++;
			} else {
				RET(c);
			}
			break;
		case '&':
			if (peek(awkp) == '&') {
				input(awkp); RET(AND);
			} else 
				RET('&');
		case '|':
			if (peek(awkp) == '|') {
				input(awkp); RET(BOR);
			} else
				RET('|');
		case '!':
			if (peek(awkp) == '=') {
				input(awkp); (*lvalp).i = NE; RET(NE);
			} else if (peek(awkp) == '~') {
				input(awkp); (*lvalp).i = NOTMATCH; RET(MATCHOP);
			} else
				RET(NOT);
		case '~':
			(*lvalp).i = MATCH;
			RET(MATCHOP);
		case '<':
			if (peek(awkp) == '=') {
				input(awkp); (*lvalp).i = LE; RET(LE);
			} else {
				(*lvalp).i = LT; RET(LT);
			}
		case '=':
			if (peek(awkp) == '=') {
				input(awkp); (*lvalp).i = EQ; RET(EQ);
			} else {
				(*lvalp).i = ASSIGN; RET(ASGNOP);
			}
		case '>':
			if (peek(awkp) == '=') {
				input(awkp); (*lvalp).i = GE; RET(GE);
			} else if (peek(awkp) == '>') {
				input(awkp); (*lvalp).i = APPEND; RET(APPEND);
			} else {
				(*lvalp).i = GT; RET(GT);
			}
		case '+':
			if (peek(awkp) == '+') {
				input(awkp); (*lvalp).i = INCR; RET(INCR);
			} else if (peek(awkp) == '=') {
				input(awkp); (*lvalp).i = ADDEQ; RET(ASGNOP);
			} else
				RET('+');
		case '-':
			if (peek(awkp) == '-') {
				input(awkp); (*lvalp).i = DECR; RET(DECR);
			} else if (peek(awkp) == '=') {
				input(awkp); (*lvalp).i = SUBEQ; RET(ASGNOP);
			} else
				RET('-');
		case '*':
			if (peek(awkp) == '=') {	/* *= */
				input(awkp); (*lvalp).i = MULTEQ; RET(ASGNOP);
			} else if (peek(awkp) == '*') {	/* ** or **= */
				input(awkp);	/* eat 2nd * */
				if (peek(awkp) == '=') {
					input(awkp); (*lvalp).i = POWEQ; RET(ASGNOP);
				} else {
					RET(POWER);
				}
			} else
				RET('*');
		case '/':
			RET('/');
		case '%':
			if (peek(awkp) == '=') {
				input(awkp); (*lvalp).i = MODEQ; RET(ASGNOP);
			} else
				RET('%');
		case '^':
			if (peek(awkp) == '=') {
				input(awkp); (*lvalp).i = POWEQ; RET(ASGNOP);
			} else
				RET(POWER);

		case '$':
			/* BUG: awkward, if not wrong */
			c = gettok(awkp, &awkp->lexbuf, &awkp->lexbufsize);
			if (isalpha(c)) {
				if (strcmp(awkp->lexbuf, "NF") == 0) {	/* very special */
					unputstr(awkp, "(NF)");
					RET(INDIRECT);
				}
				c = peek(awkp);
				if (c == '(' || c == '[' || (awkp->infunc && awklib_isarg(awkp, awkp->lexbuf) >= 0)) {
					unputstr(awkp, awkp->lexbuf);
					RET(INDIRECT);
				}
				(*lvalp).cp = setawkvar(awkp, awkp->lexbuf, "", 0.0, STR|NUM, awkp->symtab);
				RET(IVAR);
			} else if (c == 0) {	/*  */
				SYNTAX(awkp,  "unexpected end of input after $" );
				RET(';');
			} else {
				unputstr(awkp, awkp->lexbuf);
				RET(INDIRECT);
			}
	
		case '}':
			if (--awkp->bracecnt < 0)
				SYNTAX(awkp,  "extra }" );
			awkp->sc = 1;
			RET(';');
		case ']':
			if (--awkp->brackcnt < 0)
				SYNTAX(awkp,  "extra ]" );
			RET(']');
		case ')':
			if (--awkp->parencnt < 0)
				SYNTAX(awkp,  "extra )" );
			RET(')');
		case '{':
			awkp->bracecnt++;
			RET('{');
		case '[':
			awkp->brackcnt++;
			RET('[');
		case '(':
			awkp->parencnt++;
			RET('(');
	
		case '"':
			return string(lvalp, awkp);	/* BUG: should be like tran.c ? */
	
		default:
			RET(c);
		}
	}
}
