/*
 *
 * MariaDB Client for Java
 *
 * Copyright (c) 2012-2014 Monty Program Ab.
 * Copyright (c) 2015-2020 MariaDB Corporation Ab.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along
 * with this library; if not, write to Monty Program Ab info@montyprogram.com.
 *
 * This particular MariaDB Client for Java file is work
 * derived from a Drizzle-JDBC. Drizzle-JDBC file which is covered by subject to
 * the following copyright and notice provisions:
 *
 * Copyright (c) 2009-2011, Marcus Eriksson
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * Redistributions of source code must retain the above copyright notice, this list
 * of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright notice, this
 * list of conditions and the following disclaimer in the documentation and/or
 * other materials provided with the distribution.
 *
 * Neither the name of the driver nor the names of its contributors may not be
 * used to endorse or promote products derived from this software without specific
 * prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS  AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 */

package org.mariadb.jdbc;

import static org.junit.Assert.*;

import java.math.BigInteger;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

public class GeometryTest extends BaseTest {

  private static final char[] hexCode = "0123456789ABCDEF".toCharArray();

  @BeforeClass()
  public static void initClass() throws SQLException {
    drop();
    try (Statement stmt = sharedConnection.createStatement()) {
      stmt.execute("CREATE TABLE geom_test(g geometry)");
      stmt.execute("FLUSH TABLES");
    }
  }

  @AfterClass
  public static void drop() throws SQLException {
    try (Statement stmt = sharedConnection.createStatement()) {
      stmt.execute("DROP TABLE IF EXISTS geom_test");
    }
  }

  private void geometryTest(String geometryString, String geometryBinary) throws SQLException {
    try (Statement stmt = sharedConnection.createStatement()) {
      stmt.execute("TRUNCATE geom_test");
      String prefix = "";
      if (!isMariadbServer() && minVersion(8, 0, 3)) {
        prefix = "ST_";
      }

      String tmpGeometryBinary = geometryBinary;
      if (tmpGeometryBinary == null) {
        String sql =
            "SELECT " + prefix + "AsWKB(" + prefix + "GeomFromText('" + geometryString + "'))";
        try (ResultSet rs = stmt.executeQuery(sql)) {
          assertTrue(rs.next());
          tmpGeometryBinary = printHexBinary(rs.getBytes(1));
        }
      }
      String sql =
          "INSERT INTO geom_test VALUES (" + prefix + "GeomFromText('" + geometryString + "'))";
      stmt.execute(sql);
      try (ResultSet rs =
          stmt.executeQuery(
              "SELECT " + prefix + "AsText(g), " + prefix + "AsBinary(g), g FROM geom_test")) {
        assertTrue(rs.next());
        // as text
        assertEquals(geometryString, rs.getString(1));
        // as binary
        String returnWkb = printHexBinary((byte[]) rs.getObject(2));
        assertEquals(tmpGeometryBinary, returnWkb);
        // as object
        Object geometry = null;
        try {
          geometry = rs.getObject(3);
        } catch (Exception e) {
          fail();
        }
        String returnGeometry = printHexBinary((byte[]) geometry);
        BigInteger returnNumber = new BigInteger(returnGeometry, 16);
        BigInteger geometryNumber = new BigInteger(tmpGeometryBinary, 16);
        assertEquals(geometryNumber, returnNumber);
      }
    }
  }

  @Test
  public void pointTest() throws SQLException {
    String pointString = "POINT(1 1)";
    String pointWkb = "0101000000000000000000F03F000000000000F03F";
    geometryTest(pointString, pointWkb);
  }

  @Test
  public void lineStringTest() throws SQLException {
    String lineString = "LINESTRING(0 0,1 1,2 2)";
    geometryTest(lineString, null);
  }

  @Test
  public void polygonTest() throws SQLException {
    String polygonString = "POLYGON((0 0,10 0,0 10,0 0))";
    geometryTest(polygonString, null);
  }

  private String printHexBinary(byte[] data) {
    StringBuilder builder = new StringBuilder(data.length * 2);
    for (byte b : data) {
      builder.append(hexCode[(b >> 4) & 0xF]);
      builder.append(hexCode[(b & 0xF)]);
    }
    return builder.toString();
  }
}
