#ifndef CUBES_H
#define CUBES_H

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define MAX_TRIANGLES 10

/**
 * Class to read and construct a Marching Cubes contouring table
 *
 * @author Tao Ju
 */
class Cubes
{
	/// Number of triangles in each configuration
	int numtri[ 256 ] ;

	/// Table of triangles in each configuration
	int tris[ 256 ][ MAX_TRIANGLES ][ 3 ] ;

	/// Statistics
	int tricount[ MAX_TRIANGLES ] ;

public:
	/// Constructor: from a file
	Cubes( char* fname )
	{
		FILE* fp ;
		if ( ! ( fp = fopen( fname, "r" ) ) )
		{
			printf("Can not read table file: %s.\n", fname) ;
			exit(0) ;
		}
		createTable( fp ) ;
		fclose( fp ) ;
	}

	/// Get number of triangles
	int getNumTriangle ( int mask )
	{
		return numtri[ mask ] ;
	}

	/// Get a triangle
	void getTriangle( int mask, int index, int inds[3] )
	{
		for ( int i = 0 ; i < 3 ; i ++ )
		{
			inds[ i ] = tris[ mask ][ index ][ i ] ;
		}
	}

	void printTriangles( int mask )
	{
		for ( int index = 0 ; index < numtri[ mask ] ; index ++ )
		{
			for ( int i = 0 ; i < 3 ; i ++ )
			{
				printf("%d ", tris[ mask ][ index ][ i ]);
			}
			printf("\n");
		}
		printf("\n");
	}

	/// Create a table
	void createTable( FILE* fp )
	{
		// Setup transition masks
		int vmap[ 8 ] = {1,0,2,3,5,4,6,7} ;
		int emap[ 12 ] = {8,4,9,5,1,0,2,3,10,6,11,7} ;
		int coface[6][4] = {{0,1,8,10},{2,3,9,11},{6,7,10,11},{4,5,8,9},{1,3,5,7},{0,2,4,6}};
		int iscoface[12][12] ;
		for ( int i = 0 ; i < 12 ; i ++ )
			for ( int j = 0 ; j < 12 ; j ++ )
			{
				iscoface[i][j] = 0 ;
			}
		for ( i = 0 ; i < 6 ; i ++ )
			for ( int j = 0 ; j < 4 ; j ++ )
				for ( int k = 0 ; k < 4 ; k ++ )
				{
					iscoface[coface[i][j]][coface[i][k]] = 1 ;
				}

		// Read each configuration
		for ( i = 0 ; i < 256 ; i ++ )
		{
			char line[ 1024 ] ;
			fgets( line, 1024, fp ) ;
			char seps[] = " ,\t\n";
			char* token ;

			// First, extract sign mask
			int mask = 0 ;
			token = strtok( line, seps ) ;
			for ( int j = 0 ; j < 8 ; j ++ )
			{
				int sign ;
				sscanf( token, "%d", &sign ) ;
				if ( sign < 0 )
				{
					// Outside
					mask |= ( 1 << vmap[ j ] ) ;
				}

				token = strtok( NULL, seps ) ;
			}

			// Next, extract triangles
			numtri[ mask ] = 0 ;
			int buffer[ 10 ], buffersize = 0 ;
			while ( token != NULL )
			{
				if ( ! strcmp( token, "(" ) )
				{
					// Start new group
					buffersize = 0 ;
				}
				else if ( ! strcmp( token, ")" ) )
				{
					// End current group
					
					/* Old way: may introduce edges on cell faces
					for ( int k = 1 ; k < buffersize - 1 ; k ++ )
					{
						tris[ mask ][ numtri[mask] ][ 0 ] = buffer[ 0 ] ;
						tris[ mask ][ numtri[mask] ][ 1 ] = buffer[ k ] ;
						tris[ mask ][ numtri[mask] ][ 2 ] = buffer[ k + 1 ] ;
						numtri[ mask ] ++ ;

					}
					*/

					// New way: no new edges on faces
					for ( j = 0 ; j < buffersize ; j ++ )
					{
						// First, check if buffer[j] is a good vertex to fan out
						for ( int k = 2 ; k < buffersize - 1 ; k ++ )
						{
							int nj = ( j + k ) % buffersize ;
							if ( iscoface[buffer[j]][buffer[nj]] )
							{
								// printf("Break: mask %d, vertex %d\n", mask, buffer[j]) ;
								break ;
							}
						}
						if ( k == buffersize - 1 )
						{
							// good vertex, go ahead and fan out
							for ( int k = 1 ; k < buffersize - 1 ; k ++ )
							{
								tris[ mask ][ numtri[mask] ][ 0 ] = buffer[ j ] ;
								tris[ mask ][ numtri[mask] ][ 1 ] = buffer[ ( j + k ) % buffersize ] ;
								tris[ mask ][ numtri[mask] ][ 2 ] = buffer[ ( j + k + 1 ) % buffersize ] ;
								numtri[ mask ] ++ ;
							}
							break ;
						}
					}
				}
				else
				{
					int eind ;
					sscanf( token, "%d", &( eind ) ) ;
					buffer[ buffersize ] = emap[ eind - 1 ] ;
					buffersize ++ ;
				}

				token = strtok( NULL, seps ) ;
			}
		}
	}

	/// Statistics
	void countTriangles( )
	{
		for ( int i = 0 ; i < MAX_TRIANGLES ; i ++ )
		{
			this->tricount[ i ] = 0 ;
		}

		for ( i = 0 ; i < 256 ; i ++ )
		{
			this->tricount[ numtri[ i ] ] ++ ;
		}

		for ( i = 0 ; i < MAX_TRIANGLES ; i ++ )
		{
			printf("%d triangles: %d configurations.\n", i, tricount[ i ] ) ;
		}

	}
};







#endif