
######################################################################
## $Id: HierSelector.pm 10070 2007-10-10 18:55:12Z spadkins $
######################################################################

package App::Widget::HierSelector;
$VERSION = (q$Revision: 10070 $ =~ /(\d[\d\.]*)/)[0];  # VERSION numbers generated by svn

use App;
use App::Widget;
@ISA = ( "App::Widget" );

use strict;

=head1 NAME

App::Widget::HierSelector - A generic hierarchical view

=head1 SYNOPSIS

   use App::Widget::HierSelector;

   $name = "tree";
   $w = App::Widget::HierSelector->new($name);
   print $w->html();

=cut

######################################################################
# CONSTANTS
######################################################################

######################################################################
# ATTRIBUTES
######################################################################
# {node}{number}{type}       # whether open or closed
# {node}{number}{open}       # 1=open 0=closed
# {node}{number}{value}      #
# {node}{number}{label}      #
# {node}{number}{icon}       # icon to use (default, closed)
# {node}{number}{openicon}   # icon to use when open (optional)
# {node}{number}{hovericon}  # icon to use when cursor over icon

# INPUTS FROM THE ENVIRONMENT

=head1 DESCRIPTION

This class implements a generic hierarchical view such as is useful
for a TreeSelector, a Menu, a ToolbarSet, or an IconPaneSelector.
The main function of a HierSelector is to display a hierarchical set of
data and allow the user to generate events based on that view.

=cut

######################################################################
# INITIALIZATION
######################################################################

# uncomment this when I need to do more than just call SUPER::_init()
sub _init {
    &App::sub_entry if ($App::trace);
    my $self = shift;
    $self->SUPER::_init(@_);
    &App::sub_exit() if ($App::trace);
}

######################################################################
# EVENTS
######################################################################

# Usage: $widget->handle_event($wname, $event, @args);
sub handle_event {
    &App::sub_entry if ($App::trace);
    my ($self, $wname, $event, @args) = @_;
    my ($nodenumber, $x, $y);

    my $node = $self->node_list(1);

    if ($event eq "open") {
        ($nodenumber, $x, $y) = @args;
        $node->{$nodenumber}{open} = 1;
    }
    elsif ($event eq "open_exclusively") {
        ($nodenumber, $x, $y) = @args;
        $self->open_exclusively($nodenumber);
    }
    elsif ($event eq "close") {
        ($nodenumber, $x, $y) = @args;
        $node->{$nodenumber}{open} = 0;
    }
    elsif ($event eq "select") {
        ($nodenumber, $x, $y) = @args;
        $self->set("selected", $nodenumber);  # save node number
        # intentionally bubble "select" event to the container
        if ($wname =~ /^(.*)-([^.]+)$/) {
            my $parent = $1;
            my $result = $self->{context}->widget($parent)->handle_event($wname, $event, @args);
            return $result;
        }
    }
    else {
        return $self->SUPER::handle_event($wname, $event, @args);
    }
    &App::sub_exit() if ($App::trace);
    return 1;
}

sub select_first {
    &App::sub_entry if ($App::trace);
    my $self = shift;
    my $node = $self->node_list();
    
    my ($nodebase, $nodeidx, $nodenumber, $nodenumberfound, $nodelevel);
    my (@nextnodebase, @nextnodeidx, @nextnodelevel);

    @nextnodebase  = ("");   # the next nodenumber to check is "$nodebase$nodeidx" (nodenumber = "1" is first)
    @nextnodeidx   = (1);    # check nodenumber "1" next
    @nextnodelevel = (1);    # index into the resulting table that the folder icon will go

    my ($auth_key, $auth_name, $authorized);
    my $context = $self->{context};
    my $auth = $context->authorization();

    $nodenumberfound = "";
    while ($#nextnodebase > -1) {
        $nodebase  = pop(@nextnodebase);   # get info about next node to check
        $nodeidx   = pop(@nextnodeidx);
        $nodelevel = pop(@nextnodelevel);
        $nodenumber = "$nodebase$nodeidx"; # create its node number

        if (defined $node->{$nodenumber}) {      # if the node exists...

            $auth_name = $node->{$nodenumber}{auth_name};
            if (!$auth_name) {
                $authorized = 1;
            }
            else {
                if ($auth_name =~ m!^/!) {
                    $auth_key = $auth_name;
                }
                else {
                    $auth_key = "/App/SessionObject/$self->{name}/$auth_name";
                }
                $authorized = $auth->is_authorized($auth_key);
            }

            if ($nodelevel > 1 && $authorized) {  # we have found the first node below the uppermost level
                $nodenumberfound = $nodenumber;
                last;
            }

            push(@nextnodebase,    $nodebase);   #   let's search for the node's brother (same depth, next idx)
            push(@nextnodeidx,     $nodeidx+1);  #   (next idx)
            push(@nextnodelevel,   $nodelevel);  #   (same level)

            push(@nextnodebase,  "${nodenumber}."); #   let's search for the node's children (1 deeper, idx 1)
            push(@nextnodeidx,   1);                #   (idx is 1)
            push(@nextnodelevel, $nodelevel+1);     #   (1 deeper)
        }
    }
    if ($nodenumberfound) {
        $self->set("selected", $nodenumberfound);
        my $basenodenumber = $nodenumberfound;
        $basenodenumber =~ s/\..*//;
        $self->open_exclusively($basenodenumber);
    }
    else {
        $self->open_exclusively("1");
    }
    &App::sub_exit() if ($App::trace);
}

sub select {
    &App::sub_entry if ($App::trace);
    my ($self, $nodeattrib, $value) = @_;
    my $node = $self->node_list();
    my $success = 0;
    foreach my $nodenumber (keys %$node) {
        if ($node->{$nodenumber}{$nodeattrib} eq $value) {
            my $prev_nodenumber = $self->{selected} || "";
            my $prev_wname = $node->{$prev_nodenumber}{wname};
            $self->set("selected", $nodenumber);
            if ($node->{$nodenumber}{wname}) {
                my $context = $self->{context};
                my $widget_name = $node->{$nodenumber}{wname};
                my $w = $context->widget($widget_name);
                if ($w->can("navigate")) {
                    $w->navigate($prev_wname, $prev_nodenumber);
                }
            }
            $success = 1;
            last;
        }
    }
    &App::sub_exit($success) if ($App::trace);
    return($success);
}

sub open_selected_exclusively {
    &App::sub_entry if ($App::trace);
    my ($self) = @_;
    $self->open_exclusively($self->{selected});
    &App::sub_exit() if ($App::trace);
}

# i.e. $self->open_exclusively("2.2");
# this should "open" 2 and close 1,3,4,5,...
# this should "open" 2.2 and close 2.1,2.3,2.4,...
# if "2.2.1" exists, it should set the first open to the "selected"
# else it should set itself "2.2" as the "selected"
sub open_exclusively {
    &App::sub_entry if ($App::trace);
    my ($self, $opennodenumber) = @_;
    my ($nodebase, $nodeidx, $nodenumber);
    my $node = $self->node_list(1);
    # set after get to ensure a deep data structure is stored in the session
    $self->set("node", $node);

    $nodebase = $opennodenumber;   # i.e. "2.2.3", "2.2" or "2"
    if ($nodebase =~ /(.*)\.[^\.]+$/) {  # all but the last number
        $nodebase = $1 . ".";      # i.e. "2.2.",  "2."
    }
    else {
        $nodebase = "";            # if top level, $nodebase is blank ""
    }
    $nodeidx = 1;

    while (1) {
        $nodenumber = "$nodebase$nodeidx";
        last if (!defined $node->{$nodenumber});
        $node->{$nodenumber}{open} = 0;  # close all others at this level
        $nodeidx++;
    }

    if (defined $node->{$opennodenumber}) {
        $node->{$opennodenumber}{open} = 1;  # open this one
    }

    # Hmmm. I don't think I should be selecting anything here... just opening/closing.
    if (!defined $node->{"$opennodenumber.1"}) {
        $self->set("selected", $opennodenumber);
    }
    &App::sub_exit() if ($App::trace);
}

sub node_list {
    &App::sub_entry if ($App::trace);
    my ($self, $writeable) = @_;
    my ($node);
    my $auth_node = $self->{auth_node};
    if ($auth_node && ref($auth_node) eq "ARRAY") {
        my ($auth_key, $auth_name);
        my $context = $self->{context};
        my $auth = $context->authorization();
        for (my $i = 0; $i <= $#$auth_node; $i += 2) {
            $auth_name = $auth_node->[$i];
            if ($auth_name =~ m!^/!) {
                $auth_key = $auth_name;
            }
            else {
                $auth_key = "/App/SessionObject/$self->{name}/$auth_name";
            }
            if ($auth->is_authorized($auth_key)) {
                $node = $auth_node->[$i+1];
                $self->set("auth_node", $auth_node) if ($writeable);
                last;
            }
        }
    }
    if (!$node) {
        $node = $self->{node};
        $self->set("node", $node) if ($writeable);
    }
    &App::sub_exit($node) if ($App::trace);
    return($node);
}

# i.e. $self->select_first_open_leaf("2.2");
# this should scan 2.2.1 through 2.2.n for the first open
# this should "open" 2.2 and close 2.1,2.3,2.4,...
# if "2.2.1" exists, it should set the first open to the "selected"
# else it should set itself "2.2" as the "selected"
sub select_first_open_leaf {
    &App::sub_entry if ($App::trace);
    my ($self, $selected_nodenumber) = @_;

    my $node = $self->node_list(1);
    my $nodebase = $selected_nodenumber;
    my $nodeidx = 1;
    my ($nodenumber, $first_auth_nodeidx);
    my $found = 0;

    my ($auth_key, $auth_name, $authorized);
    my $context = $self->{context};
    my $auth = $context->authorization();

    while (!$found) {
        $nodenumber = "$nodebase.$nodeidx";

        if (!defined $node->{$nodenumber}) {
            if ($nodeidx == 1) {  # there are no leaves. $nodebase must be a leaf.

                if ( $nodebase =~ /([0-9\.]+)\.[0-9]+$/ ) {
                    my $node_num = $1;
                    # NOTE: SPA 2007-10-04 - I don't think this is completely correct
                    # It will work for two-level selectors only, but not deeper
                    foreach my $key (keys %$node) {  $node->{$key}{open} = 0;  } 
                    $node->{$node_num}{open} = 1;
                    $node->{$nodebase}{open} = 1;
                }

                $self->set("selected", $nodebase);
                $found = 1;
            }
            else {  # no "open" leaves on this branch
                $node->{"$nodebase.$first_auth_nodeidx"}{open} = 1;
                $self->set("selected", "$nodebase.$first_auth_nodeidx");
                $found = 1;
            }
        }
        else {
            $auth_name = $node->{$nodenumber}{auth_name};
            if (!$auth_name) {
                $authorized = 1;
            }
            else {
                if ($auth_name =~ m!^/!) {
                    $auth_key = $auth_name;
                }
                else {
                    $auth_key = "/App/SessionObject/$self->{name}/$auth_name";
                }
                $authorized = $auth->is_authorized($auth_key);
            }

            if ($node->{$nodenumber}{open} && $authorized) {
                $nodebase = $nodenumber;
                $nodeidx  = 1;
                $first_auth_nodeidx = undef;
            }
            else {
                $first_auth_nodeidx = $nodeidx if (!defined $first_auth_nodeidx && $authorized);
                $nodeidx++;
            }
        }
    }
    &App::sub_exit() if ($App::trace);
    #$self->{debug} .= "select_first_open_leaf($selected_nodenumber): [$nodenumber]<br>";
}

######################################################################
# METHODS
######################################################################

sub get_selected {
    &App::sub_entry if ($App::trace);
    my ($self, $nodeattrib) = @_;
    my ($nodenumber);

    $nodenumber = $self->{selected};  # might be a tag or a nodenumber

    return undef if (!defined $nodenumber);
    return $nodenumber if (!defined $nodeattrib);

    my $node = $self->node_list();
    $nodenumber = $self->get_tag_nodenumber($nodenumber);

    &App::sub_exit($node->{$nodenumber}{$nodeattrib}) if ($App::trace);
    return $node->{$nodenumber}{$nodeattrib};
}

sub get_tag_nodenumber {
    &App::sub_entry if ($App::trace);
    my ($self, $tag) = @_;

    my $node = $self->node_list();

    my ($nodenumber);
    if ($tag !~ /^\d+/) {  # if $tag is not a $nodenumber already ...
        foreach my $nodenum (keys %$node) {
            if ( $node->{$nodenum}{tag} && $node->{$nodenum}{tag} eq $tag) {
                $nodenumber = $nodenum;
                last;
            }
        } 
    }
    else {
        $nodenumber = $tag;
    }
   
    &App::sub_exit($nodenumber) if ($App::trace);
    return $nodenumber;
}

######################################################################
# OUTPUT METHODS
######################################################################

sub html {
    &App::sub_entry if ($App::trace);
    my $self = shift;
    my ($html, $label);

    my $node = $self->node_list();

    my ($nodebase, $nodeidx, $nodenumber, $nodelevel);
    my (@nextnodebase, @nextnodeidx, @nextnodelevel);

    @nextnodebase  = ("");   # the next nodenumber to check is "$nodebase$nodeidx" (nodenumber = "1" is first)
    @nextnodeidx   = (1);    # check nodenumber "1" next
    @nextnodelevel = (1);    # index into the resulting table that the folder icon will go

    $html = "";
    while ($#nextnodebase > -1) {
        $nodebase  = pop(@nextnodebase);   # get info about next node to check
        $nodeidx   = pop(@nextnodeidx);
        $nodelevel = pop(@nextnodelevel);
        $nodenumber = "$nodebase$nodeidx"; # create its node number
        if (defined $node->{$nodenumber}) {      # if the node exists...
            $label = $node->{$nodenumber}{label};
            $label = $node->{$nodenumber}{value} if (!defined $label);
            $label = "" if (!defined $label);
            $html .= ("&nbsp;&nbsp;" x ($nodelevel-1)) if ($nodelevel > 1);
            $html .= $label;
            $html .= $node->{$nodenumber}{open} ? " (open)" : " (closed)";
            $html .= "<br>\n";

            push(@nextnodebase,    $nodebase);   #   let's search for the node's brother (same depth, next idx)
            push(@nextnodeidx,     $nodeidx+1);  #   (next idx)
            push(@nextnodelevel,   $nodelevel);  #   (same level)

            push(@nextnodebase,  "${nodenumber}."); #   let's search for the node's children (1 deeper, idx 1)
            push(@nextnodeidx,   1);                #   (idx is 1)
            push(@nextnodelevel, $nodelevel+1);     #   (1 deeper)
        }
    }

    &App::sub_exit() if ($App::trace);
    $html;    
}

1;

