/**************************************************************************
**
** This file is part of Qt Creator
**
** Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
**
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** Commercial Usage
**
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
**
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at http://qt.nokia.com/contact.
**
**************************************************************************/

#ifndef BASEFILEWIZARD_H
#define BASEFILEWIZARD_H

#include "core_global.h"

#include <coreplugin/dialogs/iwizard.h>

#include <QtCore/QSharedDataPointer>
#include <QtCore/QList>

QT_BEGIN_NAMESPACE
class QIcon;
class QWizard;
class QWizardPage;
class QDebug;
QT_END_NAMESPACE

namespace Utils {
    class Wizard;
}

namespace Core {

class IEditor;
class IFileWizardExtension;

class BaseFileWizardParameterData;
struct BaseFileWizardPrivate;
class GeneratedFilePrivate;

/*!
 * Represents a file generated by a wizard. The Wizard class will check for
 * each file whether it already exists and will report any errors that may
 * occur during creation of the files.
 */
class CORE_EXPORT GeneratedFile
{
public:
    enum Attribute { OpenEditorAttribute = 0x01, OpenProjectAttribute = 0x02 };
    Q_DECLARE_FLAGS(Attributes, Attribute)

    GeneratedFile();
    explicit GeneratedFile(const QString &path);
    GeneratedFile(const GeneratedFile &);
    GeneratedFile &operator=(const GeneratedFile &);
    ~GeneratedFile();

    // Full path of the file should be created, or the suggested file name
    QString path() const;
    void setPath(const QString &p);

    // Contents of the file (UTF8)
    QString contents() const;
    void setContents(const QString &c);

    QByteArray binaryContents() const;
    void setBinaryContents(const QByteArray &c);

    // Defaults to false (Text file).
    bool isBinary() const;
    void setBinary(bool b);

    // Id of editor to open the file with
    QString editorId() const;
    void setEditorId(const QString &k);

    bool write(QString *errorMessage) const;

    Attributes attributes() const;
    void setAttributes(Attributes a);

private:
    QSharedDataPointer<GeneratedFilePrivate> m_d;
};

typedef QList<GeneratedFile> GeneratedFiles;

/* Parameter class for passing parameters to instances of class Wizard
 * containing name, icon and such. */

class CORE_EXPORT BaseFileWizardParameters
{
public:
    explicit BaseFileWizardParameters(IWizard::WizardKind kind = IWizard::FileWizard);
    BaseFileWizardParameters(const BaseFileWizardParameters &);
    BaseFileWizardParameters &operator=(const BaseFileWizardParameters&);
   ~BaseFileWizardParameters();

    void clear();

    IWizard::WizardKind kind() const;
    void setKind(IWizard::WizardKind k);

    QIcon icon() const;
    void setIcon(const QIcon &icon);

    QString description() const;
    void setDescription(const QString &description);

    QString displayName() const;
    void setDisplayName(const QString &name);

    QString id() const;
    void setId(const QString &id);

    QString category() const;
    void setCategory(const QString &category);

    QString displayCategory() const;
    void setDisplayCategory(const QString &trCategory);

private:
    QSharedDataPointer<BaseFileWizardParameterData> m_d;
};

CORE_EXPORT QDebug operator<<(QDebug d, const BaseFileWizardParameters &);

/* A generic wizard for creating files.
 *
 * The abstract methods:
 *
 * createWizardDialog() : Called to create the QWizard dialog to be shown
 * generateFiles()      : Generate file content
 *
 * must be implemented. The behaviour can be further customized by overwriting
 * the virtual method:
 * postGenerateFiles()  :   Called after generating the files.
 */

class CORE_EXPORT BaseFileWizard : public IWizard
{
    Q_DISABLE_COPY(BaseFileWizard)
    Q_OBJECT

public:
    virtual ~BaseFileWizard();

    // IWizard
    virtual WizardKind kind() const;
    virtual QIcon icon() const;
    virtual QString description() const;
    virtual QString displayName() const;
    virtual QString id() const;

    virtual QString category() const;
    virtual QString displayCategory() const;

    virtual QStringList runWizard(const QString &path, QWidget *parent);

    // Build a file name, adding the extension unless baseName already has one
    static QString buildFileName(const QString &path, const QString &baseName, const QString &extension);

    // Sets some standard options on a QWizard
    static void setupWizard(QWizard *);

    // Read "shortTitle" dynamic property of the pageId and apply it as the title of corresponding progress item
    static void applyExtensionPageShortTitle(Utils::Wizard *wizard, int pageId);

protected:
    typedef QList<QWizardPage *> WizardPageList;

    explicit BaseFileWizard(const BaseFileWizardParameters &parameters, QObject *parent = 0);

    // Overwrite to create the wizard dialog on the parent, adding
    // the extension pages.
    virtual QWizard *createWizardDialog(QWidget *parent,
                                        const QString &defaultPath,
                                        const WizardPageList &extensionPages) const = 0;

    // Overwrite to query the parameters from the dialog and generate the files.
    virtual GeneratedFiles generateFiles(const QWizard *w,
                                         QString *errorMessage) const = 0;

    /* Overwrite to perform steps to be done after files are actually created.
     * The default implementation opens editors with the newly generated files. */
    virtual bool postGenerateFiles(const QWizard *w, const GeneratedFiles &l, QString *errorMessage);

    // Utility that returns the preferred suffix for a mime type
    static QString preferredSuffix(const QString &mimeType);

    // Utility that performs an overwrite check on a set of files. It checks if
    // the file exists, can be overwritten at all and prompts the user.
    enum OverwriteResult { OverwriteOk,  OverwriteError,  OverwriteCanceled };
    OverwriteResult promptOverwrite(const QString &location,
                                    const QStringList &files,
                                    QString *errorMessage) const;

    // Utility to open the editors for the files whose attribute is set accordingly.
    static bool postGenerateOpenEditors(const GeneratedFiles &l, QString *errorMessage = 0);

private:
    BaseFileWizardPrivate *m_d;
};

// StandardFileWizard convenience class for creating one file. It uses
// Utils::FileWizardDialog and introduces a new virtual to generate the
// files from path and name.

class CORE_EXPORT StandardFileWizard : public BaseFileWizard
{
    Q_DISABLE_COPY(StandardFileWizard)
    Q_OBJECT

protected:
    explicit StandardFileWizard(const BaseFileWizardParameters &parameters, QObject *parent = 0);

    // Implemented to create a Utils::FileWizardDialog
    virtual QWizard *createWizardDialog(QWidget *parent,
                                        const QString &defaultPath,
                                        const WizardPageList &extensionPages) const;
    // Implemented to retrieve path and name and call generateFilesFromPath()
    virtual GeneratedFiles generateFiles(const QWizard *w,
                                         QString *errorMessage) const;

    // Newly introduced virtual that creates a file from a path
    virtual GeneratedFiles generateFilesFromPath(const QString &path,
                                                 const QString &name,
                                                 QString *errorMessage) const = 0;
};

} // namespace Core

Q_DECLARE_OPERATORS_FOR_FLAGS(Core::GeneratedFile::Attributes)

#endif // BASEFILEWIZARD_H
