#!/bin/bash
# Copyright (C) 2013 Tomasz Olszak <olszak.tomasz@gmail.com>
# All rights reserved.
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:

# 1. Redistributions of source code must retain the copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. The name of the author may not be used to endorse or promote products 
#    derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#

ROOTSTRAP_NAME=tizen-emulator-2.1.native
COMPILER_DIRNAME=i386-linux-gnueabi-gcc-4.5
COMPILER_PREFIX=i386-linux-gnueabi
DEVICE=linux-g++-tizen-emulator

set -e

CURRENT_DIR=$(pwd)

SCRIPT_DIR=$(dirname $0)

cd $SCRIPT_DIR

function usage {
    echo -e "\n"
    echo "Script for Qt for tizen compilation. "
    echo "Set Parameters or environment variables with absolute paths"
    echo ""
    echo "Usage:"
    echo "sudo [MAKE_THREADS=numberOfCPUCores] ./downloadAndBuildAll.sh \$STANDARD_USER [\$TIZENSDKSYSROOT \$TIZENSDKCOMPILERDIR \$TIZENSYSROOT \$QT5SRCDIR \$HOSTPREFIX]"
    echo ""
    echo "STANDARD_USER - your standard user, without superuser privileges"
    echo "TIZENSDKSYSROOT - should be set to tizen-sdk/platforms/tizen2.1/rootstraps/$ROOTSTRAP_NAME"
    echo "TIZENSDKCOMPILERDIR - should be set to tizen-sdk/tools/$COMPILER_DIRNAME "
    echo "QT5SRCDIR - directory with qt5 sources"
    echo "TIZENSYSROOT - sysroot created from emulator devel headers and libraries - by default qt will be installed in \$TIZENSYSROOT/opt/usr/qt5"
    echo "HOSTPREFIX - here qmake and other host binaries will be installed"
    echo "If you want to download qt source, tizen packages and autoomatically build just pass STANDARD_USER as first parameter(\$USER)"
    echo "If corresponding environment variables are set then script can be invoked without parameters"
    echo "set MAKE_THREADS environment variable to number of your CPU cores to fast up compilation step"    
    echo ""
    cd $CURRENT_DIR
    exit;
}

if [ "$1" == "-h" -o "$1" == "--help" ]; then
    usage
fi

CURRENT_USER=$(whoami)

if [ $CURRENT_USER != "root" ]; then
    echo "run this script with superuser privileges"
    usage
fi


if [ "$STANDARD_USER" == "" ]; then
    STANDARD_USER=$1
fi


if [ "$STANDARD_USER" == "" ]; then
    echo "standard user name was not passed to script"
    usage
fi

set +e

type rpm2cpio &> /dev/null

if [ "$?" != "0" ]; then
    echo "rpm2cpio is needed for creating rootfs - install rpm2cpio"
    exit 1
fi

type cpio &> /dev/null

if [ "$?" != "0" ]; then
    echo "cpio is needed for creating rootfs - install cpio"
    exit 1
fi

id $STANDARD_USER &> /dev/null

if [ "$?" != "0" ]; then
    echo "there is no user $STANDARD_USER in the system"
    exit 1
fi

set -e

if [ "$TIZENSDKSYSROOT" == "" ]; then
    if [ -f $HOME/tizen-sdk/platforms/tizen2.1/rootstraps/$ROOTSTRAP_NAME/usr/include/stdio.h ]; then        
        TIZENSDKSYSROOT=$HOME/tizen-sdk/platforms/tizen2.1/rootstraps/$ROOTSTRAP_NAME
        echo "Setting TIZENSDKSYSROOT to $TIZENSDKSYSROOT"
    else
        TIZENSDKSYSROOT=$2
    fi
fi

if [ "$TIZENSDKCOMPILERDIR" == "" ]; then
    if [ -f $HOME/tizen-sdk/tools/$COMPILER_DIRNAME/bin/$COMPILER_PREFIX-g++ ]; then
        TIZENSDKCOMPILERDIR=$HOME/tizen-sdk/tools/$COMPILER_DIRNAME
        echo "Setting TIZENSDKCOMPILERDIR to $TIZENSDKCOMPILERDIR"
    else
        TIZENSDKCOMPILERDIR=$3
    fi
fi

if [ "$TIZENSYSROOT" == "" ]; then
    if [ "$4" != "" ]; then
        TIZENSYSROOT=$4
    else 
        TIZENSYSROOT=$PWD/rootfs
        echo "Setting TIZENSYSROOT to $TIZENSYSROOT"
    fi
fi

if [ "$QT5SRCDIR" == "" ]; then
    if [ "$5" != "" ]; then
        QT5SRCDIR=$5
    else
        QT5SRCDIR=$PWD/qt5.gitorious
        echo "Setting QT5SRCDIR to $QT5SRCDIR"
    fi
fi

if [ "$HOSTPREFIX" == "" ]; then
    if [ "$6" != "" ]; then
        HOSTPREFIX=$6
    else 
        HOSTPREFIX=$PWD/qt5hostInstall
        echo "Setting HOSTPREFIX to $HOSTPREFIX"
    fi
fi


if [ -z "$STANDARD_USER" -o -z "$QT5SRCDIR" -o -z "$TIZENSYSROOT" -o -z "$TIZENSDKSYSROOT" \
     -o -z "$TIZENSDKCOMPILERDIR" -o -z "$HOSTPREFIX" ]; then
      usage
fi

if [ "$TIZENSYSROOT" == "$PWD/rootfs" ]; then

    if [ ! -d repos ]; then
        echo "getting tizen repositories"
        su $STANDARD_USER -c "./download_repos.sh"
    fi

    mkdir -p rootfs

    if [ ! -d rootfs/usr/lib ]; then
        cd rootfs
        for rpmFile in `find ../repos -name "*.rpm"`; 
        do 
            rpm2cpio $rpmFile | cpio -imdv -R $STANDARD_USER:$STANDARD_USER; 
        done
        cd ..
    else
        echo "Looks like rootfs is already prepared - rootfs/usr/lib exists"
    fi
fi

if [ "$QT5SRCDIR" == "$PWD/qt5.gitorious" ]; then
    if [ ! -d qt5.gitorious ]; then
        su $STANDARD_USER -c "git clone -b release https://git.gitorious.org/qt/qt5.git qt5.gitorious"
        cd qt5.gitorious
        su $STANDARD_USER -c "git clone -b wip/tizen https://git.gitorious.org/qt/qtquickcontrols-tizen.git"
        su $STANDARD_USER -c "./init-repository --http --no-webkit"
        cd qtquickcontrols
        su $STANDARD_USER -c "git checkout wip/tizen"
        cd ../
        cd qtbase
        su $STANDARD_USER -c "git checkout wip/tizen"
        cd ../../
    else
        echo "Looks like qt5.gitorious directory exists"
    fi;
fi;

su $STANDARD_USER -c "mkdir -p buildQt5"
cd buildQt5

echo "================================================="
echo "Configuration:"
echo "STANDARD_USER=$STANDARD_USER"
echo "QT5SRCDIR=$QT5SRCDIR"
echo "TIZENSYSROOT=$TIZENSYSROOT"
echo "TIZENSDKSYSROOT=$TIZENSDKSYSROOT"
echo "TIZENSDKCOMPILERDIR=$TIZENSDKCOMPILERDIR"
echo "HOSTPREFIX=$HOSTPREFIX"
echo "================================================="

export PKG_CONFIG_LIBDIR=$TIZENSYSROOT/usr/lib
export PKG_CONFIG_SYSROOT_DIR=$TIZENSYSROOT
export PKG_CONFIG_PATH=$TIZENSYSROOT/usr/lib/pkgconfig


su $STANDARD_USER -c "$QT5SRCDIR/configure -hostprefix $HOSTPREFIX -v \
                    -release -opengl es2 -opensource \
                    -confirm-license -device $DEVICE \
                    -device-option CROSS_COMPILE=$TIZENSDKCOMPILERDIR/bin/$COMPILER_PREFIX- \
                    -prefix /opt/usr/qt5 -sysroot $TIZENSYSROOT -nomake exmaples \
                    -qpa xcb -system-libpng -qt-xcb -no-widgets -no-eglfs"

MAKE_ARGS=""

if [ "$MAKE_THREADS" != "" ]; then
    echo "setting make thread"
    MAKE_ARGS="-j $MAKE_THREADS"
fi
echo make $MAKE_ARGS

su $STANDARD_USER -c "make $MAKE_ARGS"
echo make install $MAKE_ARGS
su $STANDARD_USER -c "make install $MAKE_ARGS"

if [ -d $QT5SRCDIR/qtquickcontrols-tizen ]; then
    su $STANDARD_USER -c "mkdir qtquickcontrols-tizen"
    cd qtquickcontrols-tizen
    su $STANDARD_USER -c "$HOSTPREFIX/bin/qmake $QT5SRCDIR/qtquickcontrols-tizen"
    su $STANDARD_USER -c "make install $MAKE_ARGS"
fi
        
cd $CURRENT_DIR    
