//
//
// This file contains sample C source code which can be used to send 
// HTML help commands to Netscape Navigator 3.0 and newer (earlier
// versions cannot display HTML help).  It is intended to be incorporated
// into existing projects which will handle reporting of errors to the
// end-user or other appropriate response.


#include "HTMLhelp.h"
#include <Processes.h>
#include <string.h>


// From Navigator's <resae.h>
#define AE_www_suite 			'MOSS'
#define AE_www_ReadHelpFile  	'help'		// keyDirectObject is the file
#define AE_www_ReadHelpFileID 	'idid'		// Help file id. If none, use "DEFAULT"
#define AE_www_ReadHelpFileSearchText  'sear' // Search text, no default


// Local Prototypes
static OSErr LaunchWithAppleEvent( OSType inAppSig, AppleEvent *inEvent,
							AppleEvent *result, AESendMode inSendMode);
static OSErr FindApplicationBySignature(OSType sig, FSSpec *appLoc);
static Boolean IsFrontApplication();
static OSErr FindProcessBySignature(OSType sig, OSType processType,
							ProcessSerialNumber *psn, FSSpec *fileSpec);


// Constants
const OSType cHelpAppSig = 'MOSS';			// Navigator's creator


// 
//	SendHelpCommand
// 
// Send Navigator the AppleEvent for displaying your HTML help file
//	inHelpFile is a valid FSSpec
//	topic is a c-string (null-terminated)

OSErr SendHelpCommand( FSSpec *inHelpFile, char *topic )
{
	OSErr err;
	AEDesc helpAppAddress, fileAddress, topicDesc;
	AppleEvent theEvent, result;
	
	if (inHelpFile == NULL)
		return nsvErr;			// what is the correct error here?
		
	// Address the event (send to Navigator)
	err = AECreateDesc(typeApplSignature, &cHelpAppSig, sizeof(cHelpAppSig), &helpAppAddress);
	if ( err )
		return err;
		
	// Create it
	err = AECreateAppleEvent(AE_www_suite, AE_www_ReadHelpFile, &helpAppAddress, 
								kAutoGenerateReturnID, kAnyTransactionID, &theEvent);
	if ( err )
	{
		AEDisposeDesc( &helpAppAddress );
		return err;
	}
	
	// delete AEDesc for application address since we no longer need it
	err = AEDisposeDesc( &helpAppAddress );
	if ( err )
	{
		AEDisposeDesc( &theEvent );		// dispose of theEvent we created
		return err;
	}
	
	// Add the arguments to the AppleEvent
	// create an AEDesc for the help-file's FSSpec
	err = AECreateDesc( typeFSS, inHelpFile, sizeof( *inHelpFile ), &fileAddress ); 
	if ( err )
	{
		AEDisposeDesc( &theEvent );		// dispose of theEvent we created
		return err;
	}

	// add AEDesc for help-file to theEvent
	err = AEPutKeyDesc( &theEvent, keyDirectObject, &fileAddress );
	if ( err )
	{
		AEDisposeDesc( &fileAddress );
		AEDisposeDesc( &theEvent );
		return err;
	}
		
	// Help ID
	// create AEDesc
	err = AECreateDesc( 'TEXT', topic, strlen(topic), &topicDesc );
	if ( err )
	{
		AEDisposeDesc( &theEvent );		// dispose of theEvent we created
		return err;
	}

	// add help topic AEDesc to theEvent 
	err = AEPutKeyDesc( &theEvent, AE_www_ReadHelpFileID, &topicDesc );
	if ( err )
	{
		AEDisposeDesc( &topicDesc );
		AEDisposeDesc( &theEvent );
		return err;
	}
	
	err = AEDisposeDesc( &topicDesc );
	if ( err )
	{
		AEDisposeDesc( &theEvent );
		return err;
	}
	
	err = LaunchWithAppleEvent( cHelpAppSig, &theEvent, &result, 
					kAENoReply + kAEAlwaysInteract + kAECanSwitchLayer);
	
	if ( err )
	{
		AEDisposeDesc( &theEvent );
		return err;
	}
	
	err = AEDisposeDesc( &theEvent );
	return err;
}


// 
// IsFrontApplication
// 
// Are we the front application?

static Boolean IsFrontApplication()
{
	ProcessSerialNumber myAppProcess, frontProcess;
	Boolean inFront;
	OSErr err;

	err = GetCurrentProcess( &myAppProcess );
	if ( err != noErr )
		return false;
	
	err = GetFrontProcess( &frontProcess );
	if ( err != noErr )
		return false;
	
	err = SameProcess( &frontProcess, &myAppProcess, &inFront );
	if ( err != noErr )
		return false;
	
	return inFront;
}


// 
// FindApplicationOnVolume
// 
// Given a signature, find the FSSpec of the application
// Searches all volumes, system volume first

static OSErr FindApplicationOnVolume(OSType sig, short vRefNum, FSSpec *appLoc)	
{
    DTPBRec pb;
    OSErr err;
    short refNum;

    pb.ioCompletion = NULL;
    pb.ioVRefNum = vRefNum;
    pb.ioNamePtr = NULL;
    if ( (err = PBDTGetPath(&pb)) != noErr )
    	return err;
    	                     	  	// Puts DT refnum into pb.ioDTRefNum
	refNum = pb.ioDTRefNum;

    pb.ioCompletion = NULL;
    pb.ioDTRefNum = refNum;
    pb.ioIndex = 0;
    pb.ioFileCreator = sig;
    pb.ioNamePtr = (StringPtr) appLoc->name;
    err = PBDTGetAPPLSync(&pb);    	// Look for it

    if ( err == fnfErr )
    	err = afpItemNotFound;		// Bug in PBDTGetAPPL
    if ( err )
    	return err;           		// Returns afpItemNotFound if app wasn't found.

    appLoc->vRefNum = vRefNum;
    appLoc->parID = pb.ioAPPLParID;
    return err;
}


// 
// VolHasDesktopDB
// 
// Given a volume reference number, check for Desktop DB

static OSErr VolHasDesktopDB(short vRefNum, Boolean *hasDesktop)
{
	HParamBlockRec pb;
	GetVolParmsInfoBuffer info;
	OSErr err;

	pb.ioParam.ioCompletion = NULL;
	pb.ioParam.ioVRefNum = vRefNum;
	pb.ioParam.ioNamePtr = NULL;
	pb.ioParam.ioBuffer = (Ptr) & info;
	pb.ioParam.ioReqCount = sizeof(GetVolParmsInfoBuffer);

	err = PBHGetVolParmsSync(&pb);

	*hasDesktop = (err == noErr) && ((info.vMAttrib & (1L << bHasDesktopMgr)) != 0);

	return err;
} 


// 
// FindApplicationBySignature
// 
// Given a signature, find the FSSpec of the application
// Searches all volumes, system volume first

static OSErr FindApplicationBySignature(OSType sig, FSSpec *appLoc)	
{
	VCB	*currentVolumeP;
	OSErr err;
	Boolean hasDesktopDB;

	currentVolumeP = (VCB *)(GetVCBQHdr())->qHead;
	while ( currentVolumeP )
	{
	    err = VolHasDesktopDB(currentVolumeP->vcbVRefNum, &hasDesktopDB);
	    if ( err )
			return err;
	    if ( hasDesktopDB )          // If volume has a desktop DB, check for app
	    {
			err = FindApplicationOnVolume( sig, currentVolumeP->vcbVRefNum, appLoc );
			if ( err == noErr )
            	return err;
		}
				
		currentVolumeP = (VCB *)currentVolumeP->qLink;
	}

	return fnfErr;
}

// FindProcessBySignature 
// Given an application signature, finds the process
static OSErr FindProcessBySignature(OSType sig, OSType processType,
							ProcessSerialNumber *psn, FSSpec *fileSpec)
{
	OSErr err;
	ProcessInfoRec info;

	psn->highLongOfPSN = 0;
	psn->lowLongOfPSN = kNoProcess;
	do
	{
		err = GetNextProcess( psn );
		if ( err == noErr )
		{
			info.processInfoLength = sizeof( ProcessInfoRec );
			info.processName = NULL;
			info.processAppSpec = fileSpec;

			err = GetProcessInformation( psn, &info );
		}
	} while ( (err == noErr) && 
			((info.processSignature != sig) || 
			(info.processType != processType)));

	if ( err == noErr )
		*psn = info.processNumber;

	return err;
}


// 
// LaunchWithAppleEvent
// 
// Launches the application with the given apple event
// If application is already running, send the event to the app
// 
// Sample usage:
// 	AEDesc myAddress;
//  err = AECreateDesc(typeApplSignature, &applSig, sizeof(applSig), &myAddress);
// 	err = AECreateAppleEvent(kCoreEventClass, kAEOpenDocuments, &myAddress, 
//								kAutoGenerateReturnID, kAnyTransactionID, &theEvent);
//	LaunchWithAppleEvent( applSig, theEvent, result, kAENoReply + kAEAlwaysInteract + kAECanSwitchLayer)

static OSErr LaunchWithAppleEvent( OSType inAppSig, AppleEvent *inEventP,
							AppleEvent *result, AESendMode inSendMode)
{
	OSErr err = noErr;
	ProcessSerialNumber appPSN;
	FSSpec appSpec;

	if ( FindProcessBySignature (inAppSig, 'APPL', &appPSN, &appSpec ) == noErr )
		// Process is running, just send the event
	{
		err = AESend(inEventP, result, inSendMode, kAENormalPriority, kAEDefaultTimeout, NULL, NULL);
		SetFrontProcess( &appPSN );	// need to explicitly bring to front since we're not launched
	}
	else
	{	// No process, need to launch the app
		LaunchParamBlockRec launchP;
		LaunchFlags memoryFlags = 0;
		AEDesc launchDesc;

		// Find the application
		err = FindApplicationBySignature( inAppSig, &appSpec );
		if ( err )
			return err;
			
		// Coerce the event into application launch parameters
		err = AECoerceDesc( inEventP, typeAppParameters, &launchDesc );
		if ( err )
			return err;
		
		// Fill in the launch block
		launchP.launchAppSpec = (FSSpecPtr)&appSpec;
		launchP.launchAppParameters = (AppParametersPtr)*(launchDesc.dataHandle);
		launchP.launchBlockID = extendedBlock;
		launchP.launchEPBLength = extendedBlockLen;
		launchP.launchFileFlags = (unsigned short)NULL;
		if ( !IsFrontApplication() )
			launchP.launchControlFlags += launchDontSwitch;
tryAgain:
		launchP.launchControlFlags = launchContinue + launchNoFileFlags + memoryFlags;

		// Launch
		err = LaunchApplication( &launchP );
		
		// If we are out of memory, try launching the app with minimum partition
		if ( (( err == memFullErr ) || ( err == memFragErr )) 
		&& ( memoryFlags == 0 ))
		{
			memoryFlags = launchUseMinimum;
			goto tryAgain;
		}
		
		AEDisposeDesc( &launchDesc );
		
		if (err)
			return err;		
	}

	return err;
}


