#!/usr/bin/perl

# ------------------------------------------------------------------
#
#    Copyright (C) 2002-2005 Novell/SUSE
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of version 2 of the GNU General Public 
#    License published by the Free Software Foundation.
#
# ------------------------------------------------------------------

################################################################################
# /usr/bin/reportgen.pl
#
#   - Calls Immunix::Reports.pm to write AppArmor events to a specified logfile 
#	- Expected to be executed by a cron job.
#
#  Requires:
#   /usr/lib/perl5/site_perl/Immunix::Reports.pm
#
#  Input (Optional):
#       -Report Name
#       -Program Name		'progName'
#       -Profile Name		'profile'
#       -PID				'pid'
#		-Severity			'sevLevel'
#       -Denied Resources	'resource'
#       -SD Mode			'sdmode'
#       -Mode				'mode'
#
################################################################################
use strict;
use Immunix::Reports;
#use POSIX;						# causes locale error w/ Exporter
use Locale::gettext;

#setlocale(LC_MESSAGES, "");	# causes locale error w/ Exporter
textdomain("yast2-apparmor");

# Sloppy Global
my $headerFile = "/var/log/apparmor/reports/reportgen.header";

# Routines
################################################################################
sub logMessage {

	my ($msg,$date) = @_;

	if (! $date) {
		$date = localtime;
	}

	my $msgLog = '/var/log/apparmor/cron-reports.log';

	if (open (MLOG, ">>$msgLog") ) {
		print MLOG "$date - reportgen.pl: $msg\n";
		close MLOG;
	}

	return 0;
}

sub makeHeader {

	my ($repType,$repConf,$filts) = @_;	# Filters & Time are only for SIR

	my $header = undef;
	my $date = localtime;
	my $start = localtime($repConf->{'startTime'});

	if ( $repType eq 'ess' ) {

		# Write ESS Header in csv format
	    $header->{'csv'} =  gettext("# Executive Security Summary - Generated by AppArmor\n");
		$header->{'csv'} .= sprintf(gettext("# Period: %s to %s\n\n"), $start,$date);

		# Write ESS Header in html format
		$header->{'html'} = "<table border='1' cellpadding='2'>\n";
		$header->{'html'} .= gettext("<tr><th colspan='7'>Executive Security
			Summary - Generated by AppArmor</th></tr>");
		$header->{'html'} .= sprintf(gettext("<tr><th colspan='7'>Period: %s to 
			%s</th></tr>\n"), $start, $date);

	} elsif ( $repType eq 'aud' ) {

		# Write AUD Header in csv format
	    $header->{'csv'} = gettext("# Application Audit Report - Generated
			by AppArmor\n");
		$header->{'csv'} .= sprintf(gettext("# Date Run: %s\n\n"), $date);

		# Write AUD Header in html format
		$header->{'html'} =  gettext("<table border='1' cellpadding='2'><tr><th
			colspan='7'>Applications Audit Report - Generated by AppArmor</th></tr>\n");

	} else {

	    # Write SIR Header in csv format
	    $header->{'csv'} =  gettext("# Security Incident Report - Generated by AppArmor\n");
	    $header->{'csv'} .= sprintf(gettext("# Period: %s - %s\n"), $start, $date);

	    my $count = 0;

        if ( $filts ) {
			# We don't want startdate/enddate to be listed as filters
            $count = keys(%$filts);
            if ($filts->{'startdate'}) {
                $filts->{'startdate'} = localtime($filts->{'startdate'});
				$count--;
            }
            if ($filts->{'enddate'}) {
                $filts->{'enddate'} = localtime($filts->{'enddate'});
				$count--;
            }
        }

		if ( $count > 0 ) {
	        $header->{'csv'} .= gettext("# The following filters were used for report generation:\n");
	        for (sort keys(%$filts)) {
				unless ( $filts->{'startdate'} || $filts->{'enddate'} ) {
		            $header->{'csv'} .= sprintf(gettext("# Filter: %s, Value: %s\n\n"), $_, $filts->{$_});
				}
	        }
	    } else {
	        $header->{'csv'} .= gettext("# No filters were used for report generation:\n\n\n");
	    }

	    # Write SIR Header in html format
        $header->{'html'} =  gettext("<h3>Security Incident Report - Generated by AppArmor</h3>\n");
        $header->{'html'} .= sprintf(gettext("<h4>Period: %s - %s</h4>\n"), $start, $date);

		if ( $count > 0 ) {

            $header->{'html'} .= gettext("<h4>The following filters were used for report generation:</h4>");

	        for (sort keys(%$filts)) {
                $header->{'html'} .= sprintf(gettext("<p>Filter: %s, Value: %s</p>"), $_, $filts->{$_});
	        }

	    } else {
            $header->{'html'} .= gettext("<h4>No filters were used for report generation.</h4>");
	    }

        $header->{'html'} .= "<hr>\n";

	}

	my $mailHeader = $header->{'csv'};
	$mailHeader =~ s/^#\s*//;

	if ( open (HDR, ">$headerFile") ) {
		print HDR "$mailHeader";
		close HDR;
	} 

	return $header;
}

# Sets name of new report file to generate
sub getRepFileName {

	my ($hashName,$dir,$repConf) = @_;
	my @dirList = ();

	my $plainName = $hashName->{'base'};

	# Append date info to help id reports
	my ($sec,$min,$hour,$mday,$month,$year,@junk) = localtime;

    $year += 1900;
    $month += 1;

    $month = sprintf("%02d", $month);
    $mday  = sprintf("%02d", $mday);
    $hour  = sprintf("%02d", $hour);
    $min   = sprintf("%02d", $min);
    $sec   = sprintf("%02d", $sec);
    $mday  = sprintf("%02d", $mday);

	my $suffix = "-$year-$month-$mday\_$hour.$min.$sec";
	$plainName = $plainName . $suffix;
	my $qtName = "\"$plainName\"";

	if (opendir (DIR, $dir)) {
		@dirList = grep(/$qtName/, readdir(DIR) );
		close DIR;
	} 
	my $numReps = scalar(@dirList) + 1;
	$numReps = sprintf("%03d", $numReps);
	my $newName = undef; 

	# Prefix all names w/ .csv -- we'll change to .html later if required
	$newName = "$dir/$plainName.$numReps.csv";

	return $newName;
}

sub getRepConf {

	my $repName = shift;

	my $repConf = '/etc/apparmor/reports.conf';
	my $tmpConf = '/var/log/apparmor/reports/reports.conf';
	my $config = undef; 
	my @mailList = ();
	my %filts = ();			# Report 'record' filters
	my $rep = Immunix::Reports::getXmlReport($repName);

    if ( scalar(keys(%$rep)) < 2 ) {
        logMessage(sprintf(gettext("Fatal Error: Couldn't get report configuration information %s.  Exiting."), $repConf));
		exit 1;
    }

    # Clear unnecessary filters
    for my $filt (keys %$rep) {
        #$rep->{$filt} =~ s/\s+//g;  # repname won't be in here, so no spaces
        if ( $rep->{$filt} eq "-" || $rep->{$filt} eq 'All' ||
            $rep->{$filt} eq '*' )
        {
            delete($rep->{$filt});
        }
    }

	#mark
	my $filts = Immunix::Reports::getFilterList($rep);

    # Mail Type - csv, html, or both
    if ( $rep->{'csv'} eq '1' && $rep->{'html'} eq '1' ) {
		$rep->{'mailType'} = 'both';
	} elsif ( $rep->{'csv'} eq '1' && $rep->{'html'} eq '0' ) {
		$rep->{'mailType'} = 'csv';
    } elsif ( $rep->{'csv'} eq '0' && $rep->{'html'} eq '1' ) {
		$rep->{'mailType'} = 'html';
	} else {
		$rep->{'mailType'} = 'none';
	}

    # Mail Address List
    if ( $rep->{'addr1'} ) { push(@mailList,$rep->{'addr1'}); }
    if ( $rep->{'addr2'} ) { push(@mailList,$rep->{'addr2'}); }
    if ( $rep->{'addr3'} ) { push(@mailList,$rep->{'addr3'}); }


    $rep->{'startTime'} = $rep->{'time'};
    $rep->{'endTime'} = time;
    $rep->{'filters'} = %$filts;	# mark ($filts needs to be hash, not hash ref
	$rep->{'mailList'} = \@mailList;

	return $rep;

}

# Updates last-run time sig. in reports.conf
sub updateConf {

    my ($repName, $repConf) = @_;
    my $time = $repConf->{'endTime'};

    if (! $time) { $time = time; }

    #if ( ref($repName) ) { $repName = $repName->{'base'}; }

    my $repCfFile = '/etc/apparmor/reports.conf';
    my $tmpConf = '/var/log/apparmor/reports/reports.conf';

    if ( open(OCF, "<$repCfFile") ) {

        if ( open(TCF, ">$tmpConf") ) {

            my $nameFlag = 0;

            while(<OCF>) {

                chomp;

                if ( /\<name\>/ ) {
                    #my $name = (split(/\"/, $_))[1];
                    /\<name\>(.+)\<\/name\>/;
                    my $name = $1;
                    if ( $name eq $repName->{'base'} ) {
                        $nameFlag = 1;
                    }
                } elsif ( $nameFlag == 1 ) {
                    if ( /\<time\>/ ) {
                        $_ = "\t<time>$time</time>";
                        $nameFlag = 0;
                    }
                }

                print TCF "$_\n";
            }

            close TCF;

        } else {
            #logMessage("Error: Couldn't open (tmp) $tmpConf.  ESS report execution not tracked.");
            logMessage(sprintf(gettext("Error: Couldn't open (tmp) %s.  ESS report execution not tracked."), $tmpConf));
        }

        close OCF;
    } else {
        logMessage(sprintf(gettext("Error: Couldn't open %s.  ESS report execution not tracked."), $repCfFile));
    }

    if ( -e $repCfFile && -e $tmpConf ) {
        Immunix::Reports::updateFiles($repCfFile,$tmpConf);
    }

    return 0;

}

sub sendMail {

    my ($repName,$repFile,$repConf) = @_;
    my $repCfFile = '/etc/apparmor/reports.conf';

    if (! $repName || ! $repFile) { 
        logMessage(gettext("Error: Necessary input missing.  Unable to generate report and send mail."));
		return 1; 
	}

	my $baseName = $repName->{'base'};

	my $host = `hostname -f`;
    chomp($host);

	my $toMail = $repConf->{'mailList'};
    my $rec = undef;

	if ( @$toMail > 0 ) {
		for (@$toMail) { 
			if ( $repConf->{'mailType'} ) { 
				if ( $repConf->{'mailType'} eq 'csv' ) { 
		            #`/usr/bin/mail $_ -s $baseName -r "AppArmor_Reporting\@$host" -a $repFile < /dev/null`;
		            `/usr/bin/mail $_ -s $baseName -r "AppArmor_Reporting\@$host" -a $repFile < $headerFile`;
				} elsif ( $repConf->{'mailType'} eq 'html' ) {
					$repFile =~ s/csv/html/;
		            #`/usr/bin/mail $_ -s $baseName -r "AppArmor_Reporting\@$host" -a $repFile < /dev/null`;
		            `/usr/bin/mail $_ -s $baseName -r "AppArmor_Reporting\@$host" -a $repFile < $headerFile`;
				} elsif ( $repConf->{'mailType'} eq 'both' ) {
					my $rep2 = $repFile;
					$rep2 =~ s/csv/html/;
		            #`/usr/bin/mail $_ -s $baseName -r "AppArmor_Reporting\@$host" -a $repFile -a $rep2 < /dev/null`;
		            `/usr/bin/mail $_ -s $baseName -r "AppArmor_Reporting\@$host" -a $repFile -a $rep2 < $headerFile`;
				}
			} else {
	            `/usr/bin/mail $_ -s $repName -r "AppArmor_Reporting\@$host" -a $repFile < $headerFile`;
			}
		}
	}

    return 0;
}

sub runEss {

	my ($repName,$repConf) = @_;
	#my ($repName,$date,$lastRun) = @_;

	my $ref = ();
    # getSevStats from ag_reports_ess
	my $db = Immunix::Reports::getEssStats($ref);

	my $header = makeHeader('ess',$repConf);

    if ( $repConf->{'startTime'} && $repConf->{'startTime'} > 0 ) {
		$ref->{'startdate'} = $repConf->{'startTime'};
    } else {
		$ref->{'startdate'} = 1104566401;
	}

	# CSV
	if ( $repConf->{'mailType'} eq 'csv' || $repConf->{'mailType'} eq 'both' ) {
		if ( open(ESS, ">$repName") ) {

			print ESS "$header->{'csv'}";

			for (@$db) {
				$_->{'startdate'} = Immunix::Reports::getDate("$repConf->{'startTime'}"); 
				$_->{'enddate'} = Immunix::Reports::getDate("$repConf->{'endTime'}"); 
				print ESS "$_->{'host'},$_->{'startdate'},$_->{'enddate'},";
				print ESS "$_->{'sevHi'},$_->{'sevMean'},$_->{'numRejects'},$_->{'numEvents'}\n";
			}

			close ESS;

		} else {
			logMessage(sprintf(gettext("Error: Couldn't open %s.  No ESS csv report generated."), $repName));
		}
	}

	# HTML 
	if ( $repConf->{'mailType'} eq 'html' || $repConf->{'mailType'} eq 'both' ) {
		$repName =~ s/csv/html/;
        if ( open(ESS, ">$repName") ) {

            print ESS "<html><body bgcolor='fffeec'><font face='Helvetica,Arial,Sans-Serif'>\n";
			print ESS "$header->{'html'}";
			print ESS gettext("<tr bgcolor='edefff'><td>Hostname</td><td>Start Date</td><td>End Date</td>");
			print ESS gettext("<td>Highest Severity</td><td>Mean Severity</td><td>Number of REJECTs</td>");
			print ESS gettext("<td>Number of Events</td></tr>\n");

            for (@$db) {
                $_->{'startdate'} = Immunix::Reports::getDate("$repConf->{'startTime'}");
                $_->{'enddate'} = Immunix::Reports::getDate("$repConf->{'endTime'}");
                print ESS "<tr><td>$_->{'host'}</td><td>$_->{'startdate'}</td><td>$_->{'enddate'}</td>";
                print ESS "<td>$_->{'sevHi'}</td><td>$_->{'sevMean'}</td><td>$_->{'numRejects'}</td>";
				print ESS "<td>$_->{'numEvents'}</td></tr>\n";
            }

			print ESS "</table><br></body></html>";
            close ESS;

        } else {
			logMessage(sprintf(gettext("Error: Couldn't open %s.  No ESS html report generated."), $repName));
        }
	}

	return 0;
}

sub runAud {

	my ($repName,$repConf) = @_;
	my $header = makeHeader('aud',$repConf);

	# CSV 
	if ( $repConf->{'mailType'} eq 'csv' || $repConf->{'mailType'} eq 'both' ) {
		if ( open(AUD, ">$repName") ) {

			my $db = Immunix::Reports::getCfInfo();

			print AUD "$header->{'csv'}";

			for (@$db) {
				print AUD "$_->{'host'},$_->{'date'},$_->{'prog'}, $_->{'prof'}, $_->{'pid'},";
				if ( $_->{'state'} eq 'confined' ) {
					print AUD "$_->{'state'},$_->{'type'}\n"; 
				} else {
					print AUD "$_->{'state'}, - \n"; 
				}
			}

			close AUD;

		} else {
			logMessage(sprintf(gettext("Error: Couldn't open %s.  No AUD report generated."), $repName));
		}
	}

	# HTML
	if ( $repConf->{'mailType'} eq 'html' || $repConf->{'mailType'} eq 'both' ) {

		$repName =~ s/csv/html/;
		if ( open(AUD, ">>$repName") ) {

			my $db = Immunix::Reports::getCfInfo();

            print AUD "<html><body bgcolor='fffeec'><font face='Helvetica,Arial,Sans-Serif'>\n";
			print AUD gettext("<h3>Applications Audit Report - Generated by AppArmor</h3>\n");
            print AUD "$header->{'html'}";
            print AUD gettext("<tr bgcolor='edefff'><td>Hostname</td><td>Date</td><td>Application</td>");
            print AUD gettext("<td>Profile</td><td>PID</td><td>State</td><td>AppArmor Profile</td></tr>\n");

			for (@$db) {
				print AUD "<tr><td>$_->{'host'}</td><td>$_->{'date'}</td><td>$_->{'prog'}</td><td>$_->{'prof'}</td>";
				if ( $_->{'state'} eq 'confined'  ) {
					print AUD "<td>$_->{'pid'}</td><td>$_->{'state'}</td><td>$_->{'type'}</td></tr>\n"; 
				} else {
					print AUD "<td>$_->{'pid'}</td><td>$_->{'state'}</td><td align='center'>&nbsp;-&nbsp;</td></tr>\n"; 
				}
			}

			print AUD "<br></table></body></html>\n";

			close AUD;

		} else {
			logMessage(sprintf(gettext("Error: Couldn't open %s.  No AUD report generated."), $repName));
		}
	}

	return 0;
}

sub runSir {

	my ($repName,$repFile,$repConf) = @_;

    my $filts = Immunix::Reports::setFormFilters($repConf);
    $filts = Immunix::Reports::rewriteFilters($filts);

	my $start = '1104566401';			# default start, Jan 1, 2005

    if ( $repConf->{'startTime'} && $repConf->{'startTime'} > 0 ) {
		$start = $repConf->{'startTime'};
    }

	my $db = Immunix::Reports::grabEvents($filts,$start);

	if ($repConf->{'start'} && $repConf->{'start'} > $start ) {
		$start = localtime($repConf->{'startTime'});
	}

	my $end = localtime($repConf->{'endTime'});

	my $header = makeHeader('sir',$repConf,$filts);

	# CSV
	if ( $repConf->{'mailType'} =~ /csv/ || $repConf->{'mailType'} =~ /both/ )  {

		if ( open(SIR, ">$repFile") ) {

			#my $ref = getSirFilters($repName);

			# Write Header
			print SIR "$header->{'csv'}";

			#Immunix::Reports::exportFormattedText($repName,$repFile,$db);		# Replaced stuff below

			for (@$db) {
		        print SIR "$_->{'host'},$_->{'time'},$_->{'prog'},$_->{'profile'},";
		        print SIR "$_->{'pid'},$_->{'severity'},$_->{'mode_deny'},$_->{'mode_req'},";
		        print SIR "$_->{'resource'},$_->{'sdmode'},$_->{'op'},$_->{'attr'},";
		        print SIR "$_->{'name_alt'},$_->{'parent'},$_->{'active_hat'},";
		        print SIR "$_->{'net_family'},$_->{'net_proto'},$_->{'net_socktype'}\n";

				# old aa-eventd
				#print SIR "$_->{'host'},$_->{'date'},$_->{'prog'},$_->{'profile'},$_->{'pid'},";
				#print SIR "$_->{'severity'},$_->{'mode'},$_->{'resource'},$_->{'sdmode'}\n"; 
			}

			close SIR;

		} else {
			logMessage(sprintf(gettext("Error: Couldn't open %s.  No SIR report generated."), $repFile));
		}
	}

	# HTML
	if ( $repConf->{'mailType'} =~ /html/ || $repConf->{'mailType'} =~ /both/ )  {
		#my $repName = $repConf->{'name'};
		#$repName =~ s/csv/html/;
		$repFile =~ s/csv/html/;

		#Immunix::Reports::exportLog($repName,$repFile,$db);
		Immunix::Reports::exportLog($repFile,$db,$header->{'html'});
	}

	return 0;
}

# Main
################################################################################
if (! $ARGV[0]) {
	logMessage(gettext("Error: No arguments passed--Unable to execute reports.  Exiting."));
	exit 1;
}

my $args = undef;
my @dirList = ();
#my $date = localtime;
my $dir = '/var/log/apparmor/reports-archived';
my $repName = undef;

$repName->{'base'} = $ARGV[0];						# Report Name
$repName->{'quoted'} = "\"$repName->{'base'}\"";	# Quoted Report Name

logMessage(sprintf(gettext("Executing Scheduled Report: %s"), $repName->{'base'}));

my $repConf = getRepConf($repName);

if ( $repConf->{'exportpath'} && -d $repConf->{'exportpath'} ) {
	$dir = $repConf->{'exportpath'};
}

my $repFile = getRepFileName($repName,$dir,$repConf);

if ($repName->{'base'} =~ /Executive.Security.Summary/) {
	runEss($repFile,$repConf);
} elsif ($repName->{'base'} =~ /Applications.Audit/) {
	runAud($repFile,$repConf);
} else {
	# Security Incident Report
	runSir($repName,$repFile,$repConf);
}

if ( ! -e $headerFile ) { $headerFile = '/dev/null'; }

sendMail($repName,$repFile,$repConf);
updateConf($repName,$repConf);

exit 0;

