/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* Copyright (C) 2001 Johan Dahlin <zilch.am@home.se>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Johan Dahlin
 */

#include <Python.h>
#include <sysmodule.h>
#include <bonobo-activation/bonobo-activation.h>
#include <orbit-python.h>

#ifdef WITH_PYPOPT
#include <pypopt/pypopt.h>
#endif /* WITH_PYPOPT */

static CORBA_ORB_PyObject *orb = NULL;
static PyObject           *dict;

static PyObject *
wrap_ba_orb_get (PyObject *self,
		  PyObject *args)
{
        if (bonobo_activation_is_initialized ()) {
                Py_INCREF (Py_None);
                return Py_None;
        }
		
	if (!PyArg_ParseTuple (args, ":bonobo.activation.orb_get")) {
		return NULL;
	}

	if (orb != NULL) {
		Py_INCREF (orb);
		return (PyObject*)orb;
	}

	orb = PyORBit_ORB_New (bonobo_activation_orb_get ());
	if (orb == NULL) {
		Py_INCREF (Py_None);
		return Py_None;
	}

	PyDict_SetItemString (dict, "is_initialized", PyInt_FromLong (1L));

	Py_INCREF ((PyObject*) orb);
	return (PyObject*) orb;
}

static PyObject *
wrap_ba_get_popt_table_name (PyObject *self,
			     PyObject *args)
{
	char *name;
    
	if (!PyArg_ParseTuple (args, ":bonobo.activation.get_popt_table_name")) {
		return NULL;
	}
	
	name = bonobo_activation_get_popt_table_name ();
	if (name == NULL) {
		return NULL;
	}
    
	return Py_BuildValue ("s", name);
    
}

static PyObject *
wrap_ba_query (PyObject *self,
	       PyObject *args)
{
	PyObject              *pysort;
	guint                  i;
	gchar                 *query;
	guint                  len;
	gchar                **sort;
	CORBA_Environment      ev;
	Bonobo_ServerInfoList *infolist;
	PyObject              *pyinfo;
	Bonobo_ServerInfo      server;
	PyObject              *infodict;
	PyObject              *propdict;
	guint                  j;
	PyObject              *value;
/*	guint                   k; */
/*	GString*                str; */
/*	gchar                  *ret; */
	
	pysort = NULL;
	i = 0;

	if (!PyArg_ParseTuple (args, "s|O:bonobo.activation.query",
			       &query, &pysort)) {
		return NULL;
	}

	/* Create sortlist */
	if (pysort != NULL) {
		if (!PyList_Check (pysort)) {
			/* FIXME: Raise exception */
			return NULL;
		}
		
		len = PyList_Size (pysort);
		sort = malloc (len * sizeof (char*));
		for (i = 0; i < len; i++)
			sort[i] = strdup (PyString_AsString (PyList_GetItem (pysort, i)));
		sort[i] = NULL;
	} else {
		sort = NULL;
	}

	CORBA_exception_init (&ev);

	infolist = bonobo_activation_query (query, sort, &ev);
	
	if (infolist == NULL) {
		CORBA_exception_free (&ev);
		Py_INCREF (Py_None);
		return Py_None;
	}
	
	/* Create a list of dictionaries */
	pyinfo = PyList_New (infolist->_length);
	for (i = 0; i < infolist->_length; i++) {
		server = infolist->_buffer[i];
		
		infodict = PyDict_New();

		PyDict_SetItemString (infodict, "iid", PyString_FromString (server.iid));
		
		PyDict_SetItemString (infodict, "server_type",
				      PyString_FromString (server.server_type));
		
		PyDict_SetItemString (infodict, "location_info",
				      PyString_FromString (server.location_info));
		
		PyDict_SetItemString (infodict, "username",
				      PyString_FromString (server.username));
		
		PyDict_SetItemString (infodict, "hostname",
				      PyString_FromString (server.hostname));
		
		PyDict_SetItemString (infodict, "domain",
				      PyString_FromString (server.domain));

		propdict = PyDict_New ();
		for (j = 0; j < server.props._length; j++) {
			value = Py_None;
			
			switch (server.props._buffer[j].v._d) {
			case Bonobo_ACTIVATION_P_STRINGV:
			case Bonobo_ACTIVATION_P_STRING:
				value = Py_BuildValue ("s", server.props._buffer[j].v._u.value_string);
				break;
			case Bonobo_ACTIVATION_P_NUMBER:
				value = Py_BuildValue ("i", server.props._buffer[j].v._u.value_number);
				break;
			case Bonobo_ACTIVATION_P_BOOLEAN:
				value = Py_BuildValue ("b", server.props._buffer[j].v._u.value_boolean);
				break;
			}
			
			if (value != Py_None) {
				PyDict_SetItemString (propdict,
						      server.props._buffer[j].name,
						      value);
				Py_DECREF (value);
			}
		}
		
		PyDict_SetItemString (infodict, "props", propdict);
		Py_DECREF (propdict);

		PyList_SetItem (pyinfo, i, infodict);
	}
	
	if (infolist) {
		CORBA_free (infolist);
	}

	CORBA_exception_free (&ev);
	
	Py_INCREF (pyinfo);
	return pyinfo;
}

static PyObject *
wrap_ba_activate (PyObject *self,
		  PyObject *args)
{
	gchar              *requirements;
	PyObject           *pysort;
	glong               flags;
	gchar              *ret_aid;
	guint               len;
	char              **sort;
	guint               i;
	CORBA_Environment   ev;
	CORBA_Object        corba_object;

	pysort = NULL;
	flags = 0L;
	ret_aid = NULL;
	
	if (!PyArg_ParseTuple (args, "z|Ols:bonobo.activation.activate",
			       &requirements,
			       &pysort,
			       &flags,
			       &ret_aid)) {
		return NULL;
	}
	
	if (pysort != NULL) {
		len = PyList_Size (pysort);
		sort = malloc (len+1 * sizeof(char*));
		for (i = 0; i < len; i++) {
			sort[i] = strdup (PyString_AsString (PyList_GetItem (pysort, i)));
		}
		sort[i] = NULL;
	} else {
		sort = NULL;
	}
	
	CORBA_exception_init (&ev);
	corba_object = bonobo_activation_activate (requirements, sort, flags,
						   (Bonobo_ActivationID *)ret_aid, &ev);
	
	if (!corba_object) {
		CORBA_exception_free (&ev);
		Py_INCREF (Py_None);
		return Py_None;
	}
	
	CORBA_exception_free (&ev);
	
	return (PyObject*)PyORBit_Object_New (corba_object);
}

/* TODO: Handle third argument */
static PyObject *
wrap_ba_activate_from_id (PyObject *self,
			  PyObject *args)
{
	CORBA_Object         object = NULL;
	CORBA_Environment    ev;
	char                *activation_id;
	Bonobo_ActivationID *ret_aid = NULL;
	long                 flags = 0;
	
	if (!PyArg_ParseTuple (args, "s|lO:bonobo.activation.activate_from_id",
			       &activation_id, &flags, &ret_aid)) {
		return NULL;
	}

	CORBA_exception_init (&ev);

	object = bonobo_activation_activate_from_id (activation_id, 
						     flags, 
						     (Bonobo_ActivationID *)ret_aid, 
						     &ev);
	
	if (ev._major != CORBA_NO_EXCEPTION) {
		PyErr_Format (PyExc_TypeError, 
			      "Could not activate object: %s", ev._id);
		return NULL;
	}
        
	CORBA_exception_free (&ev);
	
	return (PyObject*)PyORBit_Object_New (object);
}

static PyObject *
wrap_ba_active_server_register (PyObject *self,
				PyObject *args)
{
	char                     *iid;
	CORBA_Object              obj;
	CORBA_Environment         ev;
	Bonobo_RegistrationResult retval;
	        
	if (!PyArg_ParseTuple (args, 
			       "sO:bonobo.activation.active_server_register",
			       &iid, &obj)) {
		return NULL;
	}
	
	CORBA_exception_init (&ev);
	
	retval = bonobo_activation_active_server_register (iid, obj);
	
	if (ev._major != CORBA_NO_EXCEPTION) {
		PyErr_Format (PyExc_TypeError, 
			      "Could not register object: %s", ev._id);
		return NULL;
	}
	
	CORBA_exception_free (&ev);
	
	Py_INCREF (Py_None);
	return Py_None;
}

static struct PyMethodDef activationmodule_methods[] =
{
	{ "orb_get",                  wrap_ba_orb_get,                   METH_VARARGS },
	{ "get_popt_table_name",      wrap_ba_get_popt_table_name,       METH_VARARGS },
	{ "query",                    wrap_ba_query,                     METH_VARARGS },
	{ "activate",                 wrap_ba_activate,                  METH_VARARGS },
	{ "activate_from_id",         wrap_ba_activate_from_id,          METH_VARARGS },
	{ "active_server_register",   wrap_ba_active_server_register,    METH_VARARGS },
#if 0
	{ "active_server_unregister", wrap_oaf_active_server_unregister, METH_VARARGS },
	{ "name_service_get",         wrap_oaf_name_service_get,         METH_VARARGS },
	{ "activation_context_get",   wrap_oaf_activation_context_get,   METH_VARARGS },
	{ "hostname_get",             wrap_oaf_hostname_get,             METH_VARARGS },
	{ "session_name_get",         wrap_oaf_session_name_get,         METH_VARARGS },
	{ "domain_get",               wrap_oaf_domain_get,               METH_VARARGS },
#endif	
	{ NULL, NULL }
};

void
initactivation (void)
{
	PyObject  *av;
        int        argc;
        int        i;
        char     **argv;

	PyObject *activation_module;
	
#ifdef WITH_PYPOPT
	init_pypopt ();
#endif /* WITH_PYPOPT */
	
	init_orbit_python ();
		
	activation_module = Py_InitModule ("bonobo.activation", activationmodule_methods);
	dict = PyModule_GetDict (activation_module);

	av = PySys_GetObject ("argv");
        argc = PyList_Size (av);

        argv = malloc (argc * sizeof (char*));
        for (i = 0; i < argc; i++) {
                argv[i] = strdup (PyString_AsString (PyList_GetItem (av, i)));
	}
	
	bonobo_activation_init (argc, argv);
	
	PySys_SetArgv (argc, argv);
	if (argv != NULL) {
		for (i = 0; i < argc; i++) {
			g_free(argv[i]);
		}
		g_free (argv);
	}
	
#ifdef WITH_PYPOPT
        PyDict_SetItemString (dict, "popt_options",
			      PyPoptOption_New (bonobo_activation_popt_options));
#endif /* WITH_PYPOPT */
   
	if (PyErr_Occurred ()) {
		Py_FatalError ("Can't initialize module activation\n");
	}
}
