/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import org.gnu.glib.Handle;
import org.gnu.glib.Type;

/**
 * The ProgressBar is typically used to display the progress of a long 
 * running operation. It provides a visual clue that processing is 
 * underway. The ProgressBar can be used in two different modes: 
 * percentage mode and activity mode.
 *
 * <p>When an application can determine how much work needs to take place 
 * (e.g. read a fixed number of bytes from a file) and can monitor its 
 * progress, it can use the ProgressBar in percentage mode and the user 
 * sees a growing bar indicating the percentage of the work that has been 
 * completed. In this mode, the application is required to call 
 * {@link #setFraction(double)} periodically to update the progress bar.
 *
 * <p>When an application has no accurate way of knowing the amount of work 
 * to do, it can use the ProgressBar in activity mode, which shows activity 
 * by a block moving back and forth within the progress area. In this mode, 
 * the application is required to call {@link #pulse()} perodically to 
 * update the progress bar.
 *
 * <p>There is quite a bit of flexibility provided to control the appearance 
 * of the ProgressBar. Functions are provided to control the orientation of 
 * the bar, optional text can be displayed along with the bar, and the step 
 * size used in activity mode can be set.
 */
public class ProgressBar extends Progress 
{
	/**
	 * Constructs a new ProgressBar widget
	 */
	public ProgressBar(){
		super(gtk_progress_bar_new ());
	}
	
	/**
	 * Construct a new ProgressBar passing a handle to a native
	 * widget resource.
	 * 
	 * @param handle The handle to the native ProgressBar.
	 */
	public ProgressBar(Handle handle) {
	    super(handle);
	}

	/**
	 * Indicates that some progress is made, but you don't know how much. 
	 * Causes the progress bar to enter "activity mode," where a block bounces 
	 * back and forth. Each call to pulse() causes the block to move by a 
	 * little bit (the amount of movement per pulse is determined by 
	 * {@link #setPulseStep(double)}).
	 */
	public void pulse(){
		gtk_progress_bar_pulse(getHandle());
	}

	/**
	 * Causes the given text to appear superimposed on the progress bar.
	 * @param text The string to appear on the progress bar
	 */
	public void setText(String text){
		gtk_progress_bar_set_text(getHandle(), text);
	}

	/**
	 * Causes the progress bar to "fill in" the given fraction of the bar. 
	 * The fraction should be between 0.0 and 1.0, inclusive.
	 * @param fraction The fraction of the progress bar which should be filled
	 * in,
	 */
	public void setFraction(double fraction){
		gtk_progress_bar_set_fraction(getHandle(), fraction);
	}
	
	/**
	 * Returns the current value of the display.  This value will be
	 * returned as a value between 0.0 and 1.0.
	 */
	public double getFraction() {
		return gtk_progress_bar_get_fraction(getHandle());
	}

	/**
	 * Sets the fraction of total progress bar length to move the bouncing 
	 * block for each call to {@link #pulse()}.
	 * @param fraction  fraction between 0.0 and 1.0
	 */
	public void setPulseStep(double fraction){
		gtk_progress_bar_set_pulse_step(getHandle(), fraction);
	}

	/**
	 * Causes the progress bar to switch to a different orientation 
	 * (left-to-right, right-to-left, top-to-bottom, or bottom-to-top).
	 * @see ProgressBarOrientation
	 */
	public void setOrientation(ProgressBarOrientation orientation){
		gtk_progress_bar_set_orientation(getHandle(), orientation.getValue());
	}

	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static Type getType() {
		return new Type(gtk_progress_bar_get_type());
	}


    native static final protected int gtk_progress_bar_get_type ();
    native static final protected Handle gtk_progress_bar_new ();
    native static final protected void gtk_progress_bar_pulse (Handle pbar);
    native static final protected void gtk_progress_bar_set_text (Handle pbar, String text);
    native static final protected void gtk_progress_bar_set_fraction (Handle pbar, double fraction);
    native static final protected void gtk_progress_bar_set_pulse_step (Handle pbar, double fraction);
    native static final protected void gtk_progress_bar_set_orientation (Handle pbar, int orientation);
    native static final protected String gtk_progress_bar_get_text (Handle pbar);
    native static final protected double gtk_progress_bar_get_fraction (Handle pbar);
    native static final protected double gtk_progress_bar_get_pulse_step (Handle pbar);
    native static final protected int gtk_progress_bar_get_orientation (Handle pbar);

    /* Deprecated functions.
    native static final private Handle gtk_progress_bar_new_with_adjustment(Handle adjustment);
    native static final private void gtk_progress_bar_set_bar_style(Handle pbar, int style);
    native static final private void gtk_progress_bar_set_discrete_blocks(Handle pbar, int blocks);
    native static final private void gtk_progress_bar_set_activity_step(Handle pbar, int step);
    native static final private void gtk_progress_bar_set_activity_blocks(Handle pbar, int blocks);
    native static final private void gtk_progress_bar_update(Handle pbar, double percentage);
    */
}

