/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2005 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.glib;

import org.gnu.javagnome.Handle;
import org.gnu.javagnome.Struct;

/**
 * The MainLoop represents a main event loop. After it has been created, one
 * should call {@link #run()} to cause it to start processing the events. To
 * exit the main loop, call {@link #quit}.
 * 
 * @author Ismael Juma <ismael@juma.me.uk>
 */
public class MainLoop extends MemStruct    {
    
    /**
     * Creates a new MainLoop with the default context. After creation, the
     * MainLoop is not running.
     */
    public MainLoop() {
        super(g_main_loop_new(Struct.getNullHandle(), false));
    }

    /**
     * Runs a main loop until {@link #quit()} is called on the loop.
     *
     */
    public void run() {
        g_main_loop_run(getHandle());
    }

    /**
     * Stops the main loop from running.
     *
     */
    public void quit() {
        g_main_loop_quit(getHandle());
    }

    /**
     * Checks to see if the main loop is currently being run via
     * {@link #run()}.
     * @return TRUE if the main loop is currently being run.
     */
    public boolean isRunning() {
        return g_main_loop_is_running(getHandle());
    }

    /* Native methods */

    native static private Handle g_main_loop_new(Handle context, boolean
            is_running);
    native static private void g_main_loop_run(Handle loop);

    native static private void g_main_loop_quit(Handle loop);

    native static private boolean g_main_loop_is_running(Handle loop);
}
