/*
  This file is part of TALER
  Copyright (C) 2014-2024 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License along with
  TALER; see the file COPYING.LIB.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file taler_merchant_donau.h
 * @brief Structs to interact with the donau.
 * @author Bohdan Potuzhnyi
 * @author Vlada Svirsh
 */
#ifndef _TALER_MERCHANT_DONAU_H
#define _TALER_MERCHANT_DONAU_H

#include <taler/taler_util.h>
#include <taler/taler_error_codes.h>
#include <taler/taler_exchange_service.h>
#include "taler_merchant_service.h"
#include <donau/donau_service.h>
#include <donau/donau_util.h>
#include <donau/donaudb_plugin.h>

/**
 * Structure to hold charity details for the Donau service.
 */
struct TALER_MERCHANT_DONAU_Charity
{
  /**
  * name of the charity
  */
  const char *name;

  /**
  * charity url
  */
  const char *charity_url;

  /**
  * public key of the charity
  */
  struct DONAU_CharityPublicKeyP charity_pub;

  /**
  * Max donation amount for this charitiy and @e current_year.
  */
  struct TALER_Amount max_per_year;

  /**
  * Current amount of donation receipts for @e current_year.
  */
  struct TALER_Amount receipts_to_date;

  /**
  * current year
  */
  uint64_t current_year;

  /**
  * The unique identifier of the charity.
  */
  uint64_t charity_id;
};


/**
 * Structure to hold Donau instance details from the database.
 */
struct TALER_MERCHANTDB_DonauInstance
{
  /**
   * Donau instance serial
   */
  uint64_t donau_instance_serial;

  /**
   * The URL for the Donau instance.
   */
  char *donau_url;

  /**
   * The name of the charity associated with the Donau instance.
   */
  char *charity_name;

  /**
   * Pointer to the public key of the charity, used for cryptographic operations.
   * This is represented as an EDDSA public key structure.
   */
  struct DONAU_CharityPublicKeyP *charity_pub_key;

  /**
   * A unique identifier for the charity in the Donau instance.
   */
  uint64_t charity_id;

  /**
   * The maximum allowable amount for donations to this charity in the current year.
   * This is tracked for regulatory or internal business constraints.
   */
  struct TALER_Amount charity_max_per_year;

  /**
   * The total amount of donations received by the charity in the current year.
   * This field helps track progress toward the yearly donation limit.
   */
  struct TALER_Amount charity_receipts_to_date;

  /**
   * The current year being tracked for donations.
   * This is used to differentiate donation data between years.
   */
  int64_t current_year;

  /**
   * A JSON object containing key information specific to the Donau instance,
   * such as cryptographic keys or other relevant details.
   */
  json_t *donau_keys_json;
};


/**
 * Callback function typically used by `select_donau_instances` to handle
 * the details of each Donau instance retrieved from the database.
 *
 * @param cls Closure to pass additional context or data to the callback function.
 * @param donau_instance_serial Serial number of the Donau instance in the merchant database.
 * @param donau_url The URL of the Donau instance.
 * @param charity_name The name of the charity associated with the Donau instance.
 * @param charity_pub_key Pointer to the charity's public key used for cryptographic operations.
 * @param charity_id The unique identifier for the charity within the Donau instance.
 * @param charity_max_per_year Maximum allowed donations to the charity for the current year.
 * @param charity_receipts_to_date Total donations received by the charity so far in the current year.
 * @param current_year The year for which the donation data is being tracked.
 * @param donau_keys_json JSON object containing additional key-related information for the Donau instance, NULL if not (yet) available.
 */
typedef void
(*TALER_MERCHANTDB_DonauInstanceCallback)(
  void *cls,
  uint64_t donau_instance_serial,
  const char *donau_url,
  const char *charity_name,
  const struct DONAU_CharityPublicKeyP *charity_pub_key,
  uint64_t charity_id,
  const struct TALER_Amount *charity_max_per_year,
  const struct TALER_Amount *charity_receipts_to_date,
  int64_t current_year,
  const json_t *donau_keys_json
  );

/**
 * Callback function typically used by `select_donau_instances` to handle
 * the details of each Donau instance retrieved from the database.
 *
 * @param cls Closure to pass additional context or data to the callback function.
 * @param donau_url The URL of the Donau instance.
 */
typedef void
(*TALER_MERCHANTDB_DonauInstanceFilteredCallback)(
  void *cls,
  const char *donau_url
  );

/**
 * SPECIFICATION FOR REQUESTS /donau
 */

/**
 * Handle for a GET /donau operation.
 */
struct TALER_MERCHANT_DonauInstanceGetHandle;

/**
 * Individual Donau instance details.
 */
struct TALER_MERCHANT_DonauInstanceEntry
{
  /**
   * Serial number of the Donau instance in merchant database.
   */
  uint64_t donau_instance_serial;

  /**
   * The URL of the Donau service.
   */
  const char *donau_url;

  /**
   * Name of the charity associated with the Donau instance.
   */
  const char *charity_name;

  /**
   * Public key of the charity.
   */
  struct DONAU_CharityPublicKeyP charity_pub_key;

  /**
   * ID of the charity on Donau.
   */
  uint64_t charity_id;

  /**
   * Maximum donation amount per year allowed for the charity.
   */
  struct TALER_Amount charity_max_per_year;

  /**
   * Total donations received by the charity in the current year.
   */
  struct TALER_Amount charity_receipts_to_date;

  /**
   * The current year being tracked for donations.
   */
  int64_t current_year;

  /**
   * Additional key information related to the Donau instance.
   */
  const struct DONAU_Keys *donau_keys;
};

/**
 * Response for a GET /donau request.
 */
struct TALER_MERCHANT_DonauInstanceGetResponse
{
  /**
   * HTTP response details.
   */
  struct TALER_MERCHANT_HttpResponse hr;

  /**
   * Details depending on the HTTP status.
   */
  union
  {
    /**
     * Details for #MHD_HTTP_OK.
     */
    struct
    {
      /**
       * Length of the @e donau_instances array.
       */
      unsigned int donau_instances_length;

      /**
       * Details about the Donau instance.
       */
      struct TALER_MERCHANT_DonauInstanceEntry *donau_instances;
    } ok;
  } details;
};

/**
 * Function called with the result of the GET /donau operation.
 *
 * @param cls closure
 * @param dir response details
 */
typedef void
(*TALER_MERCHANT_DonauInstanceGetCallback)(
  void *cls,
  const struct TALER_MERCHANT_DonauInstanceGetResponse *dir);

/**
 * Initiate the GET /donau request.
 *
 * @param ctx CURL context
 * @param backend_url base URL for the backend
 * @param cb callback function to handle the response
 * @param cb_cls closure for the callback function
 * @return the handle for the operation, or NULL on error
 */
struct TALER_MERCHANT_DonauInstanceGetHandle *
TALER_MERCHANT_donau_instances_get (
  struct GNUNET_CURL_Context *ctx,
  const char *backend_url,
  TALER_MERCHANT_DonauInstanceGetCallback cb,
  void *cb_cls);


/**
 * Cancel the GET /donau instances operation.
 *
 * @param dgh request to cancel.
 */
void
TALER_MERCHANT_donau_instances_get_cancel (
  struct TALER_MERCHANT_DonauInstanceGetHandle *dgh);


/**
 * Handle for a POST /donau operation.
 */
struct TALER_MERCHANT_DonauInstancePostHandle;


/**
 * Function called with the result of the POST /donau operation.
 *
 * @param cls closure
 * @param hr HTTP response data
 */
typedef void
(*TALER_MERCHANT_DonauInstancePostCallback)(
  void *cls,
  const struct TALER_MERCHANT_HttpResponse *hr);

/**
 * Post a new Donau instance.
 * Sends a POST request to the backend to register a new Donau instance
 * with the provided charity details.
 *
 * @param ctx the context
 * @param backend_url the backend URL for the operation
 * @param charity the details of the Donau charity to be posted
 * @param auth_token the authentication token for the operation
 * @param cb function to call with the operation result
 * @param cb_cls closure for @a cb
 * @return the instance handle, or NULL upon error
 */
struct TALER_MERCHANT_DonauInstancePostHandle *
TALER_MERCHANT_donau_instances_post (
  struct GNUNET_CURL_Context *ctx,
  const char *backend_url,
  const struct TALER_MERCHANT_DONAU_Charity *charity,
  const char *auth_token,
  TALER_MERCHANT_DonauInstancePostCallback cb,
  void *cb_cls);

/**
 * Cancel the POST /donau instances operation.
 * This must not be called after the callback has been invoked.
 *
 * @param dph request to cancel
 */
void
TALER_MERCHANT_donau_instances_post_cancel (
  struct TALER_MERCHANT_DonauInstancePostHandle *dph);

/**
 * Handle for a DELETE /donau/$charity_id operation.
 */
struct TALER_MERCHANT_DonauInstanceDeleteHandle;

/**
 * Callback for DELETE /donau/$charity_id operation.
 *
 * @param cls Closure to pass context.
 * @param hr HTTP response data.
 */
typedef void
(*TALER_MERCHANT_DonauInstanceDeleteCallback)(
  void *cls,
  const struct TALER_MERCHANT_HttpResponse *hr);

/**
 * Initiates the DELETE /donau/$charity_id operation.
 *
 * @param ctx CURL context
 * @param backend_url Base URL for the backend
 * @param charity_id The ID of the charity to delete
 * @param cb Callback function to handle the response
 * @param cb_cls Closure for @a cb
 * @return the handle for the operation, or NULL on error
 */
struct TALER_MERCHANT_DonauInstanceDeleteHandle *
TALER_MERCHANT_donau_instance_delete (
  struct GNUNET_CURL_Context *ctx,
  const char *backend_url,
  uint64_t charity_id,
  TALER_MERCHANT_DonauInstanceDeleteCallback cb,
  void *cb_cls);

/**
 * Cancel the DELETE /donau/$charity_id operation.
 *
 * @param ddh Handle for the operation to cancel.
 */
void
TALER_MERCHANT_donau_instance_delete_cancel (
  struct TALER_MERCHANT_DonauInstanceDeleteHandle *ddh);

#endif
