/*
This is part of the audio CD player library
Copyright (C)1998 Tony Arcieri

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.
*/

#include <stdio.h>
#include <string.h>
#include <mntent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <cdaudio.h>
#include <config.h>

#ifdef HAVE_LINUX_CDROM_H
#include <linux/cdrom.h>
#else
#ifdef HAVE_SUNDEV_SRREG_H
#include <sundev/srreg.h>
#else
#error Cannot locate a CD-ROM header file for your system
#endif
#endif

#ifdef HAVE_LINUX_UCDROM_H
#include <linux/ucdrom.h>
#endif

#define BLOCKING

void
cd_version(char *buffer, int len)
{
   snprintf(buffer, len, "%s version %s", PACKAGE, VERSION);
}

/* Initialize the CD-ROM for playing audio CDs */
int
cd_init(char *device)
{
   int cd_desc;
   
   FILE *mounts;
   struct mntent *mnt;

   if((mounts = setmntent(MOUNTED, "r")) == NULL) {
      perror("setmntent");
      exit(1);
   }
   while((mnt = getmntent(mounts)) != NULL) {
      if(strcmp(mnt->mnt_fsname, device) == 0) {
	 fprintf(stderr, "%s is currently mounted under %s.\n", mnt->mnt_fsname, mnt->mnt_dir);
	 endmntent(mounts);
	 exit(1);
      }
   }
   endmntent(mounts);

#ifdef BLOCKING
   if((cd_desc = open(device, O_RDONLY | O_NONBLOCK)) < 0)
#else
   if((cd_desc = open(device, O_RDONLY)) < 0)
#endif
     return -1;
   
   return cd_desc;
}

/* Update a CD status structure */
int
cd_stat(int cd_desc, struct disc_info *disc)
{
   struct cdrom_subchnl cdsc;
   struct cdrom_tochdr cdth;
   struct cdrom_tocentry cdte;
   int readtracks;
   int pos;
   
   cdsc.cdsc_format = CDROM_MSF;
   
   if(ioctl(cd_desc, CDROMSUBCHNL, &cdsc) < 0) {
      disc->disc_status = CD_ABSENT;
      
      return 0;
   }
   
   disc->disc_status = CD_PRESENT;
   
   if(ioctl(cd_desc, CDROMREADTOCHDR, &cdth) < 0)
     return -1;
      
   disc->disc_totaltracks = cdth.cdth_trk1;
      
   for(readtracks = 0; readtracks <= disc->disc_totaltracks; readtracks++) {
      if(readtracks == disc->disc_totaltracks)
	cdte.cdte_track = CDROM_LEADOUT;
      else
	cdte.cdte_track = readtracks + 1;
      
      cdte.cdte_format = CDROM_MSF;
      if(ioctl(cd_desc, CDROMREADTOCENTRY, &cdte) < 0)
	return -1;
	 
      disc->track[readtracks].track_pos.minutes = cdte.cdte_addr.msf.minute;
      disc->track[readtracks].track_pos.seconds = cdte.cdte_addr.msf.second;
      disc->track[readtracks].track_start = (disc->track[readtracks].track_pos.minutes * 60 + disc->track[readtracks].track_pos.seconds) * 75 + cdte.cdte_addr.msf.frame;
      
      if(readtracks > 0) {
	 pos = (disc->track[readtracks].track_pos.minutes * 60 + disc->track[readtracks].track_pos.seconds) - (disc->track[readtracks - 1].track_pos.minutes * 60 + disc->track[readtracks -1].track_pos.seconds);
	 disc->track[readtracks - 1].track_length.minutes = pos / 60;
	 disc->track[readtracks - 1].track_length.seconds = pos % 60;
      }
   }
            
   disc->disc_length.minutes = disc->track[disc->disc_totaltracks].track_pos.minutes;
   disc->disc_length.seconds = disc->track[disc->disc_totaltracks].track_pos.seconds;
   
   disc->disc_time.minutes = cdsc.cdsc_absaddr.msf.minute;
   disc->disc_time.seconds = cdsc.cdsc_absaddr.msf.second;
   
   disc->disc_frame = (cdsc.cdsc_absaddr.msf.minute * 60 + cdsc.cdsc_absaddr.msf.second) * 75 + cdsc.cdsc_absaddr.msf.frame;
   
   disc->disc_track = 0;
   while(disc->disc_track < disc->disc_totaltracks && disc->disc_frame >= disc->track[disc->disc_track].track_start)
     disc->disc_track++;
   
   pos = (disc->disc_frame - disc->track[disc->disc_track - 1].track_start) / 75;
 
   disc->track_time.minutes = pos / 60;
   disc->track_time.seconds = pos % 60;
   
   switch(cdsc.cdsc_audiostatus) {
    case CDROM_AUDIO_PLAY:
      disc->disc_mode = CDAUDIO_PLAYING;
      break;
    case CDROM_AUDIO_PAUSED:
      disc->disc_mode = CDAUDIO_PAUSED;
      break;
    case CDROM_AUDIO_NO_STATUS:
      disc->disc_mode = CDAUDIO_NOSTATUS;
      break;
    case CDROM_AUDIO_COMPLETED:
      disc->disc_mode = CDAUDIO_COMPLETED;
      break;
   }
   
   return 0;
}

/* Play frames from CD */
int
cd_play_frames(int cd_desc, int startframe, int endframe)
{
   struct cdrom_msf cdmsf;
   
   cdmsf.cdmsf_min0 = startframe / (60 * 75);
   cdmsf.cdmsf_sec0 = (startframe % (60 * 75)) / 75;
   cdmsf.cdmsf_frame0 = startframe % 75;
   cdmsf.cdmsf_min1 = endframe / (60 * 75);
   cdmsf.cdmsf_sec1 = (endframe % (60 * 75)) / 75;
   cdmsf.cdmsf_frame1 = endframe % 75;
   
   if(ioctl(cd_desc, CDROMSTART) < 0)
     return -1;
   if(ioctl(cd_desc, CDROMPLAYMSF, &cdmsf) < 0)
     return -1;
   
   return 0;
}

/* Play starttrack at position pos to endtrack */
int
cd_play_track_pos(int cd_desc, int starttrack, int endtrack, int startpos)
{
   struct disc_info disc;
   
   cd_stat(cd_desc, &disc);
   return cd_play_frames(cd_desc, disc.track[starttrack - 1].track_start + startpos * 75, endtrack >= disc.disc_totaltracks ? (disc.disc_length.minutes * 60 + disc.disc_length.seconds) * 75 : disc.track[endtrack - 1].track_start - 1);
}

/* Play starttrack to endtrack */
int
cd_play_track(int cd_desc, int starttrack, int endtrack)
{
   return cd_play_track_pos(cd_desc, starttrack, endtrack, 0);
}

/* Play starttrack at position pos to end of CD */
int
cd_play_pos(int cd_desc, int track, int startpos)
{
   struct disc_info disc;
   
   cd_stat(cd_desc, &disc);
   return cd_play_track_pos(cd_desc, track, disc.disc_totaltracks, startpos);
}

/* Play starttrack to end of CD */
int
cd_play(int cd_desc, int track)
{
   struct disc_info disc;
   
   cd_stat(cd_desc, &disc);
   return cd_play_track(cd_desc, track, disc.disc_totaltracks);
}

int
cd_track_advance(int cd_desc, struct disc_timeval *time) {
   struct disc_info disc;
   
   cd_stat(cd_desc, &disc);
   
   disc.track_time.minutes += time->minutes;
   disc.track_time.seconds += time->seconds;

   if(disc.track_time.seconds > 60) {
      disc.track_time.seconds -= 60;
      disc.track_time.minutes++;
   }
   if(disc.track_time.seconds < 0) {
      disc.track_time.seconds = 60 - disc.track_time.seconds;
      disc.track_time.minutes--;
   }
   
   if(disc.track_time.minutes < 0) {
      disc.disc_track--;
      if(disc.disc_track == 0)
	disc.disc_track = 1;
      
      return cd_play(cd_desc, disc.disc_track);
   }
   
   if((disc.track_time.minutes == disc.track[disc.disc_track].track_pos.minutes && disc.track_time.seconds > disc.track[disc.disc_track].track_pos.seconds) || disc.track_time.minutes > disc.track[disc.disc_track].track_pos.minutes) {
      disc.disc_track++;
      if(disc.disc_track > disc.disc_totaltracks)
	disc.disc_track = disc.disc_totaltracks;
      
      return cd_play(cd_desc, disc.disc_track);
   }
   
   return cd_play_pos(cd_desc, disc.disc_track, disc.track_time.minutes * 60 + disc.track_time.seconds);
}

/* Stop the CD, if it is playing */
int
cd_stop(int cd_desc)
{
   if(ioctl(cd_desc, CDROMSTOP) < 0)
     return -1;
   
   return 0;
}

/* Pause the CD */
int
cd_pause(int cd_desc)
{
   if(ioctl(cd_desc, CDROMPAUSE) < 0)
     return -1;
   
   return 0;
}

/* Resume playing */
int
cd_resume(int cd_desc)
{
   if(ioctl(cd_desc, CDROMRESUME) < 0)
     return -1;
   
   return 0;
}

/* Eject the tray */
int
cd_eject(int cd_desc)
{   
   if(ioctl(cd_desc, CDROMEJECT) < 0)
     return -1;
   
   return 0;
}

/* Close the tray */
int
cd_close(int cd_desc)
{
   if(ioctl(cd_desc, CDROMCLOSETRAY) < 0)
     return -1;
   
   return 0;
}

int
cd_get_volume(int cd_desc, struct disc_volume *vol)
{
   struct cdrom_volctrl volume;
   
   if(ioctl(cd_desc, CDROMVOLREAD, &volume) < 0)
     return -1;
      
   vol->vol_front.left = volume.channel0;
   vol->vol_front.right = volume.channel1;
   vol->vol_back.left = volume.channel2;
   vol->vol_back.right = volume.channel3;
}

int
cd_set_volume(int cd_desc, struct disc_volume *vol)
{
   struct cdrom_volctrl volume;
   
   if(vol->vol_front.left > 255 || vol->vol_front.left < 0 || vol->vol_front.right > 255 || vol->vol_front.right < 0 || vol->vol_back.left > 255 || vol->vol_back.left < 0 || vol->vol_back.right > 255 || vol->vol_back.right < 0)
     return -1;
      
   volume.channel0 = vol->vol_front.left;
   volume.channel1 = vol->vol_front.right;
   volume.channel2 = vol->vol_back.left;
   volume.channel3 = vol->vol_back.right;
   
   if(ioctl(cd_desc, CDROMVOLCTRL, &volume) < 0)
     return -1;
   
   return 0;
}
