/*
This is part of the audio CD player library
Copyright (C)1998 Tony Arcieri

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.
*/

#include <cdaudio.h>
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#include <pwd.h>
#include <errno.h>
#include <unistd.h>
#include <cddb.h>

/* CDDB entry node */
struct cddb_entry {
   unsigned long entry_id;
   int entry_genre;
};

/* Static function prototypes */
static int cddb_sum(int val);
static int cddb_conf_data_mod(char *var, char *value, struct cddb_conf *conf);
static int cddb_conf_process_line(char *line, struct cddb_conf *conf);
static int cddb_connect_server(char *hostname, int port);
static int cddb_read_line(int sock, char *inbuffer, int len);
static int cddb_get_server(struct cddb_host *server, int server_index);
static int cddb_get_server_force(struct cddb_host *server, int server_index);
static int cddb_hello(int sock);
static int cddb_query(int cd_desc, int sock, struct cddb_entry *entry);
static int cddb_read(int sock, struct cddb_entry *entry, struct disc_data *data);
static int cddb_sites_process_line(char *line, struct cddb_hostelement *host);
static int cddb_sites(int sock, struct cddb_serverlist *list);
static int cddb_quit(int sock);
static int cddb_read_token(int sock);

/* CDDB sum function... got me, they invented it */
static int
cddb_sum(int val)
{
   char *bufptr, buf[16];
   int ret = 0;
   
   snprintf(buf, 16, "%lu", val);
   for(bufptr = buf; *bufptr != '\0'; bufptr++)
     ret += (*bufptr - '0');
   
   return ret;
}

/* Produce CDDB ID for CD currently in CD-ROM */
unsigned long 
cddb_discid(int cd_desc)
{
   int index, tracksum = 0, discid;
   struct disc_info disc;
   
   cd_stat(cd_desc, &disc);
   
   for(index = 0; index < disc.disc_totaltracks; index++)
     tracksum += cddb_sum(disc.track[index].track_pos.minutes * 60 + disc.track[index].track_pos.seconds);
   
   discid = (disc.disc_length.minutes * 60 + disc.disc_length.seconds) - (disc.track[0].track_pos.minutes * 60 + disc.track[0].track_pos.seconds);
      
   return (tracksum % 0xFF) << 24 | discid << 8 | disc.disc_totaltracks;
}

/* Modify a cddb_conf structure based upon a CDDB configuration command */
static int
cddb_conf_data_mod(char *var, char *value, struct cddb_conf *conf)
{   
   if(value == NULL)
     value = "";
   
   if(strcasecmp(var, "cddb") == 0) {
      if(strncasecmp(value, "on", 2) == 0)
	conf->enable = 1;
      else
	conf->enable = 0;
   } else if(strcasecmp(var, "server") == 0) {
      if(strchr(value, ':') == NULL) {
	 strncpy(conf->host[conf->hostlen].hostname, value, 256);
	 conf->host[conf->hostlen++].port = CDDB_DEFAULT_PORT;
      } else {
	 strncpy(conf->host[conf->hostlen].hostname, strtok(value, ":"), 256);
	 conf->host[conf->hostlen++].port = strtol(strtok(NULL, ":"), NULL, 10);
      }
   }
   
   return 0;
}

/* Process a line in a CDDB configuration file */
static int
cddb_conf_process_line(char *line, struct cddb_conf *conf)
{
   char *var, *value;
   
   if(strchr(line, ' ') == NULL)
     return 0;
   
   line[strlen(line) - 1] = '\0';

   var = strtok(line, " ");
   if(var == NULL)
     return 0;
   value = strtok(NULL, " ");
   cddb_conf_data_mod(var, value, conf);
   
   return 0;
}

/* Read ~/.cddbrc and /etc/cddb.conf */
static int
cddb_read_conf(struct cddb_conf *conf)
{
   FILE *cddbconf;
   int index, rconf;
   char inbuffer[256];
   char localconfpath[256];
   struct stat st;
   
   conf->hostlen = 0;
   conf->enable = 1;         /* CDDB support defaults to on */
   rconf = 0;
   
   /* The user's server list takes priority over the global server list */
   snprintf(localconfpath, 256, "%s/.cddbrc", getenv("HOME"));
   if(stat(localconfpath, &st) == 0) {
      cddbconf = fopen(localconfpath, "r");
   
      while(!feof(cddbconf)) {
         fgets(inbuffer, 256, cddbconf);
         inbuffer[255] = '\0';
      
         /* Remove comments */
      
         for(index = 0; index < strlen(inbuffer); index++)
      	   if(inbuffer[index] == '#') {
	      inbuffer[index] = '\0';
	      break;
	   }
      
         cddb_conf_process_line(inbuffer, conf);
      }
   
      fclose(cddbconf);
      rconf = 1;
   }
   
   if(stat(CDDB_CONF_PATH, &st) == 0) {
      cddbconf = fopen(CDDB_CONF_PATH, "r");
   
      while(!feof(cddbconf)) {
         fgets(inbuffer, 256, cddbconf);
         inbuffer[255] = '\0';
      
         /* Remove comments */
      
         for(index = 0; index < strlen(inbuffer); index++)
	   if(inbuffer[index] == '#') {
	      inbuffer[index] = '\0';
	      break;
	   }
	 
	 /* User configuration takes priority over global */
	 if(strncasecmp(inbuffer, "cddb", 4) != 0 || rconf != 1)
	   cddb_conf_process_line(inbuffer, conf);
      }
   
      fclose(cddbconf);
      rconf = 1;
   }
   
   if(rconf == 0)
     return -1;
   
   return 0;
}

/* Convert numerical genre to text */
char
*cddb_genre(int genre)
{
   switch(genre) {
    case BLUES:
      return "blues";
    case CLASSICAL:
      return "classical";
    case COUNTRY:
      return "country";
    case DATA:
      return "data";
    case FOLK:
      return "folk";
    case JAZZ:
      return "jazz";
    case MISC:
      return "misc";
    case NEWAGE:
      return "newage";
    case REGGAE:
      return "reggae";
    case ROCK:
      return "rock";
    case SOUNDTRACK:
      return "soundtrack";
   }
   
   return "unknown";
}

/* Connect to a CDDB server */
static int
cddb_connect_server(char *hostname, int port)
{
   int sock;
   struct sockaddr_in sin;
   struct hostent *host;
   
   sin.sin_family = AF_INET;
   sin.sin_port = htons(port);
   
   if((sin.sin_addr.s_addr = inet_addr(hostname)) == INADDR_NONE) {
      if((host = gethostbyname(hostname)) == NULL)
	return -1;
      
      bcopy(host->h_addr, &sin.sin_addr, host->h_length);
   }
    
   if((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
     return -1;
   
   if(connect(sock, (struct sockaddr *) &sin, sizeof(sin)) < 0)
     return -1;

   if(cddb_read_token(sock) != 201)
     return -1;

   return sock;
}

/* Read a single line */
static int
cddb_read_line(int sock, char *inbuffer, int len)
{
   int index;
   char inchar;
   
   for(index = 0; index < len; index++) {
      read(sock, &inchar, 1);
      if(inchar == '\n') {
	 inbuffer[index] = '\0';
	 if(inbuffer[0] == '.')
	   return 1;
	 return 0;
      }
      inbuffer[index] = inchar;
   }
   
   return index;
}

/* Pull entry from CDDB configuration files */
static int
cddb_get_server(struct cddb_host *server, int server_index)
{
   struct cddb_conf conf;
   
   /* If we can't find any server names, use the defaults */
   if(cddb_read_conf(&conf) < 0 || conf.hostlen == 0) {
      strncpy(server->hostname, CDDB_DEFAULT_HOSTNAME, 256);
      server->port = CDDB_DEFAULT_PORT;
      
      return 0;
   }
   
   if(conf.enable == 0)
     return -1;
   
   if(server_index >= conf.hostlen)
     return -1;
   
   strncpy(server->hostname, conf.host[server_index].hostname, 256);
   server->port = conf.host[server_index++].port;
   
   return 0;
}

/* Get an entry from the server list whether CDDB is on or off */
static int
cddb_get_server_force(struct cddb_host *server, int server_index)
{
   struct cddb_conf conf;
   
   /* If we can't find any server names, use the defaults */
   if(cddb_read_conf(&conf) < 0 || conf.hostlen == 0) {
      strncpy(server->hostname, CDDB_DEFAULT_HOSTNAME, 256);
      server->port = CDDB_DEFAULT_PORT;
      
      return 0;
   }
   
   if(server_index >= conf.hostlen)
     return -1;
   
   strncpy(server->hostname, conf.host[server_index].hostname, 256);
   server->port = conf.host[server_index++].port;
   
   return 0;
}

/* Initiate CDDB hello verifying username, hostname, client (which is
   always libcdaudio) and the version.  This gives us access to the
   CDDB. */
static int
cddb_hello(int sock)
{
   struct passwd *pw;
   char hostname[256];
   char outbuffer[256];
   
   pw = getpwuid(getuid());
   gethostname(hostname, 256);
   
   snprintf(outbuffer, 256, "cddb hello %s %s %s %s\n", pw->pw_name, hostname, PACKAGE, VERSION);
   write(sock, outbuffer, strlen(outbuffer));
      
   if(cddb_read_token(sock) != 200)
     return -1;

   return 0;
}

/* Query the CDDB for the CD currently in the CD-ROM, and find the ID of the
   CD (which may or may not be the one generated) and what section it is
   under (genre) */
static int
cddb_query(int cd_desc, int sock, struct cddb_entry *entry)
{
   int index;
   struct disc_info disc;
   char offsetbuffer[256], outbuffer[256], inbuffer[512], genre[32];
   
   cd_stat(cd_desc, &disc);
   
   snprintf(offsetbuffer, 256, "%d", disc.disc_totaltracks);
   for(index = 0; index < disc.disc_totaltracks; index++)
     snprintf(offsetbuffer, 256, "%s %d", offsetbuffer, disc.track[index].track_start);
   snprintf(outbuffer, 256, "cddb query %08lx %s %d\n", cddb_discid(cd_desc), offsetbuffer, disc.disc_length.minutes * 60 + disc.disc_length.seconds);
   write(sock, outbuffer, strlen(outbuffer));
   
   cddb_read_line(sock, inbuffer, 512);
   switch(strtol(strtok(inbuffer, " "), NULL, 10)) {
    /* 200 - match was made */
    case 200:
      strncpy(genre, strtok(NULL, " "), 32);
      sscanf(strtok(NULL, " "), "%xl", &entry->entry_id);

      break;
    /* 211 - imperfect match, choose the top entry on the list */
    case 211:
      cddb_read_line(sock, inbuffer, 512);
      strncpy(genre, strtok(inbuffer, " "), 32);
      sscanf(strtok(NULL, " "), "%xl", &entry->entry_id);
      
      while(!cddb_read_line(sock, inbuffer, 512));
      
      break;
    default:
      return -1;
   }
   
   if(strcmp(genre, "blues") == 0)
     entry->entry_genre = BLUES;
   else if(strcmp(genre, "classical") == 0)
     entry->entry_genre = CLASSICAL;
   else if(strcmp(genre, "country") == 0)
     entry->entry_genre = COUNTRY;
   else if(strcmp(genre, "data") == 0)
     entry->entry_genre = DATA;
    else if(strcmp(genre, "folk") == 0)
     entry->entry_genre = FOLK;
    else if(strcmp(genre, "jazz") == 0)
     entry->entry_genre = JAZZ;
    else if(strcmp(genre, "misc") == 0)
     entry->entry_genre = MISC;
    else if(strcmp(genre, "newage") == 0)
     entry->entry_genre = NEWAGE;
    else if(strcmp(genre, "reggae") == 0)
     entry->entry_genre = REGGAE;
    else if(strcmp(genre, "rock") == 0)
     entry->entry_genre = ROCK;
    else if(strcmp(genre, "soundtrack") == 0)
     entry->entry_genre = SOUNDTRACK;
    else
     entry->entry_genre = UNKNOWN;

   return 0;
}

int
cddb_data_mod(char *var, char *value, struct disc_data *data)
{
   if(value == NULL)
     value = "";      
   
   if(strcmp(var, "DTITLE") == 0) {
      strncpy(data->data_artist, strtok(value, "/"), 48);
      strncpy(data->data_title, (char *)strtok(NULL, "/") + 1, 64);
   } else if(strncmp(var, "TTITLE", 6) == 0) {
      strncpy(data->data_track[strtol((char *)var + 6, NULL, 10)].track_name, value, 56);
   } else if(strcmp(var, "EXTD") == 0) {
      strncpy(data->data_extended[data->data_extendedindex++], value, 80);
   } else if(strncmp(var, "EXTT", 4) == 0) {
      strncpy(data->data_track[strtol((char *)var + 4, NULL, 10)].track_extended[data->data_track[strtol((char *)var + 4, NULL, 10)].track_extendedindex++], value, 80);
   }
   
   return 0;
}

int
cddb_process_line(char *line, struct disc_data *data)
{
   char *var, *value;
   
   if(strchr(line, '=') == NULL)
     return 0;
   
   line[strlen(line) - 1] = '\0';

   var = strtok(line, "=");
   if(var == NULL)
     return 0;
   value = strtok(NULL, "=");
   cddb_data_mod(var, value, data);
   
   return 0;
}

/* Read the actual CDDB entry */
static int
cddb_read(int sock, struct cddb_entry *entry, struct disc_data *data)
{
   char outbuffer[256], inbuffer[512];
   
   snprintf(outbuffer, 256, "cddb read %s %08lx\n", cddb_genre(entry->entry_genre), entry->entry_id);
   write(sock, outbuffer, strlen(outbuffer));
   
   if(cddb_read_token(sock) != 210)
     return -1;
   
   while(!cddb_read_line(sock, inbuffer, 512))
     cddb_process_line(inbuffer, data);

   return 0;
}

/* Process a single line in the sites list */
static int
cddb_sites_process_line(char *line, struct cddb_hostelement *host)
{
   char *ptr;
   
   strncpy(host->host_hostname, strtok(line, " "), 256);
   host->host_port = strtol(strtok(NULL, " "), NULL, 10);
   
   ptr = strtok(NULL, " ");
   host->host_icbm_latitude = strtol((char *)ptr + 1, NULL, 10);
   if(ptr[0] == 'N')
     host->host_icbm_latitude += 90;
   else
     host->host_icbm_latitude = 90 - host->host_icbm_latitude;

   ptr = strtok(NULL, " ");
   host->host_icbm_longitude = strtol((char *)ptr + 1, NULL, 10);
   if(ptr[0] == 'W')
     host->host_icbm_longitude += 180;
   else
     host->host_icbm_longitude = 180 - host->host_icbm_longitude;

   strcpy(host->host_city, "");

   while((ptr = strtok(NULL, " ")) != NULL)
     snprintf(host->host_city, 256, "%s %s", host->host_city, ptr);

   strcpy(host->host_city, (char *)host->host_city + 1);

   return 0;
}

/* Read the CDDB sites list */
static int
cddb_sites(int sock, struct cddb_serverlist *list)
{
   char outbuffer[8];
   char inbuffer[256];
   
   strcpy(outbuffer, "sites\n");
   write(sock, outbuffer, strlen(outbuffer));
   
   if(cddb_read_token(sock) != 210)
     return -1;
   
   list->list_len = 0;
   
   while(!cddb_read_line(sock, inbuffer, 256))
     cddb_sites_process_line(inbuffer, &list->list_host[list->list_len++]);
   
   return 0;
}

/* Bye! */
static int
cddb_quit(int sock)
{
   char outbuffer[8];
   
   strcpy(outbuffer, "quit\n");
   write(sock, outbuffer, strlen(outbuffer));
   
   close(sock);
   
   return 0;
}

/* Return the numerical value of a reply, allowing us to quickly check if
   anything went wrong */
static int
cddb_read_token(int sock)
{
   char inbuffer[512];
   
   cddb_read_line(sock, inbuffer, 512);
   
   return strtol(strtok(inbuffer, " "), NULL, 10);
}

/* This is the function for completely automated CDDB operation */
int
cddb_read_data(int cd_desc, struct disc_data *data)
{
   int sock, index, server_index = 0;
   struct disc_info disc;
   struct cddb_host server;
   struct cddb_entry entry;
   
   cd_stat(cd_desc, &disc);
   if(disc.disc_status == CD_ABSENT)
     return -1;
   
   data->data_id = cddb_discid(cd_desc);
   data->data_extendedindex = 0;
   for(index = 0; index < disc.disc_totaltracks; index++)
     data->data_track[index].track_extendedindex = 0;
   
   do {
      if(cddb_get_server(&server, server_index++) < 0)
	return -1;
   } while((sock = cddb_connect_server(server.hostname, server.port)) < 0);
      
   if(cddb_hello(sock) < 0)
     return -1;   
   
   if(cddb_query(cd_desc, sock, &entry) < 0)
     return -1;
   
   data->data_genre = entry.entry_genre;
   
   if(cddb_read(sock, &entry, data) < 0)
     return -1;
   
   cddb_quit(sock);
   
   return 0;
}

/* Retrieve the server list */
int
cddb_read_sites(struct cddb_serverlist *list)
{
   int sock, server_index = 0;
   struct cddb_host server;
   
   do {
      if(cddb_get_server_force(&server, server_index++) < 0)
	return -1;
   } while((sock = cddb_connect_server(server.hostname, server.port)) < 0);
   
   if(cddb_sites(sock, list) < 0)
     return -1;
   
   cddb_quit(sock);
}

/* Generate an entry using CDDB if it's available, or Unknowns if it's not */
int
cddb_generate_new_entry(int cd_desc, struct disc_data *data)
{
   int index;
   struct disc_info disc;
   
   if(cddb_read_data(cd_desc, data) < 0) {
      cd_stat(cd_desc, &disc);
      data->data_id = cddb_discid(cd_desc);
      strcpy(data->data_title, "Unknown");
      strcpy(data->data_artist, "Unknown");
      data->data_genre = MISC;
      for(index = 0; index < disc.disc_totaltracks; index++) {
	 strcpy(data->data_track[index].track_name, "Unknown");
      }
   }
   
   cddb_write_disc_data(cd_desc, data);
}

/* Read from the local database, using CDDB if there isn't an entry cached */
int
cddb_read_disc_data(int cd_desc, struct disc_data *data)
{
   FILE *cddb_data;
   int index;
   char root_dir[256], file[256], inbuffer[512];
   struct disc_info disc;
   struct stat st;
   
   snprintf(root_dir, 256, "%s/.cddb", getenv("HOME"));
   
   if(stat(root_dir, &st) < 0) {
      if(errno != ENOENT)
	return -1;
      else {
	 cddb_generate_new_entry(cd_desc, data);
	 cddb_write_disc_data(cd_desc, data);
	 return 0;
      }
   } else {
      if(!S_ISDIR(st.st_mode)) {
	 errno = ENOTDIR;
	 return -1;
      }
   }
   
   cd_stat(cd_desc, &disc);
      
   data->data_id = cddb_discid(cd_desc);
   data->data_extendedindex = 0;
   for(index = 0; index < disc.disc_totaltracks; index++)
     data->data_track[index].track_extendedindex = 0;
   
   for(index = 0; index < 12; index++) {
      snprintf(file, 256, "%s/%s/%08lx", root_dir, cddb_genre(index), cddb_discid(cd_desc));
      if(stat(file, &st) == 0) {
	 cddb_data = fopen(file, "r");
	 while(!feof(cddb_data)) {
	    fgets(inbuffer, 512, cddb_data);
	    cddb_process_line(inbuffer, data);
	 }
	 
	 data->data_id = cddb_discid(cd_desc);
	 data->data_genre = index;
	 fclose(cddb_data);
	 return 0;
      }
   }
   
   cddb_generate_new_entry(cd_desc, data);
   
   return 0;
}

/* Write to the local cache */
int
cddb_write_disc_data(int cd_desc, struct disc_data *data)
{
   FILE *cddb_data;
   int index, tracks;
   char root_dir[256], genre_dir[256], file[256];
   struct stat st;
   struct disc_info disc;
   
   cd_stat(cd_desc, &disc);
   
   snprintf(root_dir, 256, "%s/.cddb", getenv("HOME"));
   snprintf(genre_dir, 256, "%s/%s", root_dir, cddb_genre(data->data_genre));
   snprintf(file, 256, "%s/%08lx", genre_dir, data->data_id);
   
   if(stat(root_dir, &st) < 0) {
      if(errno != ENOENT)
	return -1;
      else
	mkdir(root_dir, 0755);
   } else {
      if(!S_ISDIR(st.st_mode)) {
	 errno = ENOTDIR;
	 return -1;
      }   
   }
   
   if(stat(genre_dir, &st) < 0) {
      if(errno != ENOENT)
	return -1;
      else
	mkdir(genre_dir, 0755);
   } else {
      if(!S_ISDIR(st.st_mode)) {
	 errno = ENOTDIR;
	 return -1;
      }
   }
   
   if((cddb_data = fopen(file, "w")) == NULL)
     return -1;
   
   fprintf(cddb_data, "# xmcd CD database file generated by %s %s\n", PACKAGE, VERSION);
   fputs("# \n", cddb_data);
   fputs("# Track frame offsets:\n", cddb_data);
   for(index = 0; index < disc.disc_totaltracks; index++)
     fprintf(cddb_data, "#       %d\n", disc.track[index].track_start);
   fputs("# \n", cddb_data);
   fprintf(cddb_data, "# Disc length: %d seconds\n", disc.disc_length.minutes * 60 + disc.disc_length.seconds);
   fputs("# \n", cddb_data);
   fprintf(cddb_data, "# Revision: %d\n", CURRENT_CDDBREVISION);
   fprintf(cddb_data, "# Submitted via: %s %s\n", PACKAGE, VERSION);
   fputs("# \n", cddb_data);
   fprintf(cddb_data, "DISCID=%08lx\n", data->data_id);
   if(data->data_artist[strlen(data->data_artist) - 1] == ' ')
     fprintf(cddb_data, "DTITLE=%s/ %s\n", data->data_artist, data->data_title);
   else
     fprintf(cddb_data, "DTITLE=%s / %s\n", data->data_artist, data->data_title);
   for(index = 0; index < disc.disc_totaltracks; index++)
     fprintf(cddb_data, "TTITLE%d=%s\n", index, data->data_track[index].track_name);
   if(data->data_extendedindex == 0)
     fputs("EXTD=\n", cddb_data);
   else {
      for(index = 0; index < data->data_extendedindex; index++)
	fprintf(cddb_data, "EXTD=%s\n", data->data_extended[index]);
   }
   
   for(tracks = 0; tracks < disc.disc_totaltracks; tracks++) {
      if(data->data_track[tracks].track_extendedindex == 0)
	fprintf(cddb_data, "EXTT%d=\n", tracks);
      else {
	 for(index = 0; index < data->data_track[tracks].track_extendedindex; index++)
	   fprintf(cddb_data, "EXTT%d=%s\n", tracks, data->data_track[tracks].track_extended[index]);
      }
   }	 
	 
   fputs("PLAYORDER=", cddb_data);
   
   fclose(cddb_data);
   
   return 0;
}

/* Delete an entry from the local cache based upon a data structure */
int
cddb_erase_entry(struct disc_data *data) {
   char root_dir[256], genre_dir[256], file[256];
   struct stat st;
   
   snprintf(root_dir, 256, "%s/.cddb", getenv("HOME"));
   snprintf(genre_dir, 256, "%s/%s", root_dir, cddb_genre(data->data_genre));
   snprintf(file, 256, "%s/%08lx", genre_dir, data->data_id);
   
   if(stat(root_dir, &st) < 0) {
      if(errno != ENOENT)
	return -1;
      else
	return 0;
   } else {
      if(!S_ISDIR(st.st_mode))
	return 0;  
   }
   
   if(stat(genre_dir, &st) < 0) {
      if(errno != ENOENT)
	return -1;
      else
	return 0;
   } else {
      if(!S_ISDIR(st.st_mode))
	return 0;
   }
   
   if(unlink(file) < 0) {
      if(errno != ENOENT)
	return -1;
   }
   
   return 0;
}

/* Refresh a CDDB entry */
int
cddb_refresh_disc_data(int cd_desc, struct disc_data *data)
{
   cddb_read_disc_data(cd_desc, data);
   cddb_erase_entry(data);
   cddb_generate_new_entry(cd_desc, data);
}
