/*
 * CrazySumForPasswords v.3b
 *
 * AUTHOR:
 *  Robert Fahy <rfahy@ymail.com>
 *
 * DATE:
 *  December 2010
 *
 * ABOUT:
 *  - the old versions were poorly written and commented, and the executables were insecure
 *  - this version is cleaner, and slightly more professional (more secure and more maintainable)
 *  - this version produces output similar to old CSFP-P, but they are not compatible!
 *  - my style of writing has also changed a lot since August '09 (mostly for the better, I think)
 *
 * TODO:
 *  - maybe incorporate SHA512 algorithm into own program... or maybe not
 *  - maybe use something other (better) than pipes (like sockets?)
 *
 */

// include files
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/wait.h>

// preprocessor definitions with obvious purposes
#define CSFP_NAME_VERSION				"CSFP v.3b Beta 2"
#define CSFP_GENERIC_PROGRAM_FILENAME	"<csfp3.elf>"
#define CSFP_GENERIC_MESSAGE			"<Error text is unavailable.>"
#define CSFP_DEFAULT_PASS_SIZE			128
#define CSFP_MAX_PASS_SIZE				4096

// ABOUT: un-comment to build the binary password version of CSFP instead of the text password version
//#define CSFP_BINARY_VERSION

// ABOUT: un-comment to allow printing a newline character after password
//#define CSFP_PRINT_NEWLINE

// ABOUT: un-comment to allow clearing of environment variables
#define CSFP_CLEAR_ENVIRON

// ABOUT: maximum length to accept in csfp_strlen() function
#define CSFP_STRLEN_MAXLENGTH			CSFP_MAX_PASS_SIZE

// declarations without definitions (aka prototypes)
char * csfp_strcpy(char *dest, const char *src);
size_t csfp_strlen(const char *str);
size_t csfp_hasletters(const char *str);
void print_usage(const char *program_filename, const char *message);
int compute(const char *argument1, const char *argument2);
void encrypt(char *result, const char *original, const unsigned int size);
int main(int argc, char **argv); // for completeness of this "index"

char * csfp_strcpy(char *dest, const char *src)
// ABOUT: a stupider version of strcpy
// WARN: requires length of `dest' to be one unit larger than length of `src'!
{
	size_t i=0;

	// NOTE: variable added for performance
	const size_t src_length_ = csfp_strlen(src);

	if (src != NULL &&
		*src != '\0' &&
		dest != NULL)
	{
		while (i < src_length_)
		{
			dest[i] = src[i];
			++i;
		}

		dest[src_length_] = '\0';

		return dest;
	}
	else
		return NULL;
} // ### int csfp_strcpy(char *dest, const char *src)

size_t csfp_strlen(const char *str)
// ABOUT: a slightly more secure version of strlen(), which stops counting bytes after a predefined limit
{
#ifndef CSFP_STRLEN_MAXLENGTH
#define CSFP_STRLEN_MAXLENGTH			4096
#endif

	size_t i=0;

	if (str != NULL)	// WARN: safer but possibly redundant!
		while (str[i] != '\0')
			if (i < CSFP_STRLEN_MAXLENGTH)
				++i;
			else
				break;

	return i;
} // ### size_t csfp_strlen(const char *str)

size_t csfp_hasletters(const char *str)
// ABOUT: returns non-zero value if finds a non-digit character in `str' (that includes symbols)
{
	size_t
		i=0,
		str_length_ = csfp_strlen(str);

	if (str != NULL)	// WARN: safer but possibly redundant!
		while (i < str_length_)
			if (!isdigit(str[i++]))
				return i;

	return 0;
} // ### size_t csfp_hasletters(const char *str)

void print_usage(const char *program_filename, const char *message)
// ABOUT: prints usage help plus additional `message'
{
	char
		*program_filename_ = NULL,
		*message_ = NULL;

	// check against bogus `program_filename'
	if (program_filename != NULL &&
		*program_filename != '\0')
	{
		if ((program_filename_ = (char *)malloc(csfp_strlen(program_filename) + 1))
			!= NULL)
			csfp_strcpy(program_filename_, program_filename);
		else
			return;
	}
	else	// bad `program_filename'
		if ((program_filename_ = (char *)malloc(csfp_strlen(CSFP_GENERIC_PROGRAM_FILENAME) + 1))
			!= NULL)
			csfp_strcpy(program_filename_, CSFP_GENERIC_PROGRAM_FILENAME);
		else
			return;

	// check against bogus `message'
	if (message != NULL &&
		*message != '\0')
	{
		if ((message_ = (char *)malloc(csfp_strlen(message) + 1))
			!= NULL)
			csfp_strcpy(message_, message);
		else
			return;
	}
	else	// bad `message'
		if ((message_ = (char *)malloc(csfp_strlen(CSFP_GENERIC_MESSAGE) + 1))
			!= NULL)
			csfp_strcpy(message_, CSFP_GENERIC_MESSAGE);
		else
			return;

	printf("\n%s usage:\n %s <[file_to_be_checksumed] [desired_length_of_result]>\n"
		"Also accepting standard input (stdin) and providing length is not compulsory\n"
		" (default is \"%d\", maximum is \"%d\").\n\nERROR: %s\n\n\n",
		CSFP_NAME_VERSION, program_filename_, CSFP_DEFAULT_PASS_SIZE, CSFP_MAX_PASS_SIZE, message_);

	free(program_filename_);
	program_filename_ = NULL;
	free(message_);
	message_ = NULL;
} // ### void print_usage(const char *program_filename, const char *message)

int compute(const char *argument1, const char *argument2)
// ABOUT: verifies `argument1' and `argument2' and computes the password data accordingly
{
	int
		chksum2prog_[2],	// needed for "pipe" data transfer from checksum utility to our program
		exit_status_;		// needed for returning EXIT_FAILURE or EXIT_SUCCESS status

	pid_t pid_;				// needed to determine which process we're in after forking

	unsigned int desired_length_ = CSFP_DEFAULT_PASS_SIZE;	// desired length of the computed password

	char
		sha512_checksum_[128],		// will store the first 128 characters of sha512sum's output
		*csfp_checksum_,			// will store the CSFP password-checksum
		*execlp_argument_ = NULL;	// file argument for execlp's call to `sha512sum' (NULL for stdin)

	if (argument1 != NULL)
	{
		// optimistically presume that `argument1' is filename
		if ((execlp_argument_ = (char *)malloc(csfp_strlen(argument1) + 1))
			!= NULL)
			csfp_strcpy(execlp_argument_, argument1);
		else
		{
			perror("compute() -> malloc(execlp_argument_)");
			exit(EXIT_FAILURE);
		}

		if (argument2 == NULL)
		{
			if (csfp_hasletters(argument1))
			// we have letters, consider `argument1' to be a filename
				desired_length_ = CSFP_DEFAULT_PASS_SIZE;
			else // `argument1' is all-digit, consider it to be desired length
			{
				free(execlp_argument_);
				execlp_argument_ = NULL;
				desired_length_ = (unsigned int)atoi(argument1);
			}
		}
		else // consider `argument1' to be filename, `argument2' to be desired length
			desired_length_ = (unsigned int)atoi(argument2);

		// make sure we stay within bounds
		if (desired_length_ > CSFP_MAX_PASS_SIZE)
			desired_length_ = CSFP_MAX_PASS_SIZE;
	}

	if (pipe(chksum2prog_) == -1)
	// initialise pipe and check success
	{
		perror("compute() -> pipe(chksum2prog_)");

		if (execlp_argument_ != NULL)
		{
			free(execlp_argument_);
			execlp_argument_ = NULL;
		}

		exit(EXIT_FAILURE);
	}

	if ((pid_ = fork()) == -1)
	// check for unsuccessful fork
	{
		perror("compute() -> fork()");

		if (execlp_argument_ != NULL)
		{
			free(execlp_argument_);
			execlp_argument_ = NULL;
		}

		exit(EXIT_FAILURE);
	}
	else
	{
		if (!pid_) // child process, calls `/usr/bin/sha512sum' and returns checksum through pipe to parent
		{
			exit_status_ = EXIT_SUCCESS;
#ifdef CSFP_CLEAR_ENVIRON
			clearenv();
#endif
			if (close(chksum2prog_[0]) == -1)
			// close input end of the pipe and check for error
			{
				perror("compute() -> close(chksum2prog_[0])");
				exit_status_ = EXIT_FAILURE;
			}

			if (dup2(chksum2prog_[1], 1) == -1)
			// bind stdout to output end of the pipe and check for error
			{
				perror("compute() -> dup2(chksum2prog_)");
				exit_status_ = EXIT_FAILURE;
			}

			if (execlp("/usr/bin/sha512sum", "/usr/bin/sha512sum", "--", execlp_argument_, NULL) < 0)
			// check for unsuccessful exec
			{
				perror("compute() -> execlp(\"/usr/bin/sha512sum\")");
				exit_status_ = EXIT_FAILURE;
			}
		}
		else // parent process, gets checksum through pipe from child, mangles and outputs it
		{
			exit_status_ = EXIT_SUCCESS;

			if (close(chksum2prog_[1]) == -1)
			// close output end of the pipe and check for error
			{
				perror("compute() -> close(chksum2prog_[1])");
				exit_status_ = EXIT_FAILURE;
			}

			if (read(chksum2prog_[0], sha512_checksum_, sizeof sha512_checksum_) == -1)
			// read data from pipe and check for error
			{
				perror("compute() -> read(chksum2prog_[0])");
				exit_status_ = EXIT_FAILURE;
			}

			if ((csfp_checksum_ = (char *)malloc(desired_length_ + 1))
				!= NULL)
			{
				encrypt(csfp_checksum_, sha512_checksum_, desired_length_);
				fwrite(csfp_checksum_, (size_t)desired_length_, sizeof (char), stdout);
#ifdef CSFP_PRINT_NEWLINE
				putchar('\n');
#endif
				free(csfp_checksum_);
				csfp_checksum_ = NULL;
			}
			else
			{
				perror("compute() -> malloc(csfp_checksum_)");
				exit_status_ = EXIT_FAILURE;
			}

			wait(NULL);	// wait for child process to terminate
		}

		if (execlp_argument_ != NULL)
		{
			free(execlp_argument_);
			execlp_argument_ = NULL;
		}

		return exit_status_;
	}
} // ### int compute(const char *argument1, const char *argument2)

void encrypt(char *result, const char *original, const unsigned int size)
// ABOUT: changes the `result' array by processing `original'
{
	// TODO: change this array to better suit your needs (be careful not to lose your program afterwards)
	//  also, be careful not to break the special `\\' and `\"' characters
	const char
// ABOUT: decide if we should compile using the text pool, or binary pool
#ifndef CSFP_BINARY_VERSION
		symbol_pool_[] = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ%/\\'`\"[]{}()<>*+-=?!_^&$#@~|;:.,";
#else
		symbol_pool_[] =
			"\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			"\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			"\x20\x21\x22\x23\x24\x25\x26\x27\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			"\x30\x31\x32\x33\x34\x35\x36\x37\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			"\x40\x41\x42\x43\x44\x45\x46\x47\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			"\x50\x51\x52\x53\x54\x55\x56\x57\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			"\x60\x61\x62\x63\x64\x65\x66\x67\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			"\x70\x71\x72\x73\x74\x75\x76\x77\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			"\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			"\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			"\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			"\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			"\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			"\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			"\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			"\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff";
#endif

	// NOTE: these variables vastly increase performance
	const size_t
		symbol_pool_length_ = sizeof symbol_pool_ - 1,	// minus one, because ending '\0' is also counted
		original_length_ = csfp_strlen(original);

	unsigned int
		i=0,
		k=0,
		salt_ = 0;	// this is probably not what specialists would call "salt"

	if (original != NULL &&
		*original != '\0' &&
		result != NULL)
	{
		while (i < size)
		{
			salt_ += (unsigned int)original[k++];
			salt_ += size * k;

			while (salt_ >= symbol_pool_length_)
				salt_ -= symbol_pool_length_;

			result[i++] = symbol_pool_[salt_];

			if (k == original_length_)
				k=0;
		}

		result[size] = '\0'; // make sure our array ends
	}
	else
		exit(EXIT_FAILURE);
} // ### void encrypt(char *result, const char *original, const unsigned int size)

// ### MAIN FUNCTION ###
int main(int argc, char **argv)
// ### MAIN FUNCTION ###
{
	if (argc > 3)
	{
		print_usage(argv[0], "Too many arguments.");
		return EXIT_FAILURE;
	}
	else
		if (argc < 1)
		// NOTE: this shouldn't normally be possible except with a
		//  specially crafted exec() call from another program
		{
			print_usage(CSFP_GENERIC_PROGRAM_FILENAME, "Nice try, Joe.");
			return EXIT_FAILURE;
		}
		else
			switch (argc)
			{
				case 1: return compute(NULL, NULL);			// case 1: standard input
				case 2: return compute(argv[1], NULL);		// case 2a and 2b: standard input and desired length, or filename
				case 3: return compute(argv[1], argv[2]);	// case 3: filename and desired length
				default:
					print_usage(argv[0], "Unknown. Email <rfahy@ymail.com> and include detail, please.");
					return EXIT_FAILURE;
			}
} // ### MAIN FUNCTION ###
