/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */

#include "nactv-debug.h"
#include "net.h"
#include "process.h"

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <glib.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>


static const char *protocol_name[NC_PROTOCOLS_NUMBER] =
{
	"tcp",
	"udp",
	"tcp6",
	"udp6"
};

static const char *protocol_file[NC_PROTOCOLS_NUMBER] =
{
	"/proc/net/tcp",
	"/proc/net/udp",
	"/proc/net/tcp6",
	"/proc/net/udp6"
};

static const char *tcp_state_name[NC_TCP_STATES_NUMBER] =
{
    "",
    "ESTABLISHED",
    "SYN_SENT",
    "SYN_RECV",
    "FIN_WAIT1",
    "FIN_WAIT2",
    "TIME_WAIT",
    "CLOSE",
    "CLOSE_WAIT",
    "LAST_ACK",
    "LISTEN",
    "CLOSING",
	"CLOSED"
};


static GHashTable *services_hash = NULL;

void nactv_net_init ()
{
	/*Load the services database*/
	struct servent *sentry;
	char key_port[128];
	
	g_assert(services_hash == NULL);
	services_hash = g_hash_table_new_full(&g_str_hash, &g_str_equal, &g_free, &g_free);
	
	setservent(0);
	
	while ( (sentry = getservent()) != NULL )
	{
		g_snprintf(key_port, sizeof(key_port), "%d/%s", 
				   ntohs(sentry->s_port), sentry->s_proto);
		g_hash_table_insert(services_hash, g_strdup(key_port), g_strdup(sentry->s_name));
	}
	
	endservent();
}

void nactv_net_free ()
{	
	if (services_hash != NULL)
	{
		g_hash_table_destroy(services_hash);
		services_hash = NULL;
	}
}

static const char *service_protocol_name[NC_PROTOCOLS_NUMBER] = {
	"tcp", "udp", "tcp", "udp"
};

static const char *net_service_get(int protocol, int port)
{
	char *service_name = NULL;
	if (services_hash != NULL)
	{
		char key_port[128];
		g_assert(protocol>=0 && protocol<NC_PROTOCOLS_NUMBER);
		
		g_snprintf(key_port, sizeof(key_port), "%d/%s", 
				   port, service_protocol_name[protocol]);
		service_name = (char*)g_hash_table_lookup(services_hash, key_port);
	}
	return service_name;
}


static GHashTable *get_open_sockets_for_processes (Process **processes, unsigned int *nprocesses)
{
	GHashTable *open_sockets_hash;
	unsigned int i;
	
	open_sockets_hash = g_hash_table_new(NULL, NULL);
	
	*nprocesses = get_running_processes(processes);
	for (i=0; i<*nprocesses; i++)
	{
		long *sockets;
		unsigned int nsockets, j;
		Process *process = *processes+i;
		
		nsockets = process_get_socket_inodes(process->pid, &sockets);
		if (nsockets > 0)
		{
			for (j=0; j<nsockets; j++)
				g_hash_table_insert(open_sockets_hash, (gpointer)sockets[j], process);
			
			g_free(sockets);
		}
	}
	return open_sockets_hash;
}


#define IN6_ADDR_IS_ZERO(addr) ((addr).s6_addr32[0]==0 && (addr).s6_addr32[1]==0 && \
	(addr).s6_addr32[2]==0 && (addr).s6_addr32[3]==0)


static void get_connections_from_kernel(int protocol, GArray *connections, GHashTable *open_sockets_hash)
{
	char buffer[8192];
	FILE *f;
	g_assert(protocol>=0 && protocol<NC_PROTOCOLS_NUMBER);
	
	f = fopen(protocol_file[protocol], "r");
	if (f!=NULL)
	{
		fgets(buffer, sizeof(buffer), f); /*skip the first line*/
		while (fgets(buffer, sizeof(buffer), f)!=NULL)
		{
			NetConnection net_line;
			Process *process;
			unsigned long rxq, txq, time_len, retr;
			long inode;
			int num, local_port, rem_port, d, state, uid, timer_run, timeout;
			char rem_addr[136], local_addr[136], more[1032]="";
			
			memset(&net_line, 0, sizeof(net_line));
			
			state = -1;
			num = sscanf(buffer,
				"%d: %64[0-9A-Fa-f]:%X %64[0-9A-Fa-f]:%X %X %lX:%lX %X:%lX %lX %d %d %ld %512s\n",
				&d, local_addr, &local_port, rem_addr, &rem_port, &state,
				&txq, &rxq, &timer_run, &time_len, &retr, &uid, &timeout, &inode, more);
			
			if (strlen(local_addr) > 8) /*IP v6*/
			{
				struct in6_addr localaddr, remaddr;
				
				sscanf(local_addr, "%08X%08X%08X%08X",
					   &localaddr.s6_addr32[0], &localaddr.s6_addr32[1],
					   &localaddr.s6_addr32[2], &localaddr.s6_addr32[3]);
				sscanf(rem_addr, "%08X%08X%08X%08X",
					   &remaddr.s6_addr32[0], &remaddr.s6_addr32[1],
					   &remaddr.s6_addr32[2], &remaddr.s6_addr32[3]);
				
				if (!IN6_ADDR_IS_ZERO(localaddr))
					inet_ntop(AF_INET6, &localaddr, local_addr, sizeof(local_addr));
				else
					strcpy(local_addr, "*");
				if (!IN6_ADDR_IS_ZERO(remaddr))
					inet_ntop(AF_INET6, &remaddr, rem_addr, sizeof(rem_addr));
				else
					strcpy(rem_addr, "*");
			}else /*IP v4*/
			{
				struct in_addr localaddr, remaddr;
				
				sscanf(local_addr, "%X", &(localaddr.s_addr));
				sscanf(rem_addr, "%X", &(remaddr.s_addr));
				if (localaddr.s_addr != 0)
					inet_ntop(AF_INET, &localaddr, local_addr, sizeof(local_addr));
				else
					strcpy(local_addr, "*");
				if (remaddr.s_addr != 0)
					inet_ntop(AF_INET, &remaddr, rem_addr, sizeof(rem_addr));
				else
					strcpy(rem_addr, "*");
			}
			
			net_line.inode = inode;
			net_line.protocol = protocol;
			net_line.localaddress = g_strdup(local_addr);
			net_line.remoteaddress = g_strdup(rem_addr);
			net_line.localport = local_port;
			net_line.remoteport = rem_port;
			net_line.state = state;
			
			if (inode > 0)
			{
				process = (Process*)g_hash_table_lookup(open_sockets_hash, (gpointer)inode);
				if (process != NULL)
				{
					net_line.pid = process->pid;
					net_line.programpid = process->pid;
					net_line.programname = (process->name!=NULL) ? g_strdup(process->name) : NULL;
					net_line.programcommand = (process->commandline!=NULL) ? 
						g_strdup(process->commandline) : NULL;
				}
			}
			
			g_array_append_val(connections, net_line);
		}
		fclose(f);	
	}
}


NetConnection *net_connection_new()
{
	NetConnection *line = (NetConnection*)g_malloc0(sizeof(NetConnection));
	return line;
}

void net_connection_delete_contents(NetConnection *line)
{
	if (line != NULL)
	{
		if (line->localhost != NULL)
			g_free(line->localhost);
		if (line->localaddress != NULL)
			g_free(line->localaddress);
		if (line->remotehost != NULL)
			g_free(line->remotehost);
		if (line->remoteaddress != NULL)
			g_free(line->remoteaddress);
		if (line->programname != NULL)
			g_free(line->programname);
		if (line->programcommand != NULL)
			g_free(line->programcommand);
	}
}


void net_connection_delete(NetConnection *line)
{
	if (line != NULL)
	{
		net_connection_delete_contents(line);	
		g_free(line);
	}
}

void net_connection_copy(NetConnection *destination, NetConnection *source)
{
	destination->protocol = source->protocol;
	destination->localhost = (source->localhost!=NULL) ? g_strdup(source->localhost) : NULL;
	destination->localaddress = (source->localaddress!=NULL) ? g_strdup(source->localaddress) : NULL;
	destination->localport = source->localport;
	destination->remotehost = (source->remotehost!=NULL) ? g_strdup(source->remotehost) : NULL;
	destination->remoteaddress = (source->remoteaddress!=NULL) ? g_strdup(source->remoteaddress) : NULL;
	destination->remoteport = source->remoteport;
	destination->state = source->state;
	destination->pid = source->pid;
	destination->programpid = source->programpid;
	destination->programname = (source->programname!=NULL) ? g_strdup(source->programname) : NULL;
	destination->programcommand = (source->programcommand!=NULL) ? g_strdup(source->programcommand) : NULL;
	destination->inode = source->inode;
	destination->operation = source->operation;
	destination->user_data = source->user_data;
}

const char *net_connection_get_protocol_name (NetConnection *conn)
{
	g_assert(conn->protocol >=0 && conn->protocol < NC_PROTOCOLS_NUMBER);
	return protocol_name[conn->protocol];
}

const char *net_connection_get_state_name (NetConnection *conn)
{
	if (conn->protocol == NC_PROTOCOL_TCP || conn->protocol == NC_PROTOCOL_TCP6)
	{
		if (conn->state>=0 && conn->state<NC_TCP_STATES_NUMBER)
			return tcp_state_name[conn->state];
		else
			return "";
	}else /*UDP*/
	{
		switch (conn->state) 
		{
		case NC_TCP_ESTABLISHED:
		case NC_TCP_CLOSED:
			return tcp_state_name[conn->state];
			break;
		default: /*TCP_CLOSE is a kind of LISTEN, but netstat shows an empty string for that too*/
			return "";
		}		
	}		
}

int net_connection_get_address_family(NetConnection *conn)
{
	return (conn->protocol==NC_PROTOCOL_TCP || conn->protocol==NC_PROTOCOL_UDP) ? AF_INET : AF_INET6;
}

unsigned int get_net_connections(NetConnection **connections)
{
	unsigned int nr_connections = 0, nr_processes = 0;
	GArray *aconnections;
	GHashTable *open_sockets_hash = NULL;
	Process *processes = NULL;
	*connections = NULL;
	
	open_sockets_hash = get_open_sockets_for_processes(&processes, &nr_processes);
	
	aconnections = g_array_new(FALSE, TRUE, sizeof(NetConnection));
	get_connections_from_kernel(NC_PROTOCOL_TCP, aconnections, open_sockets_hash);
	get_connections_from_kernel(NC_PROTOCOL_TCP6, aconnections, open_sockets_hash);
	get_connections_from_kernel(NC_PROTOCOL_UDP, aconnections, open_sockets_hash);
	get_connections_from_kernel(NC_PROTOCOL_UDP6, aconnections, open_sockets_hash);

	nr_connections = aconnections->len;
	*connections = (nr_connections!=0) ? (NetConnection*)aconnections->data : NULL;
	g_array_free(aconnections, (*connections==NULL));
	
	free_processes (processes, nr_processes);
	g_hash_table_destroy(open_sockets_hash);
	
	return nr_connections;	
}

void free_net_connections(NetConnection *connections, unsigned int nconnections)
{
	unsigned int i;
	if (connections != NULL)
	{
		for(i=0; i<nconnections; i++)
			net_connection_delete_contents(connections+i);
		g_free(connections);
	}
}

void free_net_connections_array(GArray *connections)
{
	unsigned int i;
	if (connections != NULL)
	{
		for(i=0; i<connections->len; i++)
			net_connection_delete(g_array_index(connections, NetConnection*, i));
		g_array_free(connections, TRUE);
	}
}


void net_statistics_get (NetStatistics *net_stats)
{
	FILE *f;
	
	memset(net_stats, 0, sizeof(NetStatistics));
	f = fopen("/proc/net/dev", "r");
	if (f!=NULL)
	{
		char buffer[8192];
		/*skip the headers*/
		fgets(buffer, sizeof(buffer), f);
		fgets(buffer, sizeof(buffer), f);
		
		/*parse all network interfaces*/
		while(fgets(buffer, sizeof(buffer), f) != NULL)
		{
			char *ptok;
			int i;
			unsigned long long rx_bytes, rx_packets, tx_bytes, tx_packets;
			
			ptok = strchr(buffer, ':'); /*skip the interface name*/
			if (ptok==NULL)
				continue;
			ptok++;
			while (isspace (*ptok))
				ptok++;
			if (!isdigit (*ptok)) /*!digit => probably an error message*/
				continue;
			
			rx_bytes = strtoull(ptok, &ptok, 0);
			rx_packets = strtoull(ptok, &ptok, 0);
			for(i=0; i<6; i++)
				strtoull(ptok, &ptok, 0); /*skip:  errs drop fifo frame compressed multicast*/
			tx_bytes = strtoull(ptok, &ptok, 0);
			tx_packets = strtoull(ptok, &ptok, 0);
			
			net_stats->bytes_sent += tx_bytes;
			net_stats->packets_sent += tx_packets;
			net_stats->bytes_received += rx_bytes;
			net_stats->packets_received += rx_packets;
		}
		fclose(f);
	}
}


char *get_port_text (int port)
{
	char *port_text;
	if (port > 0)
		port_text = g_strdup_printf("%d", port);
	else
		port_text = g_strdup("*");
	return port_text;
}

char *get_port_name (int protocol, int port)
{
	const char *service_name = net_service_get(protocol, port);
	return (service_name != NULL) ? g_strdup(service_name) : NULL;
}

char *get_full_port_text(int protocol, int port)
{
	char *port_text;
	if (port > 0)
	{
		const char *service_name = net_service_get(protocol, port);
		if (service_name != NULL)
			port_text = g_strdup_printf("%d %s", port, service_name);
		else
			port_text = g_strdup_printf("%d", port);
	}
	else
		port_text = g_strdup("*");
	return port_text;
}

/* At any moment protocol, addresses and ports should form a unique combination.
 * They don't, but it happens rarely and for short periods of time.
 * As we compare connections at different time moments we might also get the 
 * same combinations from different processes.
 * Inode is not reliable to use in this comparision (it changes when connections changes to some states).
 */
int net_connection_net_equals (NetConnection *nc1, NetConnection *nc2)
{
	return ((nc1->localport == nc2->localport) &&
			(nc1->remoteport == nc2->remoteport) &&
			(nc1->protocol == nc2->protocol) &&
			(strcmp(nc1->remoteaddress, nc2->remoteaddress)==0) &&
			(strcmp(nc1->localaddress, nc2->localaddress)==0) && 
			(nc1->pid == 0 || nc2->pid==0 || nc1->pid == nc2->pid)
			);
}

int net_connection_equals(NetConnection *nc1, NetConnection *nc2)
{
	return ((nc1->state == nc2->state) && 
			(nc1->pid == nc2->pid) && /*Assuming name and command modify in sync with pid*/
			(net_connection_net_equals(nc1, nc2))
			);
}


static void update_string (char **old_str, const char *new_str)
{
	if (*old_str != NULL)
		g_free(*old_str);
	*old_str = (new_str != NULL) ? g_strdup(new_str) : NULL;
}

void net_connection_update (NetConnection *old_conn, NetConnection *new_conn)
{
	old_conn->state = new_conn->state;
	if (old_conn->localhost==NULL)
		update_string(&(old_conn->localhost), new_conn->localhost);
	if (old_conn->remotehost==NULL)
		update_string(&(old_conn->remotehost), new_conn->remotehost);
	old_conn->pid = new_conn->pid;
	if (new_conn->programpid != 0)
		old_conn->programpid = new_conn->programpid;
	if (new_conn->programname!=NULL)
		update_string(&(old_conn->programname), new_conn->programname);
	if (new_conn->programcommand!=NULL)
		update_string(&(old_conn->programcommand), new_conn->programcommand);
}

void net_connection_update_list_full (GArray *connections, NetConnection *latest_connections, 
									  unsigned int nr_latest_connections)
{
	unsigned int i, j;
	GArray *valid_connections;
	
	if (latest_connections == NULL || nr_latest_connections <=0)
		return;
	
	valid_connections = g_array_new(FALSE, FALSE, sizeof(NetConnection*));
	
	for (i=0; i<connections->len; i++)
	{
		NetConnection* conn = g_array_index(connections, NetConnection*, i);
		if (conn->operation != NC_OP_DELETE)
		{
			conn->operation = NC_OP_DELETE; /*DELETE if not found for update*/
			g_array_append_val(valid_connections, conn);
		}
	}
	
	for (i=0; i<nr_latest_connections; i++)
	{
		NetConnection *new_conn = latest_connections + i, *old_conn = NULL;
		
		for (j=0; j<valid_connections->len; j++)
		{
			old_conn = g_array_index(valid_connections, NetConnection*, j);
			if (net_connection_net_equals(new_conn, old_conn))
				break;
		}
		if (j < valid_connections->len) /*UPDATE or NONE*/
		{
			if (!net_connection_equals(new_conn, old_conn))
			{
				old_conn->operation = NC_OP_UPDATE;
				net_connection_update(old_conn, new_conn);
			}else
				old_conn->operation = NC_OP_NONE;
		}else /*INSERT*/
		{
			NetConnection *added_conn = net_connection_new();
			net_connection_copy(added_conn, new_conn);
			added_conn->operation = NC_OP_INSERT;
			g_array_append_val(connections, added_conn);;
		}
	}
	
	g_array_free(valid_connections, TRUE);
}

void net_connection_update_list (GArray *connections)
{
	NetConnection *latest_connections = NULL;
	unsigned int nr_latest_connections;
	nr_latest_connections = get_net_connections(&latest_connections);
	if (latest_connections != NULL)
	{
		net_connection_update_list_full(connections, latest_connections, nr_latest_connections);
		free_net_connections(latest_connections, nr_latest_connections);
	}
}


typedef union
{
	struct in_addr addr;
	struct in6_addr addr6;
}generic_in_addr;

static int generic_in_addr_is_zero (generic_in_addr *addr)
{
	return ( (addr->addr6.s6_addr32[0] == 0) && (addr->addr6.s6_addr32[1] == 0) && 
			(addr->addr6.s6_addr32[2] == 0) && (addr->addr6.s6_addr32[3] == 0) );
}

char *get_host_name_by_address (const char* saddress)
{
	char *host = NULL;
	
	if (strlen(saddress) <= 1)
	{
		host = NULL;
	}else
	{
		generic_in_addr addr;
		int address_format;
		
		memset(&addr, 0, sizeof(generic_in_addr));
		address_format = (strchr(saddress, '.')!=NULL) ? AF_INET : AF_INET6;
		inet_pton(address_format, saddress, &addr);
		if (generic_in_addr_is_zero(&addr))
		{
			host = NULL;
		}else
		{
			int result, error;
			size_t buffer_length;
			struct hostent hostbuffer, *host_info = NULL;
			char *buffer;
			
			buffer_length = 1024;
			buffer = g_malloc(buffer_length);
			
			do
			{
				result = gethostbyaddr_r((char*)&addr, sizeof(generic_in_addr), address_format, 
										 &hostbuffer, buffer, buffer_length, &host_info, &error);
				if (result == ERANGE)
				{
					buffer_length *= 2;
					buffer = g_realloc(buffer, buffer_length);
				}
			}while (result == ERANGE);
			
			host = (result == 0 && host_info!=NULL && host_info->h_name!=NULL) ? 
				g_strdup(host_info->h_name) : NULL;
			
			g_free(buffer);
		}
	}
	
	return host;
}

static int network_value_compare(unsigned int v1, unsigned int v2)
{
	v1 = ntohl(v1);
	v2 = ntohl(v2);
	return (v1 < v2) ? -1 : ( (v1 > v2) ? 1 : 0 );
}

int compare_addresses(const char *saddr1, const char *saddr2)
{
	int result = 0;
	
	if (strlen(saddr1)<=1 || strlen(saddr2)<=1) /*including '*' which is equivalent for 0.0.0.0, ::*/
	{
		result = strcmp(saddr1, saddr2);
	}else
	{
		int format1, format2;
		format1 = (strchr(saddr1, '.')!=NULL) ? AF_INET : AF_INET6;
		format2 = (strchr(saddr2, '.')!=NULL) ? AF_INET : AF_INET6;
		if (format1 == format2)
		{
			if (format1 == AF_INET)
			{
				struct in_addr addr1, addr2;		
				inet_pton(format1, saddr1, &addr1);
				inet_pton(format2, saddr2, &addr2);			
				result = network_value_compare(addr1.s_addr, addr2.s_addr);
			}else
			{
				struct in6_addr addr1, addr2;		
				inet_pton(format1, saddr1, &addr1);
				inet_pton(format2, saddr2, &addr2);	
				result = network_value_compare(addr1.s6_addr32[0], addr2.s6_addr32[0]);
				if (result == 0)
					result = network_value_compare(addr1.s6_addr32[1], addr2.s6_addr32[1]);
				if (result == 0)
					result = network_value_compare(addr1.s6_addr32[2], addr2.s6_addr32[2]);
				if (result == 0)
					result = network_value_compare(addr1.s6_addr32[3], addr2.s6_addr32[3]);
			}
		}else
		{
			result = (format1 == AF_INET) ? -1 : 1;
		}
	}
	return result;
}

/* Compare domain names before subdomain names
 */
int compare_hosts(const char *host1, const char *host2)
{
	int result = 0;
	int len1 = strlen(host1), len2 = strlen(host2);
	if (len1 > 0 && len2 > 0)
	{
		while (result == 0 && len1 > 0 && len2 > 0)
		{
			int i1, i2, l1, l2;
			for(i1=len1-1; i1>=0; i1--)
				if (host1[i1]=='.')
					break;
			l1 = len1 - i1 - 1;
			for(i2=len2-1; i2>=0; i2--)
				if (host2[i2]=='.')
					break;
			l2 = len2 - i2 - 1;
			
			if (l1 > 0 && l2 > 0)
			{
				result = strncmp(host1+i1+1, host2+i2+1, (l1<l2) ? l1 : l2);
				if (result == 0 && l1 != l2)
					result = (l1 < l2) ? -1 : 1;
			}else
				result = (l1 < l2) ? -1 : ((l1 > l2) ? 1 : 0);
			
			len1 = i1;
			len2 = i2;
		}
		
	}else
		result = (len1 < len2) ? -1 : ((len1 > len2) ? 1 : 0);
	
	return result;
}

