/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.util;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.function.Supplier;
import org.apache.ratis.util.IOUtils;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.LogUtils;
import org.apache.ratis.util.TimeDuration;
import org.apache.ratis.util.function.CheckedSupplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public interface FileUtils {
    public static final Logger LOG = LoggerFactory.getLogger(FileUtils.class);
    public static final int NUM_ATTEMPTS = 5;
    public static final TimeDuration SLEEP_TIME = TimeDuration.ONE_SECOND;

    public static <T> T attempt(CheckedSupplier<T, IOException> op, Supplier<?> name) throws IOException {
        try {
            return JavaUtils.attempt(op, 5, SLEEP_TIME, name, LOG);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw IOUtils.toInterruptedIOException("Interrupted " + name.get(), e);
        }
    }

    public static void truncateFile(File f, long target) throws IOException {
        long original = f.length();
        LogUtils.runAndLog(LOG, () -> {
            try (FileChannel channel = FileChannel.open(f.toPath(), StandardOpenOption.WRITE);){
                channel.truncate(target);
            }
        }, () -> "FileChannel.truncate " + f + " length: " + original + " -> " + target);
    }

    public static InputStream newInputStream(String s, OpenOption ... options) throws IOException {
        return FileUtils.newInputStream(Paths.get(s, new String[0]), options);
    }

    public static InputStream newInputStream(File f, OpenOption ... options) throws IOException {
        return FileUtils.newInputStream(f.toPath(), options);
    }

    public static InputStream newInputStream(Path p, OpenOption ... options) throws IOException {
        return LogUtils.supplyAndLog(LOG, () -> Files.newInputStream(p, options), () -> "Files.newInputStream " + p + " with options " + Arrays.asList(options));
    }

    public static OutputStream newOutputStream(File f, OpenOption ... options) throws IOException {
        return FileUtils.newOutputStream(f.toPath(), options);
    }

    public static OutputStream newOutputStream(Path p, OpenOption ... options) throws IOException {
        return LogUtils.supplyAndLog(LOG, () -> Files.newOutputStream(p, options), () -> "Files.newOutputStream " + p + " with options " + Arrays.asList(options));
    }

    public static OutputStream newOutputStream(final FileChannel channel, final boolean forceAtClose) {
        final byte[] single = new byte[]{0};
        return new OutputStream(){

            @Override
            public void write(int b) throws IOException {
                single[0] = (byte)b;
                this.write(single);
            }

            @Override
            public void write(byte[] b, int off, int len) throws IOException {
                while (len > 0) {
                    int written = channel.write(ByteBuffer.wrap(b, off, len));
                    off += written;
                    len -= written;
                }
            }

            @Override
            public void close() throws IOException {
                try (FileChannel c = channel;){
                    if (forceAtClose) {
                        c.force(true);
                    }
                }
            }
        };
    }

    public static OutputStream newOutputStreamForceAtClose(Path p, OpenOption ... options) throws IOException {
        return FileUtils.newOutputStream(FileUtils.newFileChannel(p, options), true);
    }

    public static OutputStream newOutputStreamForceAtClose(File f, OpenOption ... options) throws IOException {
        return FileUtils.newOutputStreamForceAtClose(f.toPath(), options);
    }

    public static FileChannel newFileChannel(File f, OpenOption ... options) throws IOException {
        return FileUtils.newFileChannel(f.toPath(), options);
    }

    public static FileChannel newFileChannel(Path p, OpenOption ... options) throws IOException {
        return LogUtils.supplyAndLog(LOG, () -> FileChannel.open(p, options), () -> "FileChannel.open " + p + " with options " + Arrays.asList(options));
    }

    public static void createDirectories(File dir) throws IOException {
        FileUtils.createDirectories(dir.toPath());
    }

    public static void createDirectories(Path dir) throws IOException {
        LogUtils.runAndLog(LOG, () -> Files.createDirectories(dir, new FileAttribute[0]), () -> "Files.createDirectories " + dir);
    }

    public static void createDirectoriesDeleteExistingNonDirectory(File dir) throws IOException {
        FileUtils.createDirectoriesDeleteExistingNonDirectory(dir.toPath());
    }

    public static void createDirectoriesDeleteExistingNonDirectory(Path dir) throws IOException {
        try {
            FileUtils.createDirectories(dir);
        }
        catch (FileAlreadyExistsException e) {
            LOG.warn("Failed to create directory " + dir + " since it already exists as a non-directory.  Trying to delete it ...", (Throwable)e);
            FileUtils.delete(dir);
            FileUtils.createDirectories(dir);
        }
    }

    public static void move(File src, File dst, CopyOption ... options) throws IOException {
        FileUtils.move(src.toPath(), dst.toPath(), options);
    }

    public static void move(Path src, Path dst, CopyOption ... options) throws IOException {
        Objects.requireNonNull(options, "options == null");
        List<CopyOption> original = Arrays.asList(options);
        boolean atomicMove = original.contains(StandardCopyOption.ATOMIC_MOVE);
        if (atomicMove) {
            LogUtils.runAndLog(LOG, () -> Files.move(src, dst, options), () -> "Files.move " + src + " to " + dst + " with options " + original);
            return;
        }
        CopyOption[] optionsWithAtomicMove = new CopyOption[options.length + 1];
        optionsWithAtomicMove[0] = StandardCopyOption.ATOMIC_MOVE;
        System.arraycopy(options, 0, optionsWithAtomicMove, 1, options.length);
        Supplier<String> suffix = () -> original.isEmpty() ? "" : " with options " + original;
        try {
            LogUtils.runAndLog(LOG, () -> Files.move(src, dst, optionsWithAtomicMove), () -> "Atomic Files.move " + src + " to " + dst + (String)suffix.get());
        }
        catch (AtomicMoveNotSupportedException e) {
            LogUtils.runAndLog(LOG, () -> Files.move(src, dst, options), () -> "Atomic move not supported. Fallback to Files.move " + src + " to " + dst + (String)suffix.get());
        }
    }

    public static File move(File src, String suffix) throws IOException {
        File newFile = new File(src.getPath() + suffix);
        FileUtils.move(src, newFile, new CopyOption[0]);
        return newFile;
    }

    public static Path deleteFile(File f) throws IOException {
        Path path = f.toPath();
        FileUtils.delete(path);
        return path;
    }

    public static void moveDirectory(final Path source, final Path dest) throws IOException {
        if (LOG.isTraceEnabled()) {
            LOG.trace("moveDirectory source: {} dest: {}", (Object)source, (Object)dest);
        }
        FileUtils.createDirectories(dest);
        Files.walkFileTree(source, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                Path targetPath = dest.resolve(source.relativize(dir));
                if (!Files.exists(targetPath, new LinkOption[0])) {
                    FileUtils.createDirectories(targetPath);
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                try {
                    FileUtils.move(file, dest.resolve(source.relativize(file)), new CopyOption[0]);
                }
                catch (IOException e) {
                    LOG.info("Files.moveDirectory: could not delete {}", (Object)file.getFileName());
                }
                return FileVisitResult.CONTINUE;
            }
        });
        FileUtils.deleteFully(source);
    }

    public static boolean deleteFileQuietly(File f) {
        return FileUtils.deletePathQuietly(f.toPath());
    }

    public static boolean deletePathQuietly(Path p) {
        try {
            FileUtils.delete(p);
            return true;
        }
        catch (Exception ex) {
            LOG.debug("Failed to delete " + p.toAbsolutePath(), (Throwable)ex);
            return false;
        }
    }

    public static void delete(Path p) throws IOException {
        LogUtils.runAndLog(LOG, () -> Files.delete(p), () -> "Files.delete " + p);
    }

    public static void deleteIfExists(Path p) throws IOException {
        LogUtils.runAndLog(LOG, () -> Files.deleteIfExists(p), () -> "Files.deleteIfExists " + p);
    }

    public static void deleteIfExists(File f) throws IOException {
        FileUtils.deleteIfExists(f.toPath());
    }

    public static void deleteFully(File f) throws IOException {
        LOG.trace("deleteFully {}", (Object)f);
        FileUtils.deleteFully(f.toPath());
    }

    public static void deleteFully(Path p) throws IOException {
        if (!Files.exists(p, LinkOption.NOFOLLOW_LINKS)) {
            LOG.trace("deleteFully: {} does not exist.", (Object)p);
            return;
        }
        Files.walkFileTree(p, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                FileUtils.delete(file);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException e) throws IOException {
                if (e != null) {
                    throw e;
                }
                FileUtils.delete(dir);
                return FileVisitResult.CONTINUE;
            }
        });
    }
}

