/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "MPL"); you may not use this file except in
 * compliance with the MPL. You may obtain a copy of the MPL at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the MPL is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the MPL
 * for the specific language governing rights and limitations under the
 * MPL.
 *
 * The Original Code is Enigmail.
 *
 * The Initial Developer of the Original Code is
 * Ramalingam Saravanan <sarava@sarava.net>
 * Portions created by the Initial Developer are Copyright (C) 2002
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or 
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

// Logging of debug output 
// The following define statement should occur before any include statements
#define FORCE_PR_LOG       /* Allow logging even in release build */

#include "mimeenig.h"
#include "nsEnigModule.h"
#include "nsEnigPop3Proxy.h"
#include "nspr.h"
#include "nsString.h"
#include "nsCOMPtr.h"
#include "nsIThread.h"

#include <assert.h>
#include <string.h>

/* RFC 1939 (POP3) specifies a maximum response line length of 512 octets
 * including the terminating CRLF
 */
#define BUF_SIZE 512

#ifdef PR_LOGGING
PRLogModuleInfo* gEnigPop3ProxyLog = NULL;
#endif

#define ERROR_LOG(args)    PR_LOG(gEnigPop3ProxyLog,PR_LOG_ERROR,args)
#define WARNING_LOG(args)  PR_LOG(gEnigPop3ProxyLog,PR_LOG_WARNING,args)
#define DEBUG_LOG(args)    PR_LOG(gEnigPop3ProxyLog,PR_LOG_DEBUG,args)


static void
enig_pop3_proxy_thread(void *arg1);

static
int enig_pop3_command(PRFileDesc* client_socket, PRFileDesc* server_socket,
                      PRIntervalTime timeout);

static
int enig_pop3_response(PRFileDesc* client_socket, PRFileDesc* server_socket,
                       PRIntervalTime timeout, int multiline);

#define MAX_MULTILINE_COMMAND_NO 4
#define MAX_MULTILINE_NO_ARG_COMMAND_NO 2

#define CAPA_COMMAND 1
#define RETR_COMMAND 2

static
const char* pop3_commands[] = {"CAPA", "RETR", "LIST", "UIDL",
                               "STAT", "DELE", "NOOP", "RSET",
                               "USER", "PASS", "APOP", "AUTH",
                               "QUIT", NULL};

/* TOP command capability is not allowed for consistency.
 * The output of the CAPA command should be filtered by the proxy
 * to remove the TOP capability
 */

// nsEnigPop3Proxy implementation

// nsISupports implementation
NS_IMPL_THREADSAFE_ISUPPORTS1(nsEnigPop3Proxy,
                              nsIEnigPop3Proxy);


// nsEnigPop3Proxy implementation
nsEnigPop3Proxy::nsEnigPop3Proxy()
  : mInitialized(PR_FALSE)
{
  nsresult rv;

  NS_INIT_ISUPPORTS();

#ifdef PR_LOGGING
  if (gEnigPop3ProxyLog == nsnull) {
    gEnigPop3ProxyLog = PR_NewLogModule("nsEnigPop3Proxy");
  }
#endif

#ifdef FORCE_PR_LOG
  nsCOMPtr<nsIThread> myThread;
  rv = nsIThread::GetCurrent(getter_AddRefs(myThread));
  DEBUG_LOG(("nsEnigPop3Proxy:: <<<<<<<<< CTOR(%x): myThread=%x\n",
         (int) this, (int) myThread.get()));
#endif
}


nsEnigPop3Proxy::~nsEnigPop3Proxy()
{
  nsresult rv;
#ifdef FORCE_PR_LOG
  nsCOMPtr<nsIThread> myThread;
  rv = nsIThread::GetCurrent(getter_AddRefs(myThread));
  DEBUG_LOG(("nsEnigPop3Proxy:: >>>>>>>>> DTOR(%x): myThread=%x\n",
         (int) this, (int) myThread.get()));
#endif

}


///////////////////////////////////////////////////////////////////////////////
// nsIEnigPop3Proxy methods:
///////////////////////////////////////////////////////////////////////////////

NS_IMETHODIMP
nsEnigPop3Proxy::Init()
{
  DEBUG_LOG(("nsEnigMimeService::Init:\n"));

  PRThread * proxyThread = PR_CreateThread( PR_USER_THREAD,
                                            enig_pop3_proxy_thread,
                                            (void*) NULL,
                                            PR_PRIORITY_NORMAL,
                                            PR_GLOBAL_THREAD,
                                            PR_UNJOINABLE_THREAD, 0);

  if (!proxyThread)
    return NS_ERROR_FAILURE;

  mInitialized = PR_TRUE;

  return NS_OK;
}

static void
enig_pop3_proxy_thread(void *arg1)
{

  PRStatus status;

  PRHostEnt host_entry;

  PRNetAddr proxy_addr, client_addr, server_addr, peer_addr;

  PRFileDesc *proxy_socket = NULL;
  PRFileDesc *client = NULL;
  PRFileDesc *server = NULL;

  PRIntervalTime timeout;

  PRInt32 client_ip;

  int command_no, retcode;

  PRIntn next_index;

  char netdb_buf[PR_NETDB_BUF_SIZE];

  char pop3_server[] = "pop.dnvr.qwest.net";
  PRUint16 server_port = 110;
  PRUint16 proxy_port = 51213;

  timeout = PR_SecondsToInterval(120); /* 120 sec timeout */

  proxy_addr.inet.family = PR_AF_INET;
  proxy_addr.inet.ip     = PR_htonl(PR_INADDR_LOOPBACK);
  proxy_addr.inet.port   = PR_htons(proxy_port);

  proxy_socket = PR_NewTCPSocket();
  if (!proxy_socket)
    return;

  status = PR_Bind(proxy_socket, &proxy_addr);
  if (status != PR_SUCCESS) {
    fprintf(stderr, "enig_pop3_proxy_thread: PR_Bind failed (%d)\n", PR_GetError());
    return;
  }

  PRIntn backlog = 4;
  status = PR_Listen(proxy_socket, backlog);
  if (status != PR_SUCCESS) {
    fprintf(stderr, "enig_pop3_proxy_thread: PR_Listen failed\n");
    return;
  }

  while (1) {
    /* Close any open client/server connection */

    if (client) {
      /* Shutdown client connection */
      fprintf(stderr, "enig_pop3_proxy_thread: client shutdown\n");
      status = PR_Shutdown(client, PR_SHUTDOWN_BOTH);
      PR_Close(client);
      client = NULL;
    }

    if (server) {
      /* Shutdown server connection */
      fprintf(stderr, "enig_pop3_proxy_thread: server shutdown\n");
      status = PR_Shutdown(server, PR_SHUTDOWN_BOTH);
      PR_Close(server);
      server = NULL;
    }

    fprintf(stderr, "enig_pop3_proxy_thread: listening on loopback port=%d\n", proxy_port);

    /* Wait for next connection attempt from POP3 client */
    client = PR_Accept(proxy_socket, &client_addr, PR_INTERVAL_NO_TIMEOUT);

    if (client == NULL) {
      fprintf(stderr, "enig_pop3_proxy_thread: PR_Accept failed\n");
      break;
    }

    client_ip = PR_ntohl(client_addr.inet.ip);
    fprintf(stderr, "enig_pop3_proxy_thread: client ip=%x\n", client_ip);

    /* Only local host can connect as client */
    if ( (client_addr.inet.family != PR_AF_INET) ||
         (client_ip               != PR_INADDR_LOOPBACK) ) {
      continue;
    }

    /* Connect to POP3 server */
    status = PR_GetHostByName( pop3_server,
                               netdb_buf, sizeof(netdb_buf), &host_entry);
    if (status == PR_FAILURE) {
      fprintf(stderr, "enig_pop3_proxy_thread: failed to lookup POP3 server %s\n", pop3_server);
      continue;
    }

    next_index = PR_EnumerateHostEnt(0, &host_entry, server_port, &server_addr);
    if (next_index == -1) {
      fprintf(stderr, "enig_pop3_proxy_thread: PR_EnumerateHostEnt failed\n");
      continue;
    }

    fprintf(stderr, "enig_pop3_proxy_thread: server ip=%x\n", server_addr.inet.ip);

    server = PR_NewTCPSocket();
    if (server == NULL) {
      fprintf(stderr, "enig_pop3_proxy_thread: PR_NewTCPSocket failed for server\n");
      continue;
    }

    status = PR_Connect(server, &server_addr, timeout);
    if (status == PR_FAILURE) {
      fprintf(stderr, "enig_pop3_proxy_thread: failed to connect to server\n");
      continue;
    }

    status = PR_GetPeerName(server, &peer_addr);
    if (status == PR_FAILURE) {
      fprintf(stderr, "enig_pop3_proxy_thread: PR_GetPeerName failed\n");
      continue;
    }

    while (1) {
      command_no = enig_pop3_command(client, server, timeout);
      if (command_no < -1)
        break;

      fprintf(stderr, "enig_pop3_proxy_thread: command no = %d\n", command_no);

      if (command_no >= 0) {
        retcode = enig_pop3_response(client, server, timeout, command_no > 0);
        if (retcode < -1)
          break;
      }
    }
  }

  fprintf(stderr, "enig_pop3_proxy_thread: shutting down proxy\n");

  /* Terminate proxy socket */
  status = PR_Shutdown(proxy_socket, PR_SHUTDOWN_BOTH);
  PR_Close(proxy_socket);

  return;
}

static
void enig_pop3_err(PRFileDesc* socket, PRIntervalTime timeout,
                   const char* errmsg)
{
  PRInt32 n_written;

  n_written = PR_Send(socket, "-ERR ", 5, 0, timeout);

  n_written = PR_Send(socket, errmsg, strlen(errmsg), 0, timeout);

  n_written = PR_Send(socket, "\r\n", 2, 0, timeout);

}

static char enig_cmd_buf[BUF_SIZE];
static long enig_cmd_count = 0;
static int enig_cmd_cr_found = 0;

/* enig_pop3_command: receives single command from client and relays it
 * to the server
 * return codes: -3 => error in relaying command to server
 *               -2 => error/EOF in command received from client
 *               -1 => non-fatal error (no response expected)
 *                0 => command with single line response
 *                >= 1 command no. for multiline response command
 * A return code of -2 could mean a normal EOF or error. In any case,
 * the session should be terminated for return codes < -1
 * (The appropriate error responses would already have been sent.)
 */

static
int enig_pop3_command(PRFileDesc* client_socket, PRFileDesc* server_socket,
                      PRIntervalTime timeout)
{
  PRInt32 n_read, n_written;
  int multiline, lf_found, m, k;
  int command_no, arg_found;

  assert(BUF_SIZE >= 5);

  fprintf(stderr, "enig_pop3_command: %d\n", timeout);

  while (enig_cmd_count < 5) {
    /* Read at least 5 characters into buffer */
    fprintf(stderr, "enig_pop3_command: reading from client\n");
    n_read = PR_Recv(client_socket, enig_cmd_buf+enig_cmd_count,
                     5-enig_cmd_count, 0, timeout);

    fprintf(stderr, "enig_pop3_command: command n_read=%d\n", n_read);
    if (n_read <= 0) {
      /* Error/end-of-file */
      return -2;
    }

    enig_cmd_count += n_read;
  }

  arg_found = 0;
  for (k=0; k < enig_cmd_count; k++) {
    if (enig_cmd_buf[k] == ' ') {
      arg_found = 1;
      break;

    } else if ((enig_cmd_buf[k] == '\r') || (enig_cmd_buf[k] == '\n')) {
      break;
    }
  }

  if ((k < 3) || (k > 4)) {
    /* Keyword too short/long */
    enig_pop3_err(client_socket, timeout,
                  "Invalid/missing keyword in command");
    return -2;
  }

  m = 0;
  command_no = 0;

  while (pop3_commands[m]) {
    if (PL_strncasecmp(enig_cmd_buf, pop3_commands[m], k) == 0) {
      command_no = m+1;
    }
    m++;
  }

  multiline = 0;

  fprintf(stderr, "enig_pop3_command: k=%d, command_no=%d\n", k, command_no);

  if ( (command_no > 0) &&
       ( (command_no <= MAX_MULTILINE_NO_ARG_COMMAND_NO) ||
         ((command_no <= MAX_MULTILINE_COMMAND_NO) && !arg_found) ) ) {
    multiline = 1;
  }

  lf_found = 0;

  while (1) {

    while (!lf_found && (k < enig_cmd_count)) {
      if (enig_cmd_cr_found) {

        if (enig_cmd_buf[k] != '\n') {
          /* CR not followed by LF */
          enig_pop3_err(client_socket, timeout,
                        "CR not followed by LF in command");
          return -2;
        }

        enig_cmd_cr_found = 0;
        lf_found = 1;

      } else if (enig_cmd_buf[k] == '\n') {
        /* LF before CR */
        enig_pop3_err(client_socket, timeout,
                      "LF encountered before CR in command");
        return -2;

      } else if (enig_cmd_buf[k] == '\r') {
        enig_cmd_cr_found = 1;
      }

      k++;
    }

    if (command_no > 0) {
      /* Relay only recognized commands */
      n_written = PR_Send(server_socket, enig_cmd_buf, k, 0, timeout);
      fprintf(stderr, "enig_pop3_command: n_written=%d\n", n_written);

      if (n_written < k) {
        /* Not all data written */
        enig_pop3_err(client_socket, timeout,
                    "Failed to relay command to POP3 server");
        return -3;
      }
    }

    enig_cmd_count -= k;
    if (enig_cmd_count > 0) {
      memmove(enig_cmd_buf, enig_cmd_buf+k, enig_cmd_count);
    }

    if (lf_found) {
      lf_found = 0;
      break;
    }

    assert(enig_cmd_count == 0);

    fprintf(stderr, "enig_pop3_command: reading from client\n");
    n_read = PR_Recv(client_socket, enig_cmd_buf, BUF_SIZE, 0, timeout);

    fprintf(stderr, "enig_pop3_command: client n_read=%d\n", n_read);
    if (n_read <= 0) {
      /* Error/end-of-file */
      return -2;
    }

    enig_cmd_count = n_read;
    k = 0;
  }

  if (command_no == 0) {
    enig_pop3_err(client_socket, timeout,
                  "Command not recognized");
    return -1;
  }

  return multiline ? command_no : 0;
}


static char enig_resp_buf[BUF_SIZE];
static long enig_resp_count = 0;
static int enig_resp_dot_found = 0;
static int enig_resp_cr_found = 0;

/* enig_pop3_response: receives response to signle command from server and
 * relays it to the client
 * return codes: -3 => error in receiving response from server
 *               -2 => error in relaying response to client
 *               -1 => non-fatal error
 *                0 => normal exit
 * If multiline, on normal exit, the data, including
 * the OK response status line, is available in the buffer.
 * The proxy should process the data in the buffer, and send the status line
 * followed by the processed data.
 * For return code < -1, all connections should be shutdown, and if multiline,
 * the buffer should be cleared.
 * (The appropriate error response would already have been sent.)
 */

static
int enig_pop3_response(PRFileDesc* client_socket, PRFileDesc* server_socket,
                       PRIntervalTime timeout, int multiline)
{
  PRInt32 n_read, n_written;
  int term_found, lf_found, k;
  int line_len, line_count;

  assert(BUF_SIZE >= 512);

  fprintf(stderr, "enig_pop3_response: %d\n", multiline);

  while (enig_resp_count < 5) {
    /* Read at least 5 characters into buffer */
    n_read = PR_Recv(server_socket, enig_resp_buf+enig_resp_count,
                     5-enig_resp_count, 0, timeout);

    fprintf(stderr, "enig_pop3_response: status n_read=%d\n", n_read);
    if (n_read <= 0) {
      /* Error/end-of-file */
      enig_pop3_err(client_socket, timeout,
                    "Error in receiving data from POP3 server");
      return -3;
    }

    enig_resp_count += n_read;
  }

  for (k=0; k < enig_resp_count; k++) {
    if ((enig_resp_buf[k] == ' ') ||
        (enig_resp_buf[k] == '\r') ||
        (enig_resp_buf[k] == '\n')) {
      break;
    }
  }

  if ((k < 3) || (k > 4)) {
    /* Response code too short/long */
    enig_pop3_err(client_socket, timeout,
                  "Invalid/missing response status indicator");
    return -3;
  }

  if (PL_strncasecmp(enig_resp_buf, "-ERR", k) == 0) {
    /* Error response; no multiline */
    multiline = 0;

  } else if (PL_strncasecmp(enig_resp_buf, "+OK", k) != 0) {
    enig_pop3_err(client_socket, timeout,
                  "Invalid response status indicator");
    return -3;
  }

  lf_found = 0;
  term_found = 0;

  line_len = enig_resp_count;
  line_count = 0;

  while (1) {

    while (!lf_found && (k < enig_resp_count)) {

      if (enig_resp_cr_found) {

        if (enig_resp_buf[k] != '\n') {
          /* CR not followed by LF */
          enig_pop3_err(client_socket, timeout,
                        "CR not followed by LF in response");
          return -3;
        }

        enig_resp_dot_found = 0;
        enig_resp_cr_found = 0;
        lf_found = 1;

      } else if (enig_resp_buf[k] == '\n') {
        /* LF before CR */
        enig_pop3_err(client_socket, timeout,
                      "LF encountered before CR in response");
        return -3;

      } else if (enig_resp_buf[k] == '\r') {
        enig_resp_cr_found = 1;

        if ((line_len == 1) && enig_resp_dot_found)
          term_found = 1;

      } else if (enig_resp_buf[k] == '.') {
        enig_resp_dot_found = 1;
      }

      line_len++;
      k++;
    }

    if (!lf_found) {
      /* Not yet end of line; read some more */

      if (enig_resp_count >= BUF_SIZE) {
        enig_pop3_err(client_socket, timeout,
                      "Response line too long");
        return -3;
      }

      n_read = PR_Recv(server_socket, enig_resp_buf+enig_resp_count,
                       BUF_SIZE-enig_resp_count, 0, timeout);

      fprintf(stderr, "enig_pop3_response: server n_read=%d\n", n_read);
      if (n_read <= 0) {
        /* Error/end-of-file */
        enig_pop3_err(client_socket, timeout,
                      "Error in receiving data from POP3 server");
        return -3;
      }

      enig_resp_count += n_read;

    } else {
      /* End of line */

      if (0 && multiline) {
        /* At this point, all lines, including the OK response status line
           should be written to a buffer for processing. If an error occurs
           subsequently, an error response should be returned instead of
           the OK response.
           The output of the CAPA should be filtered to remove the TOP
           capability, for consistency
        */

      } else {
        /* Relay complete line to client */
        n_written = PR_Send(client_socket, enig_resp_buf, k, 0, timeout);
        fprintf(stderr, "enig_pop3_command: n_written=%d\n", n_written);

        if (n_written < k) {
          /* Not all data written */
          return -2;
        }
      }

      enig_resp_count -= k;

      if (enig_resp_count > 0) {
        memmove(enig_resp_buf, enig_resp_buf+k, enig_resp_count);
      }

      k = 0;

      lf_found = 0;
      line_len = 0;
      line_count++;

      if (!multiline || term_found)
        break;
    }

  }

  return 0;
}
