/******************************************************************************
 JErrorState.cc

	The JErrorState base class gives an object a JError error state, which can
	be used as an alternative to returning an error every time.  If derived
	objects write their methods to take no action when the object is in an
	error state ('if (!OK()) return;'), then this system can provide some of
	the convenience of exceptions (errors can be dealt earlier on the stack
	than the point at which they occur) without the egregiously dangerous
	features of C++ exceptions.  It won't work that way for all objects or in
	all code, but it also won't drop memory....

	This implementation should suffice for most purposes, but all functions
	are virtual for classes with more complex error reporting needs (JIPCIos,
	for example, must sync its error state with the underlying ios object).

	Warning: the state is mutable and classes can set and clear an error state
	on a const object through protected functions, arguably a nasty feature.
	This is so because it was necessary for JIPCIos, but it might also be
	useful in other cases.  If you do not want this behavior, the fix is
	simple: don't use the functions with names beginning with 'Mutable', an
	easy restriction to live with.  They are protected, so you never have to
	worry about others modifying your const objects unless you allow them to.

	Question: should this class be considered a mixin and always be inherited
	virtually?

	BASE CLASS = <NONE>

	Copyright  1997 by Dustin Laurence.  All rights reserved.
	
	Base code generated by Codemill v0.1.0

 *****************************************************************************/

#include <JErrorState.h>
#include <jAssert.h>

/******************************************************************************
 Constructor

 *****************************************************************************/

JErrorState::JErrorState()
	:
	itsError( JNoError() )
{
}

/******************************************************************************
 Destructor

 *****************************************************************************/

JErrorState::~JErrorState()
{
}

/******************************************************************************
 OK (virtual)

	Returns kTrue if the object is not in an error state (its error is JNoError)
	for convenience in boolean tests.

 *****************************************************************************/

JBoolean
JErrorState::OK() const
{
	return itsError.OK();
}

/******************************************************************************
 Error (virtual)

	Returns the current error state.

 *****************************************************************************/

JError
JErrorState::Error() const
{
	return itsError;
}

/******************************************************************************
 ClearError (virtual)

	Clears any error state, equivalent to SetError( JNoError() ).  Make sure
	you dealt with it first...

 *****************************************************************************/

void
JErrorState::ClearError()
{
	itsError = JNoError();
}

/******************************************************************************
 SetError (virtual protected)

	Sets the error state.

 *****************************************************************************/

void
JErrorState::SetError
	(
	const JError error
	)
{
	itsError = error;
}

/******************************************************************************
 MutableClearError (virtual protected)

	As ClearError(), but can clear the error state on a const object.  Avoid
	where possible.

 *****************************************************************************/

void
JErrorState::MutableClearError()
	const // Nasty!
{
	JError* mutableError = const_cast<JError*>(&itsError);
	*mutableError = JNoError();
}

/******************************************************************************
 MutableSetError (virtual protected)

	As SetError(), but can change the error state on a const object.  Avoid
	where possible.

 *****************************************************************************/

void
JErrorState::MutableSetError
	(
	const JError error
	)
	const // Nasty!
{
	JError* mutableError = const_cast<JError*>(&itsError);
	*mutableError = error;
}

