/******************************************************************************
 JUNIXPrefsFile.cc

							The UNIX Preferences File Class

	Manages a preferences file (.*.pref) stored in the user's home directory.

	To guarantee that your preference file signature is unique, you should
	register it with jx-registry@alice.wonderland.caltech.edu
	(This is a free service.  We simply check whether anybody else has
	 already registered the signature.)

	BASE CLASS = JPrefsFile

	Copyright  1995 John Lindal. All rights reserved.

 ******************************************************************************/

#include <JUNIXPrefsFile.h>
#include <JString.h>
#include <jDirUtil.h>
#include <jAssert.h>

// JError data

const JCharacter* JUNIXPrefsFile::kNoHomeDirectory =
	"NoHomeDirectory::JUNIXPrefsFile";

const JCharacter* JUNIXPrefsFile::kNoHomeDirectoryMsg =
	"Unable to create a preferences file because you don't have a home directory.";

/******************************************************************************
 Constructor function (static)

	By forcing everyone to use this function, we avoid having to worry
	about errors within the class itself.

 ******************************************************************************/

JError
JUNIXPrefsFile::Create
	(
	const JCharacter*	fileName,
	JUNIXPrefsFile**	obj
	)
{
	const JError err = OKToCreate(fileName);
	if (err.OK())
		{
		*obj = new JUNIXPrefsFile(fileName);
		assert( *obj != NULL );
		}
	else
		{
		*obj = NULL;
		}
	return err;
}

JError
JUNIXPrefsFile::OKToCreate
	(
	const JCharacter* fileName
	)
{
	JString filePath;
	if (!JGetHomeDirectory(&filePath))
		{
		return NoHomeDirectory();
		}

	const JString fullName = GetFullName(fileName);
	return OKToCreateBase(fullName, "");
}

/******************************************************************************
 Constructor (protected)

	Opens the specified preferences file.  The . is prepended and the .pref
	is appended automatically.

 ******************************************************************************/

JUNIXPrefsFile::JUNIXPrefsFile
	(
	const JCharacter* fileName
	)
	:
	JPrefsFile(GetFullName(fileName))
{
}

/******************************************************************************
 Destructor

 ******************************************************************************/

JUNIXPrefsFile::~JUNIXPrefsFile()
{
}

/******************************************************************************
 GetFullName (static)

 ******************************************************************************/

JString
JUNIXPrefsFile::GetFullName
	(
	const JCharacter* fileName
	)
{
	JString filePath;
	if (!JGetHomeDirectory(&filePath))
		{
		filePath.Clear();
		}

	filePath += ".";
	filePath += fileName;
	filePath += ".pref";
	return filePath;
}
