/* ************************************************************************ 
 *         The Amulet User Interface Development Environment              *
 * ************************************************************************
 * This code was written as part of the Amulet project at                 *
 * Carnegie Mellon University, and has been placed in the public          *
 * domain.  If you are using this code or any part of Amulet,             *
 * please contact amulet@cs.cmu.edu to be put on the mailing list.        *
 * ************************************************************************/

/* This file contains member function definitions for the Am_Drawonable_Impl
   object primarily concerned with interaction.
   
   Designed and implemented by Brad Myers
*/

extern "C" {
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include <X11/Xatom.h>
}

#include <iostream.h>

#include <am_inc.h>

#include GEM__H

#include "gemX.h"

extern Screen_Manager Scrn_Mgr;

bool Am_Main_Loop_Go = true;

// An array of XEvent names used for debugging.  If you have an
// XEvent variable named event_return, do event_names[event_return.type] to
// print the name of it's XEvent type.
// Taken from Xlib Programming Manual Vol. One, p. 256-57
//
static char *event_names[] = {
  "",
  "",
  "KeyPress",
  "KeyRelease",
  "ButtonPress",
  "ButtonRelease",
  "MotionNotify",
  "EnterNotify",
  "LeaveNotify",
  "FocusIn",
  "FocusOut",
  "KeymapNotify",
  "Expose",
  "GraphicsExpose",
  "NoExpose",
  "VisibilityNotify",
  "CreateNotify",
  "DestroyNotify",
  "UnmapNotify",
  "MapNotify",
  "MapRequest",
  "ReparentNotify",
  "ConfigureNotify",
  "ConfigureRequest",
  "GravityNotify",
  "ResizeRequest",
  "CirculateNotify",
  "CirculateRequest",
  "PropertyNotify",
  "SelectionClear",
  "SelectionRequest",
  "SelectionNotify",
  "ColormapNotify",
  "ClientMessage",
  "MappingNotify",
};


Bool is_mapnotify(Display * /* dpy */, XEvent *event_return, XPointer xlib_window) {
  switch (event_return->type) {
  case MapNotify:
    // Got MapNotify event, but if multiple windows have been created and
    // Process_Event has not yet been called, then old MapNotify events may
    // still be in the queue.  Check that we get the one for our window.
    if (event_return->xmap.window == *(Window*)xlib_window) {
      return True;
    }
    break;
  default:
    break;
  }
  return False;
}

// // // // // // // // // // // // // // // // // // // //
// Convert an X input event into a Am_Input_Char
// // // // // // // // // // // // // // // // // // // //

int Am_Double_Click_Time = 250;  // in milleseconds

Am_Click_Count Check_Multi_Click(int code, unsigned int state,
				 Am_Button_Down down,
				 Time time, Screen_Desc* screen) {

    Am_Click_Count result = Am_SINGLE_CLICK;

    if (Am_Double_Click_Time) { // else not interested in multi-click
	// if a down press, then check if double click. If up, then use current
	// down count.  If other mouse event, then ignore multi-click
	if ( down == Am_NEITHER) ; // result is OK, do nothing
	else if ( down == Am_BUTTON_UP) { // use current value
	    
	    if (screen->click_counter >= 7) result = Am_MANY_CLICK;
	    else result = (Am_Click_Count) (screen->click_counter + 1);
		// otherwise, just use single click, so result OK
	}
	else { // is a down press
	    if ( (code  == screen->last_click_code) &&
		 (state == screen->last_click_state) &&
		 ( (time - screen->last_click_time) <= Am_Double_Click_Time)){ 
		// is multi-click
		++(screen->click_counter);
	
		if (screen->click_counter >= 7) result = Am_MANY_CLICK;
		else result = (Am_Click_Count) (screen->click_counter + 1);
	    }
	    else screen->click_counter = 0;
    
	    // in either case, set up variables for next time
	    screen->last_click_code = code;
	    screen->last_click_state = state;
	    screen->last_click_time = time;

	}
    }
    return result;
}
	

Am_Input_Char create_input_char_from_code (short code,
					   unsigned int state,
					   Am_Button_Down down,
					   Am_Click_Count mouse) {

    bool shft = false;
    bool ctrl = false;
    bool meta = false;

    if ((state & ShiftMask)) shft = true;

    //only use shift lock for alphabetic characters
    if ((state & LockMask) &&
	(code >= 'a') && (code <= 'z')) shft = true;

    if (state & ControlMask) ctrl = true;
    if (state & Mod1Mask) meta = true;

    Am_Input_Char ic = Am_Input_Char(code, shft, ctrl, meta, down, mouse);

    return ic;
}

// returns character code or 0 if modifier or -1 if illegal
short Map_Sym_To_Code(KeySym sym)
{
  //  cout << "Map_Sym_To_Code sym = " << sym << endl;
    short c;
    switch (sym) {
    case 65470: c = Am_F1; break;
    case 65471: c = Am_F2; break;
    case 65472: c = Am_F3; break;
    case 65473: c = Am_F4; break;
    case 65474: c = Am_F5; break;
    case 65475: c = Am_F6; break;
    case 65476: c = Am_F7; break;
    case 65477: c = Am_F8; break;
    case 65478: c = Am_F9; break;
    case 65479: c = Am_F10; break;

    case 65480: c = Am_L1; break;
    case 65481: c = Am_L2; break;
    case 65482: c = Am_L3; break;
    case 65483: c = Am_L4; break;
    case 65484: c = Am_L5; break;
    case 65485: c = Am_L6; break;
    case 65486: c = Am_L7; break;
    case 65487: c = Am_L8; break;
    case 65488: c = Am_L9; break;
    case 65489: c = Am_L10; break;

    case 65490: c = Am_R1; break;
    case 65491: c = Am_R2; break;
    case 65492: c = Am_R3; break;
    case 65493: c = Am_R4; break;
    case 65494: c = Am_R5; break;
    case 65495: c = Am_R6; break;
    case 65496: c = Am_R7; break;
    case 65498: c = Am_R9; break;
    case 65500: c = Am_R11; break;
    case 65502: c = Am_R13; break;
    case 65504: c = Am_R15; break;

    case 65361: c = Am_LEFT_ARROW; break;
    case 65362: c = Am_UP_ARROW; break;
    case 65364: c = Am_DOWN_ARROW; break;
    case 65363: c = Am_RIGHT_ARROW; break;

    case 65535: c = Am_DELETE; break;
    case 65289: c = Am_TAB; break;
    case 65259: c = Am_TAB; break;
    case 65307: c = Am_ESC; break;
    case 65288: c = Am_BACKSPACE; break;
    case 65293: c = Am_RETURN; break;

    case 65421: c = Am_ENTER; break; // enter on the keypad
    case 65407: c = Am_NUMLOCK; break; // numlock on the keypad

    case 65453: c = (int)'-'; break;  //keypad -
    case 65451: c = (int)'+'; break;  //keypad +
      
      // keysyms not defined submitted by Jerry VanPelt
    case 65454: c = (short)'.'; break; // keypad .
    case 65456: c = (short)'0'; break; // keypad 0
    case 65457: c = (short)'1'; break; // keypad 1
    case 65458: c = (short)'2'; break; // keypad 2
    case 65459: c = (short)'3'; break; // keypad 3
    case 65460: c = (short)'4'; break; // keypad 4
    case 65461: c = (short)'5'; break; // keypad 5
    case 65462: c = (short)'6'; break; // keypad 6
    case 65463: c = (short)'7'; break; // keypad 7
    case 65464: c = (short)'8'; break; // keypad 8
    case 65465: c = (short)'9'; break; // keypad 9


      // end of Jerry VanPelt contributions

    case 32: c = Am_SPACE; break;

    case 97: c = (short)'a'; break;
    case 65: c = (short)'A'; break;
    case 98: c = (short)'b'; break;
    case 66: c = (short)'B'; break;
    case 99: c = (short)'c'; break;
    case 67: c = (short)'C'; break;
    case 100: c = (short)'d'; break;
    case 68: c = (short)'D'; break;
    case 101: c = (short)'e'; break;
    case 69: c = (short)'E'; break;
    case 102: c = (short)'f'; break;
    case 70: c = (short)'F'; break;
    case 103: c = (short)'g'; break;
    case 71: c = (short)'G'; break;
    case 104: c = (short)'h'; break;
    case 72: c = (short)'H'; break;
    case 105: c = (short)'i'; break;
    case 73: c = (short)'I'; break;
    case 106: c = (short)'j'; break;
    case 74: c = (short)'J'; break;
    case 107: c = (short)'k'; break;
    case 75: c = (short)'K'; break;
    case 108: c = (short)'l'; break;
    case 76: c = (short)'L'; break;
    case 109: c = (short)'m'; break;
    case 77: c = (short)'M'; break;
    case 110: c = (short)'n'; break;
    case 78: c = (short)'N'; break;
    case 111: c = (short)'o'; break;
    case 79: c = (short)'O'; break;
    case 112: c = (short)'p'; break;
    case 80: c = (short)'P'; break;
    case 113: c = (short)'q'; break;
    case 81: c = (short)'Q'; break;
    case 114: c = (short)'r'; break;
    case 82: c = (short)'R'; break;
    case 115: c = (short)'s'; break;
    case 83: c = (short)'S'; break;
    case 116: c = (short)'t'; break;
    case 84: c = (short)'T'; break;
    case 117: c = (short)'u'; break;
    case 85: c = (short)'U'; break;
    case 118: c = (short)'v'; break;
    case 86: c = (short)'V'; break;
    case 119: c = (short)'w'; break;
    case 87: c = (short)'W'; break;
    case 120: c = (short)'x'; break;
    case 88: c = (short)'X'; break;
    case 121: c = (short)'y'; break;
    case 89: c = (short)'Y'; break;
    case 122: c = (short)'z'; break;
    case 90: c = (short)'Z'; break;

    case 49: c = (short)'1'; break;
    case 33: c = (short)'!'; break;
    case 50: c = (short)'2'; break;
    case 64: c = (short)'@'; break;
    case 51: c = (short)'3'; break;
    case 35: c = (short)'#'; break;
    case 52: c = (short)'4'; break;
    case 36: c = (short)'$'; break;
    case 53: c = (short)'5'; break;
    case 37: c = (short)'%'; break;
    case 54: c = (short)'6'; break;
    case 94: c = (short)'^'; break;
    case 55: c = (short)'7'; break;
    case 38: c = (short)'&'; break;
    case 56: c = (short)'8'; break;
    case 42: c = (short)'*'; break;
    case 57: c = (short)'9'; break;
    case 40: c = (short)'('; break;
    case 48: c = (short)'0'; break;
    case 41: c = (short)')'; break;

    case 96: c = (short)'`'; break;
    case 126: c = (short)'~'; break;
    case 45: c = (short)'-'; break;
    case 95: c = (short)'_'; break;
    case 61: c = (short)'='; break;
    case 43: c = (short)'+'; break;
    case 91: c = (short)'['; break;
    case 123: c = (short)'{'; break;
    case 93: c = (short)']'; break;
    case 125: c = (short)'}'; break;
    case 92: c = (short)'\\'; break;
    case 124: c = (short)'|'; break;
    case 59: c = (short)';'; break;
    case 58: c = (short)':'; break;
    case 39: c = (short)'\''; break;
    case 34: c = (short)'\"'; break;
    case 44: c = (short)','; break;
    case 60: c = (short)'<'; break;
    case 46: c = (short)'.'; break;
    case 62: c = (short)'>'; break;

    case 47: c = (short)'/'; break;

    case 63: c = (short)'?'; break;

    case 65290: c = Am_LINEFEED; break;

    case 65312: c = Am_COMPOSE_CHARACTER; break;
    case 65376: c = Am_SELECT; break;
    case 65379: c = Am_INSERT_HERE; break;
    case 65383: c = Am_DO_KEY; break;
    case 65384: c = Am_FIND; break;
    case 65386: c = Am_HELP; break;
    case 65425: c = Am_PF1; break;
    case 65426: c = Am_PF2; break;
    case 65427: c = Am_PF3; break;
    case 65428: c = Am_PF4; break;
    case 65452: c = (short)','; break;

    case 268500736: c = Am_REMOVE; break;

    case 65360: c = Am_HOME; break;
    case 65387: c = Am_BREAK; break;
    case 65385: c = Am_CANCEL; break;
    case 65291: c = Am_CLEAR; break;
    case 65378: c = Am_EXECUTE; break;
    case 268500850: c = Am_INSERT_CHAR; break;
    case 268500848: c = Am_INSERT_LINE; break;
    case 268500849: c = Am_DELETE_LINE; break;
    case 268500851: c = Am_DELETE_CHAR; break;
    case 268500847: c = Am_CLEAR_LINE; break;
    case 268500845: c = Am_USER; break;

      // from the sparc 20 running SUNOS:
    case 65365: c = Am_PAGE_UP; break;
    case 65366: c = Am_PAGE_DOWN; break;
    case 65367: c = Am_END; break;
    case 65382: c = Am_L2; break;
    case 268828528: c = Am_L3; break;
    case 65381: c = Am_L4; break;
    case 268828529: c = Am_L5; break;
    case 268828530: c = Am_L6; break;
    case 268828531: c = Am_L7; break;
    case 268828532: c = Am_L8; break;
    case 268828533: c = Am_L10; break;
    case 65299: c = Am_R1; break;
    case 65377: c = Am_R2; break;
    case 65300: c = Am_R3; break;
    case 268828536: c = Am_R4; break;
    case 65455: c = (short)'/'; break; // kp / was Am_R5
    case 65450: c = (short)'*'; break; // kp * was Am_R6
    case 65429: c = Am_HOME; break; // kp home
    case 65469: c = (short)'='; break; // kp =
    case 65431: c = Am_UP_ARROW; break;
    case 65434: c = Am_PAGE_UP; break;
    case 65430: c = Am_LEFT_ARROW; break; 
    case 65432: c = Am_RIGHT_ARROW; break;
    case 65436: c = Am_R13; break; 
    case 65433: c = Am_DOWN_ARROW; break;
    case 65435: c = Am_R15; break; 
    case 65438: c = Am_INSERT_HERE; break;
    case 65439: c = Am_DELETE_CHAR; break;
    case 65406: c = Am_ALT_GRAPH; break;

      
    default:
	if ( (sym >= 65505) && (sym <= 65518) ) {
	    // symbol is a modifier key
	    c = 0;
	}
	else { // unknown character
	    c = -1;
	}
	break;
    } // switch
    return c;
}
    
						    
Am_Input_Char create_input_char_from_key (unsigned int keycode,
					  unsigned int state,
					  Display* disp) {
    int index;
   
    if (state & ShiftMask) index = 1;
    else index = 0;
    
    KeySym sym = XKeycodeToKeysym(disp, keycode, index);

    if ( (sym == NoSymbol) && (index == 1) ) {
	// try again with unshifted index
	// This makes SHIFT-F1 etc. work on Sun
	sym = XKeycodeToKeysym(disp, keycode, 0);
    }
    
    short code;

    
    if (sym == NoSymbol) code = 0;
    else code = Map_Sym_To_Code(sym);
    
    
    if (code == -1) {// try again with unshifted symbol
	// This makes SHIFT-R7 etc. work on Sun
	KeySym second_sym = XKeycodeToKeysym(disp, keycode, 0);
	code = Map_Sym_To_Code(second_sym);
    }
	   
    if (code > 0) {
	// only support keyboard keys going down
	return create_input_char_from_code (code, state, Am_NEITHER,
					     Am_NOT_MOUSE);
    }
    else { // in case is an illegal character or modifier
	if (code < 0)
	    cout << "** Unknown keyboard symbol " << sym << " ignored\n"
		 << flush;
	return Am_Input_Char(); // null means illegal
    }
}

Am_Input_Char create_input_char_from_mouse (unsigned int button,
					    unsigned int state,
					    Am_Button_Down down,
					    Time time, Screen_Desc* screen)
{
    int code;
    if (button == Button1) code = Am_LEFT_MOUSE;
    else if (button == Button2) code = Am_MIDDLE_MOUSE;
    else if (button == Button3) code = Am_RIGHT_MOUSE;
    else {
      cerr << "** Unknown mouse button " << button << "." << endl;
      Am_Error ();
    }

    Am_Click_Count cnt = Check_Multi_Click(code, state, down, time, screen);

    return create_input_char_from_code (code, state, down, cnt);
}

// // // // // // // // // // // // // // // // // // // //
// Main Input Event Handler
// // // // // // // // // // // // // // // // // // // //


void set_input_event (Am_Input_Event *ev, Am_Input_Char ic, int x, int y,
		      unsigned long time, Am_Drawonable *draw) {
    ev->x = x;
    ev->y = y;
    ev->input_char = ic;
    ev->time_stamp = time;
    ev->draw = draw;
    if (Am_Debug_Print_Input_Events)
	cout << "\n<><><><><> " << ic << " x=" << x << " y=" << y <<
	    " time=" << time << " drawonable=" << draw << endl;
}

bool exit_if_stop_char(Am_Input_Char ic)
{ 
  if (ic == Am_Stop_Character) {
//    cerr << "Got stop event: exiting Amulet main loop." << endl;
    Am_Main_Loop_Go = false;
    return true;
  }
  else return false;
}

void handle_selection_request (XEvent& ev, Am_Drawonable_Impl* draw) {
  // someone wants our selection.
  XEvent notify;
  Atom p;
  if (draw->screen->cut_data == NULL)
    p = None;
  else {
    p = ev.xselectionrequest.property;
    XChangeProperty(ev.xselectionrequest.display, //draw->screen->display, 
		    ev.xselectionrequest.requestor,
		    ev.xselectionrequest.property,
		    XA_STRING, 8, PropModeReplace, 
		    (unsigned char*)draw->screen->cut_data, 
		    strlen(draw->screen->cut_data));
  }
  notify.xany.type = SelectionNotify;
  notify.xselection.display = 
    ev.xselectionrequest.display;
  notify.xselection.send_event = True;
  notify.xselection.requestor = 
    ev.xselectionrequest.requestor;
  notify.xselection.selection = 
    ev.xselectionrequest.selection;
  notify.xselection.target = XA_STRING;
  notify.xselection.property = p;
  notify.xselection.time = 
    ev.xselectionrequest.time;
  
  int error = XSendEvent(draw->screen->display, 
			 ev.xselectionrequest.requestor, 
			 True, 0, &notify);
  cerr << "send event result = " << error << endl;
  
  XSync(draw->screen->display, False);
  // I think we're possibly leaving the property we changed hanging.
}

Bool selection_event(Display* /* d */, XEvent *ev, char*)
{
  if ((ev->xany.type == SelectionNotify) || ev->xany.type == SelectionRequest)
    return True;
  else return False;
}

char* Am_Drawonable_Impl::Get_Cut_Buffer()
{
  if (offscreen) return NULL; // not meaningful for offscreen bitmaps

  // To get a selection properly, we need to send off a SelectionRequest event 
  // (this is done with the call to XConvertSelection), and then wait for a
  // SelectionNotify event.  Since we need to finish this routine synchronously
  // instead of waiting for a SelectionNotify event, we'll just wait until
  // any one event comes in, and if it's a SelectionNotify we'll take it
  // and get the selection, otherwise we'll put it back and pretend the request
  // never happened.  This is a gross hack, but it seems to work fairly well
  // when running on my machine, everything locally.  Probably it will fail more
  // if the selection is on a different machine than the Amulet program.

  if (this == screen->root) { // then we're in a root window
    cerr << "** Gem warning: Get_Cut_Buffer() won't work in a root window.\n";
    return NULL;
  }
  // Make an atom to get the selection in.

  // Request the selection from the X server
  XConvertSelection(screen->display, XA_PRIMARY, XA_STRING, 
		    screen->cut_buffer, xlib_drawable, CurrentTime);
  XSync(screen->display, False);
  // Wait for an event, handling selection requests.
  // if it's a selection notify, get the selection
  XEvent ev;
  while (true) {
    while (!XPending (screen->display));
    //    XNextEvent(screen->display, &ev);
    if (XCheckIfEvent(screen->display, &ev, selection_event, NULL) == False
	//	&& XCheckIfEvent(screen->display, &ev, selection_event, NULL) == False)
	) {
	// if we get here, we didn't get any selection notify event back.
	// should we use cut buffer0 here instead?
	cerr << "** missing selection notify event." << endl;
	//	XPutBackEvent(screen->display, &ev); // checkifevent doesn't dequeue nonmatching events
	return NULL;
      }
    if (ev.xany.type == SelectionNotify) break;
    //    if (ev.xany.type == SelectionRequest) {
    Am_Drawonable_Impl *draw =
      Get_Drawable_Backpointer(ev.xany.display,
			       ev.xany.window);
    if (draw) handle_selection_request(ev, draw);
    //    else handle_selection_request(ev, screen->root);
    //    }
  }
  // if we get here, then we have a selection
  if (ev.xselection.property == None)
    { // then there is no selection value, use cut buffer 0 instead.
      cerr << "** No primary selection, using cut buffer." << endl;
      int n;
      char* tempstr = XFetchBytes(screen->display, &n);
      char* str = new char[n+1];
      strcpy (str, tempstr);
      XFree(tempstr);
    }
  else { // we have a selection value
    // Get the property's value.  I got this from an email from
    // "S.Ramakrishnan" <ramakris@vtopus.cs.vt.edu> to bam@cs.cmu.edu
    char str[200];
    char *buff;
    int actual_format; 
    unsigned long num_items_ret, rest;
    Atom actual_type;
    long begin = 0;
    for (*str = 0, rest = 1; rest; strcat(str, buff), begin++, XFree(buff)) {
      XGetWindowProperty(screen->display, xlib_drawable, ev.xselection.property,
			 begin, 1, True, AnyPropertyType, &actual_type,
			 &actual_format, &num_items_ret, &rest, (unsigned char**)&buff);
      if (!buff) break;
    }
    buff = new char[strlen(str) + 1];
    strcpy (buff, str);
    return buff;
  }
}


void Handle_Event_Received (XEvent& event_return) {
    
    Am_Drawonable_Impl *draw =
      Get_Drawable_Backpointer(event_return.xany.display,
			       event_return.xany.window);
    if (!draw)
      return;
    Am_Input_Event_Handlers *evh = draw->event_handlers;
    Am_Input_Char ic;

    if (!evh) {
	if (Am_Debug_Print_Input_Events)
	    cout << "<> Input ignored for " << draw <<
		" because no Event_Handler\n";
	return;
    }
    
    switch (event_return.xany.type) {
    case KeyPress:
	ic = create_input_char_from_key (event_return.xkey.keycode,
					 event_return.xkey.state,
					 draw->screen->display);
	if (ic.code != 0) { // then is a legal code
	    if (exit_if_stop_char(ic)) return;
	    set_input_event (Am_Current_Input_Event, ic, event_return.xkey.x,
			 event_return.xkey.y, event_return.xkey.time,
			 draw);
	    draw->event_handlers->Input_Event_Notify(draw,
						     Am_Current_Input_Event);
	}
	break;
    case ButtonPress:
	ic = create_input_char_from_mouse (event_return.xbutton.button,
					   event_return.xbutton.state,
					   Am_BUTTON_DOWN,
					   event_return.xbutton.time,
					   draw->screen);
	if (exit_if_stop_char(ic)) return;
	set_input_event (Am_Current_Input_Event, ic, event_return.xbutton.x,
			 event_return.xbutton.y, event_return.xbutton.time,
			 draw);
	draw->event_handlers->Input_Event_Notify(draw, Am_Current_Input_Event);
	break;
    case ButtonRelease:
	ic = create_input_char_from_mouse (event_return.xbutton.button,
					   event_return.xbutton.state,
					   Am_BUTTON_UP,
					   event_return.xbutton.time,
					   draw->screen);
	if (exit_if_stop_char(ic)) return;
	set_input_event (Am_Current_Input_Event, ic, event_return.xbutton.x,
			 event_return.xbutton.y, event_return.xbutton.time,
			 draw);
	draw->event_handlers->Input_Event_Notify(draw, Am_Current_Input_Event);
	break;
    case MotionNotify:
	ic = create_input_char_from_code (Am_MOUSE_MOVED,
					   event_return.xmotion.state,
					   Am_NEITHER, Am_SINGLE_CLICK);
	if (exit_if_stop_char(ic)) return;
	set_input_event (Am_Current_Input_Event, ic, event_return.xmotion.x,
			 event_return.xmotion.y, event_return.xmotion.time,
			 draw);
	draw->event_handlers->Input_Event_Notify(draw, Am_Current_Input_Event);
	break;
    case EnterNotify:
	ic = create_input_char_from_code (Am_MOUSE_ENTER_WINDOW,
					   event_return.xcrossing.state,
					   Am_NEITHER, Am_SINGLE_CLICK);
	if (exit_if_stop_char(ic)) return;
	set_input_event (Am_Current_Input_Event, ic, event_return.xcrossing.x,
			 event_return.xcrossing.y,
			 event_return.xcrossing.time, draw);
	draw->event_handlers->Input_Event_Notify(draw, Am_Current_Input_Event);
	break;
    case LeaveNotify:
	ic = create_input_char_from_code (Am_MOUSE_LEAVE_WINDOW,
					   event_return.xcrossing.state,
					   Am_NEITHER, Am_SINGLE_CLICK);
	if (exit_if_stop_char(ic)) return;
	set_input_event (Am_Current_Input_Event, ic, event_return.xcrossing.x,
			 event_return.xcrossing.y,
			 event_return.xcrossing.time, draw);
	draw->event_handlers->Input_Event_Notify(draw, Am_Current_Input_Event);
	break;
    case Expose:
	if (Am_Debug_Print_Input_Events)
	    cout << "<> Exposure Event, x=" << event_return.xexpose.x
		 << " y=" << event_return.xexpose.y <<
		" width=" << event_return.xexpose.width <<
		" height=" << event_return.xexpose.height <<
		" drawonable=" << draw << endl;
	draw->event_handlers->Exposure_Notify(draw,
//					      event_return.xexpose.count,
					      event_return.xexpose.x,
					      event_return.xexpose.y,
					      event_return.xexpose.width,
					      event_return.xexpose.height);
	break;
    case DestroyNotify:
	if (Am_Debug_Print_Input_Events)
	    cout << "<> DestroyNotify, drawonable=" << draw << endl;
	draw->event_handlers->Destroy_Notify(draw);
	break;
//// BUG: Not used
//    case UnmapNotify:
//	if (Am_Debug_Print_Input_Events)
//	    cout << "<> UnmapNotify, drawonable=" << draw << endl;
//	draw->event_handlers->Unmap_Notify(draw);
//	break;
//    case ReparentNotify:
//	if (Am_Debug_Print_Input_Events)
//	    cout << "<> ReparentNotify, drawonable=" << draw << endl;
//	draw->event_handlers->Reparent_Notify(draw);
//      break;
//    case MapNotify:
//	if (Am_Debug_Print_Input_Events)
//	    cout << "<> MapNotify, drawonable=" << draw << endl;
//	draw->event_handlers->Map_Notify(draw);
//	break;
    case ConfigureNotify:
	// Configure events may come in for windows that haven't really
	// changed size or position, and in this case the events are ignored
	// (i.e., Gem does not pass them on to the user's event-handler).
	// For example, a 'valid' configure event may be percieved and
	// dispatched that causes the user's ConfigureNotify Event Handler
	// to set the size of the window, which in turn generates a 'bogus'
	// configure event.  When the bogus event is percieved here, we
	// detect it by noticing that its values are equal to those already
	// installed in the drawonable, and throw it away.
	//
        // A Good ConfigureNotify handler should respond to the event
	// not by moving the window again, but by just setting its internal
	// state to correspond to where the window was moved to.  The window
	// manager doesn't have to put the window where you request, or make
	// it the correct size.  It tells you where it put it through a
	// ConfigureNotify event.  If you try to force the WM to put it
	// where you want by looking at the configureNotify event to find
	// out where it actually put it, you might get walking windows
	// no matter what you do.
	//
	// Note: Gem specifications dictate that the "left" and "top" of a
	// window is the left and top of its frame (not necessarily its
	// drawable area).  The "width" and "height" of a window is the
	// width and height of its drawable area.
	// Also, things drawn inside the window have their coordinates
	// relative to the drawable area's left and top.
	//
	int x, y, w, h, left, top, width, height, outer_left, outer_top,
	  lb, tb, rb, bb;

	// We need to use inquire_window_borders to calculate the correct
	// x and y values, because X windows returns local coordinates
	// in the event, not global coordinates, so they're always (0,0).
	
	if (draw->Inquire_Window_Borders (lb, tb, rb, bb,
					  outer_left, outer_top)) {
	  x = outer_left;
	  y = outer_top;
	  w = event_return.xconfigure.width;
	  h = event_return.xconfigure.height;
	  draw->Get_Position (left, top);
	  draw->Get_Size (width, height);
	  //	  cout << "Configure notify: " << (void*)draw <<  " " << w << " " << h << " " << event_return.xany.window << endl;
	  
	  // Only generate a Gem configure-notify event if something changed
	  if ((x != left) || (y != top) || (w != width) || (h != height)) {
	    if (Am_Debug_Print_Input_Events)
	      cout << "<> Configure Notify, x=" << x << " y=" << y <<
		" width=" << width << " height=" << height <<
		" drawonable=" << draw << endl;
	    draw->reconfigure (x, y, w, h);
	    draw->event_handlers->Configure_Notify(draw, x, y, w, h);
	  }
	}
	break;
    case ClientMessage:
      // this is where we get window destroy messages 
      // from the window manager.
      {
	// this is speed-inefficient, but it never happens, so who cares.
	Atom wm_delete_window = XInternAtom(event_return.xclient.display, 
					    "WM_DELETE_WINDOW", False);
	if (event_return.xclient.data.l[0] != wm_delete_window) break; 
	// not a delete window client message.
	draw->event_handlers->Destroy_Notify(draw);
      }
      break;
    case SelectionClear: 
      // We get this event when we're forced to release our selection.
      // We'll ignore it for now.
      break;
    case SelectionRequest:
      handle_selection_request(event_return, draw);
      break;
      // next all the events we don't expect to occur
    case KeyRelease:
    case FocusIn:
    case FocusOut:
    case KeymapNotify:
    case VisibilityNotify:
    case MapRequest:
    case ConfigureRequest:
    case GravityNotify:
    case ResizeRequest:
    case CirculateNotify:
    case CirculateRequest:
    case PropertyNotify:
    case ColormapNotify:
    case MappingNotify:
	cout << "** Received event of unexpected type: " <<
	    event_names[event_return.type] << endl;
    case CreateNotify:
    case GraphicsExpose:
    case NoExpose:
    case SelectionNotify: // we get this here if get_cut_buffer misses it.
      
	// we do get these events unfortunately, so silently ignore them.
	break;
    } // end switch
}	


Bool is_input_event (XEvent& event) {
  switch (event.xany.type) {
  case ButtonPress:
  case ButtonRelease:
  case KeyPress:
  case MotionNotify:
  case EnterNotify:
  case LeaveNotify:
    return true;
  default:
    return false;
  }
}

/*
 *  Flush_Extra_Move_Events:  If parameter is a mouse move event, then throw
 *				away all other contiguous move events until
 * 				the last one, and leave event_return filled
 *				with the last one.  Disp_ version for multi-
 *				screen, and UseX_ for not
 */
void Disp_Flush_Extra_Move_Events(XEvent& event_return) {
  int cnt = 0;
  if (event_return.xany.type == MotionNotify) {
    XEvent next_event_return;
    while (Scrn_Mgr.Pending (&next_event_return)) {
      if (next_event_return.xany.type == MotionNotify) {
	event_return = next_event_return;
	cnt ++;
      }
      else {
	Scrn_Mgr.Put_Event_Back (next_event_return);
	break;
      }
    }
  }
  if (Am_Debug_Print_Input_Events && cnt > 0)
    cout << "<> Multi Ignoring " << cnt << " move events\n" << flush;
}

void UseX_Flush_Extra_Move_Events(XEvent& event_return) {
  int cnt = 0;
  if (event_return.xany.type == MotionNotify) {
    XEvent next_event_return;
    while (XPending (Main_Display)) {
      XNextEvent (Main_Display, &next_event_return);
      if (next_event_return.xany.type == MotionNotify) {
	event_return = next_event_return;
	cnt ++;
      }
      else {
	XPutBackEvent(Main_Display, &next_event_return);
	break;
      }
    }
  }
  if (Am_Debug_Print_Input_Events && cnt > 0)
    cout << "<> Ignoring " << cnt << " move events\n" << flush;
}



/*
 *  Process_Immediate_Event:  Does not wait for an event, but processes
 *                            the first event in the queue and all non-input
 *                            events after it until an input event is seen.
 *                            The function returns when it encounters an input
 *                            event (excluding the case where the first event
 *                            is an input event) or when the queue is empty.
 */
void Am_Drawonable::Process_Immediate_Event ()
{
  XEvent event_return;
  
  if (More_Than_One_Display) {
    while (Scrn_Mgr.Pending (&event_return)) {
      Disp_Flush_Extra_Move_Events(event_return);
      Handle_Event_Received (event_return);
      // If that was an input event, then process all the remaining
      // non-input events (and don't process another input event).
      if (is_input_event (event_return)) {
        while (Scrn_Mgr.Pending (&event_return)) {
          if (is_input_event (event_return)) {
            Scrn_Mgr.Put_Event_Back (event_return);
            return;
          }
          else
            Handle_Event_Received (event_return);
        }
	return;
      }
    }
  }
  else {
    while (XPending (Main_Display)) {
      XNextEvent (Main_Display, &event_return);
      UseX_Flush_Extra_Move_Events(event_return);
      Handle_Event_Received (event_return);
      // If that was an input event, then process all the remaining
      // non-input events (and don't process another input event).
      if (is_input_event (event_return)) {
        while (XPending (Main_Display)) {
	  XNextEvent (Main_Display, &event_return);
	  if (is_input_event (event_return)) {
	    XPutBackEvent(Main_Display, &event_return);
	    return;
	  }
	  else
	    Handle_Event_Received (event_return);
	}
	return;
      }
    }
  }
}
  

/*
 *  Process_Event:  waits for the next event, and processes exactly one
 *                  input event and all non-input events before and after
 *                  that input event before returning.  For example
 *                          before            after
 *                       xxxIyyyIzzz   --->   Izzz
 *                  The function returns when it encounters a second input
 *                  event or when the queue is empty.
 */
void Am_Drawonable::Process_Event ()
{
  XEvent event_return;

  if (More_Than_One_Display) {
    Scrn_Mgr.Next_Event (&event_return);
    Disp_Flush_Extra_Move_Events(event_return); 
    Handle_Event_Received (event_return);
    // If that was not an input event, then process all the remaining
    // non-input events until we have processed an input event.
    if (!is_input_event (event_return)) {
      while (Scrn_Mgr.Pending (&event_return)) {
	Handle_Event_Received (event_return);
	if (is_input_event (event_return))
	  break;
      }
    }
    // Process all remaining non-input events
    while (Scrn_Mgr.Pending (&event_return)) {
      if (is_input_event (event_return)) {
	Scrn_Mgr.Put_Event_Back (event_return);
	return;
      }
      else
	Handle_Event_Received (event_return);
    }
  }
  else {
    XNextEvent (Main_Display, &event_return);
    UseX_Flush_Extra_Move_Events(event_return); 
    Handle_Event_Received (event_return);
    // If that was not an input event, then process all the remaining
    // non-input events until we have processed an input event.
    if (!is_input_event (event_return)) {
      while (XPending (Main_Display)) {
	XNextEvent (Main_Display, &event_return);
	Handle_Event_Received (event_return);
	if (is_input_event (event_return))
	  break;
      }
    }
    // Process all remaining non-input events
    while (XPending (Main_Display)) {
      XNextEvent (Main_Display, &event_return);
      if (is_input_event (event_return)) {
	XPutBackEvent(Main_Display, &event_return);
	return;
      }
      else
	Handle_Event_Received (event_return);
    }
  }
}

/*
 *  Wait_For_Event: waits until the event queue is not empty.  Will return
 *                  immediately if queue is already not empty.  Does not
 *                  process anything
 */
void Am_Drawonable::Wait_For_Event ()
{
  if (More_Than_One_Display)
    Scrn_Mgr.Wait_For_Event ();
  else {
    XEvent event_return;
    XPeekEvent (Main_Display, &event_return);
  }
}

void Am_Drawonable::Main_Loop ()
{
  while (Am_Main_Loop_Go)
    Process_Event ();
}


// // // // // // // // // // // // // // // // // // // //
// Am_Drawonable member functions
// // // // // // // // // // // // // // // // // // // //

void Am_Drawonable_Impl::Initialize_Event_Mask ()
{
    want_enter_leave = false;
    want_multi_window = false;
    want_move = false;
    current_event_mask = ( ButtonPressMask     | ButtonReleaseMask |
			   KeyPressMask        | ExposureMask      |
			   StructureNotifyMask );
    // current_event_mask can be now passed to XCreateWindow as part
    // of the XSetWindowAttributes
}
	
void Am_Drawonable_Impl::set_drawable_event_mask ()
{
  if (want_move) {
    unsigned int pointer_active_mask;

    current_event_mask = // *report-motion-em*
	  ExposureMask | PointerMotionMask |
	  ButtonPressMask | ButtonReleaseMask | 
	  KeyPressMask | StructureNotifyMask;
    pointer_active_mask = // *report-motion-pem*
	  (unsigned int)
	  (ButtonPressMask | ButtonReleaseMask | PointerMotionMask);

    if (want_enter_leave) { // add enter leave masks
      current_event_mask  |= EnterWindowMask | LeaveWindowMask;
      pointer_active_mask |=
	    (unsigned int)(EnterWindowMask | LeaveWindowMask);
    }
    if (want_multi_window) { // add owner-grab-button
      current_event_mask |= OwnerGrabButtonMask;
	  // don't use OwnerGrabButtonMask in the p_a_m mask for grab
    }
      
    // this will change an active grab if one is in process because
    // changing the window's event mask will have no effect if there is
    // an active grab already.  Active grabs happen whenever
    // the mouse button is pressed down.
    XChangeActivePointerGrab (screen->display, pointer_active_mask,
                              None, CurrentTime);
  }
  else {  // don't want motion events, don't need a pointer mask
    current_event_mask = // *ignore-motion-em*
	ExposureMask | ButtonPressMask | ButtonReleaseMask | 
	KeyPressMask | StructureNotifyMask;

    if (want_enter_leave) 
      current_event_mask |= EnterWindowMask | LeaveWindowMask;
    if (want_multi_window) current_event_mask |= OwnerGrabButtonMask;
  }

  if (Am_Debug_Print_Input_Events)
    cout << "Changing Event Mask to " << current_event_mask << " for "
	 << this << endl;

  // now call X to install event mask
  XSelectInput(screen->display, xlib_drawable, current_event_mask);

  // After the XSelectInput, you need to force-output because when
  // using the backgroup m-e-l process, otherwise this doesn't get noticed.
  // *no m-e-l yet* this->Flush_Output();

  // Ignore NoExpose and GraphicsExpose events (double buffering)
  XSetGraphicsExposures(screen->display, screen->gc, false);
}

void Am_Drawonable_Impl::Set_Enter_Leave ( bool want_enter_leave_events )
{
   if (want_enter_leave_events != want_enter_leave) {
	want_enter_leave = want_enter_leave_events;
	this->set_drawable_event_mask ();
    }
}

void Am_Drawonable_Impl::Set_Want_Move ( bool want_move_events )
{
    if (want_move != want_move_events) { // then changing
	want_move = want_move_events;
	this->set_drawable_event_mask ();
    }
}

void Am_Drawonable_Impl::Set_Multi_Window ( bool want_multi )
{
    if (want_multi != want_multi_window) { // then changing
	want_multi_window = want_multi;
	this->set_drawable_event_mask ();
    }
}

void Am_Drawonable_Impl::Discard_Pending_Events()
{
    // * NIY *
    cout << "** Discarding pending input events NIY \n";
}

void Am_Drawonable_Impl::Set_Input_Dispatch_Functions
          (Am_Input_Event_Handlers* evh)
{
  event_handlers = evh;
}

void Am_Drawonable_Impl::Get_Input_Dispatch_Functions
          (Am_Input_Event_Handlers*& evh)
{
  evh = event_handlers;
}

//Find the child-most drawonable at the current cursor position
Am_Drawonable* Am_Drawonable_Impl::Get_Drawonable_At_Cursor() {
  Display* display = screen->display;
  Window parent, root_return, child_return;
  int root_x_return, root_y_return, win_x_return, win_y_return;
  unsigned int mask_return;
  parent = screen->root->xlib_drawable;
  Am_Drawonable_Impl *return_draw = NULL;
  Am_Drawonable_Impl *draw = NULL;
  // need to find leaf-most window that contains the cursor, keep
  // going down until fail
  while(true) {
    if(!XQueryPointer (display, parent, &root_return, &child_return,
		       &root_x_return, &root_y_return, &win_x_return,
		       &win_y_return, &mask_return))
      return NULL; //if XQueryPointer returns false, then not on right screen
    if (child_return) {
      if (child_return == parent) {
	// looping, return last good drawonable
	return return_draw;
      }
      else draw = Get_Drawable_Backpointer(display, child_return);
    }
    else return return_draw; //no child found, return last good drawonable
    if (draw) { //save in case this is the last good one
      return_draw = draw;
    }
    //have a child, loop to see if can find a child of that child
    parent = child_return;
  }
}
 
// This function is called during window creation, and when a window changes
// from invisible to visible.  If you do not wait for the MapNotify event
// before drawing, then your window may come up blank.
//

void wait_for_mapnotify (Display *dpy, Window *xlib_window) {
  XEvent event_return;

  while(1) {
    XPeekIfEvent(dpy, &event_return, is_mapnotify, (XPointer)xlib_window);
    // XPeekIfEvent may have returned without setting event_return (in other
    // words, is_mapnotify returned False).
    if (event_return.type == MapNotify)
      return;
  }
}
