// $Id: DlgObject.h,v 1.5 1997/11/02 23:00:33 jharris Exp $

#ifndef DlgObject_included
#define DlgObject_included

#include <qobject.h>
#include <qlist.h>
#include <qpopmenu.h>
#include <qintdict.h>

class DlgParser;
class ProjectWnd;
class Project;

enum DlgObjectType {
    DlgObject_Dialog,
    DlgObject_Project,
    DlgObject_Pixmap
};

class DlgObject : public QObject
{
    Q_OBJECT

public: // methods
    virtual QPopupMenu &GetListPopup();
    // Return popupmenu for DialogList

    virtual ~DlgObject();
    // dtor, destroys at least our lists and popups

    virtual QString GetType() = 0;
    virtual DlgObjectType GetObjType() = 0;
    // Retrieves Object Type

    QString GetExtension();
    // Retrieves Filename extension

    static QString GetExtension( DlgObjectType type );
    // Get the extension for the object type

    const QString &GetName();
    virtual void SetName( const QString &, bool = TRUE );
    // get and set the filename

    static DlgObject *NewObject( ProjectWnd &, const QString & );
    // Build up a new object

    virtual void GetGeneratedFiles( QString &, QString &, QString & );
    // Get back the generated Sources, Headers and Headers which contains
    // Metastructure and need to be compiled with moc
    // Is needed for Makefilegeneration.

    QList<DlgObject> &GetProjects();
  // Returns a list of projects to which this obj belongs
  // List length can be retrieved by InProject()

public slots:
     virtual bool Load() = 0;
    // Load from disk

    virtual bool Save() = 0;
    // Save to disk

    virtual bool SaveAs();
    // Save to disk, different filename

    virtual bool Generate( bool ) = 0;
    virtual bool Generate();
    // Generate c++ files (the object must be shown/loaded or not)

    virtual void Hide() = 0;
    // Hide from screen

    virtual void Show() = 0;
    // Show (again)

    virtual void Raise() = 0;
    // Raise Window

    virtual bool EnterProject( const Project * );
    // Added in a project, remember this project

    virtual bool LeaveProject( const Project * );
    // Removed from a Project, forget this project

    virtual bool Remove( );
    // Remove from a Project

    int InProject( const Project * = NULL );
    // Is in (specified) Project

    virtual void CPAvailable( bool ) = 0;
    // Somehow c&p data was made available

    signals:
    void Changed( DlgObject * );
    // DlgObject has changed, to be connected to DialogList

    void Closed( DlgObject * );
    // DlgObject was closed

    void Saved( DlgObject * );
    // Emitted after being saved

    void CPDataSet( DlgObject * );
    // Emitted after the c&p buffer was set by this object

protected: // Attributes
    ProjectWnd &itsProjectWnd;
    // Superclass

    DlgParser *itsParser;
    // Parser, used to read input files

    QList<DlgObject> itsProjects;
    // The list of loaded projects to which we belong

    QPopupMenu *unboundPopup;
    // Popup when not inside any project

    QPopupMenu *unboundHiddenPopup;
    // Popup when not inside any project, but hidden from screen

    QPopupMenu *boundPopup;
    // Popup when inside a project

    QPopupMenu *boundHiddenPopup;
    // Popup when inside a project, but hidden from screen

    QString itsFilename;
    // Filename to load/save

    enum Status {
	DlgObject_Init,
	DlgObject_Loaded,
	DlgObject_Saved,
	DlgObject_Generated,
	DlgObject_Shown,
	DlgObject_Hidden,
	DlgObject_Changed
    };

    Status status;

protected: // Methods
    DlgObject( ProjectWnd &, const QString &name );
    // Ctor, name is passed further down to Qt


private: // Attributes

    static QIntDict<char>	itsExtensionDict;
    // dictionary of file extensions indexed by object type
};

typedef QList<DlgObject> DlgObjList;

inline const QString &DlgObject::GetName()
{
    return itsFilename;
}


inline QString DlgObject::GetExtension()
{
    return GetExtension( GetObjType() );
}


inline void DlgObject::SetName( const QString &newName, bool change )
{
    itsFilename = newName;
    status = DlgObject_Changed;
    if( change )
	emit Changed( this );
}

inline void DlgObject::GetGeneratedFiles( QString &src, 
                                          QString &hdr, 
                                          QString &mhdr )
{
    // We haven't got anything 
    src = "";
    hdr = "";
    mhdr = "";
}

inline bool DlgObject::Generate()
{
    return Generate( FALSE );
}

inline QList<DlgObject> &DlgObject::GetProjects()
{
  return itsProjects;
}

#endif
