/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,   
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

#ifndef TGadget
#define TGadget TGadget

#include <toad/toad.hh>
#include <toad/gadgetwindow.hh>
#include <toad/io/serializable.hh>
#include <vector>

class TGadgetEditor;

class TGadget:
	public TSerializable
{
	public:
		TGadget();
		virtual ~TGadget();
		enum EPaintType {
			NORMAL,
			SELECT,
			EDIT
		};
		
		//. Called to paint the gadget.
		virtual void paint(TPen& pen, EPaintType) = 0;
		virtual void paintSelection(TPen &pen);
		//. Called to get the gadgets bounding rectangle.
		virtual void getShape(TRectangle&) = 0;

		TRGB line_color;
		TRGB fill_color;
		bool filled:1;				// true when filled
		bool removeable:1;		// true when object is removeable (for gadgeteditor)
	
		// editor related stuff per gadget for manipulation
		//-------------------------------------------------
		static const unsigned NOTHING  = 0;
		static const unsigned CONTINUE = 1;	// continue editing
		static const unsigned STOP     = 2;	// stop editing
		static const unsigned REPEAT   = 4;	// repeat the last event
		static const unsigned DELETE   = 8;	// delete this object

		// stage 1: select:
		virtual double distance(int x, int y) = 0;
		
		// stage 2: move
		virtual void translate(int dx, int dy) = 0 ;
		
		// stage 3: manipulate
		static const int NO_HANDLE = -1;
		virtual bool getHandle(unsigned n, TPoint &p);
		virtual void translateHandle(unsigned handle, int x, int y);

		// stage 4: in place editing
		//. Return `true' when in-place editing is desired.
		virtual bool startInPlace() { return false; }
		virtual unsigned stop(TGadgetEditor*) { return NOTHING; };
		virtual unsigned keyDown(TGadgetEditor*, TKey, char*, unsigned) { return CONTINUE; }

		// editor related stuff for manipulation & creation
		//--------------------------------------------------
		virtual void startCreate() {};
		virtual unsigned mouseLDown(TGadgetEditor*, int, int, unsigned) { return STOP; }
		virtual unsigned mouseMove(TGadgetEditor*, int, int, unsigned) { return CONTINUE; }
		virtual unsigned mouseLUp(TGadgetEditor*, int, int, unsigned) { return CONTINUE; }

		// editor related stuff for all gadgets
		//--------------------------------------
		static double Distance2Line(int x, int y, int x1, int y1, int x2, int y2);

		static const double OUT_OF_RANGE = 100.0;
		static const double RANGE = 5.0;
		static const double INSIDE = 2.5;

		// storage stuff
		//--------------------------------------
		void store(TOutObjectStream&, ulong);
		void restore(TInObjectStream&, ulong);

		// storage stuff for all gadgets
		//-------------------------------------- 
		static void InitStorage();
		static void LoseStorage();
		static void Store(TOutObjectStream &stream, TGadget *gadget)
      { serialize.Store(stream, gadget); }
		static TGadget* Restore(TInObjectStream &stream)
			{ return (TGadget*)serialize.Restore(stream); }
		static TSerializeBase serialize;
};

class TGRectangle:
	public TGadget
{
		typedef TGadget super;
	public:
		TGRectangle() {
			filled = false;
			line_color.Set(0,0,0);
			fill_color.Set(0,0,0);
		}
		TGRectangle(TGadgetWindow *parent, int x,int y,int w, int h) {
			parent->gadgets.push_back(this);
			SetShape(x, y, w, h);
			filled = false;
		};
		void SetShape(int x, int y, int w, int h) {
			p1.x = x;
			p1.y = y;
			p2.x = x+w-1;
			p2.y = y+h-1;
		}
		void paint(TPen &, EPaintType);
		void getShape(TRectangle&);

		double distance(int x, int y);
		void translate(int dx, int dy);
		bool getHandle(unsigned n, TPoint &p);
		void translateHandle(unsigned handle, int mx, int my);

		unsigned mouseLDown(TGadgetEditor*, int, int, unsigned);
		unsigned mouseMove(TGadgetEditor*, int, int, unsigned);
		unsigned mouseLUp(TGadgetEditor*, int, int, unsigned);
		
		TCloneable* clone() { return new TGRectangle(*this); }

		void store(TOutObjectStream&, ulong);
		void restore(TInObjectStream&, ulong);
		
		TPoint p1, p2;
};

class TGCircle:
	public TGRectangle
{
	public:
		TGCircle(){}
		TGCircle(TGadgetWindow *parent, int x, int y, int w, int h):
			TGRectangle(parent,x,y,w,h) {}
		void paint(TPen &, EPaintType);
		
		double distance(int x, int y);
		
		TCloneable* clone() { return new TGCircle(*this); }
};

class TGText:
	public TGRectangle
{
		typedef TGRectangle super;
	public:
		TGText() {}
		TGText(TGadgetWindow *parent, int x,int y, const string &text) {
			parent->gadgets.push_back(this);
			p1.x = x;
			p1.y = y;
			this->text = text;
			CalcSize();
		}
		void SetText(const string &t) {
			text = t;
			CalcSize();
		}
		void paint(TPen &, EPaintType);

		double distance(int x, int y);
		bool getHandle(unsigned n, TPoint &p);

		bool startInPlace();
		void startCreate();
		unsigned stop(TGadgetEditor*);

		unsigned keyDown(TGadgetEditor*, TKey, char*, unsigned);
		unsigned mouseLDown(TGadgetEditor*, int, int, unsigned);
		unsigned mouseMove(TGadgetEditor*, int, int, unsigned);
		unsigned mouseLUp(TGadgetEditor*, int, int, unsigned);

		TCloneable* clone() { return new TGText(*this); }

		void store(TOutObjectStream&, ulong);
		void restore(TInObjectStream&, ulong);
		
	protected:
		string text;
		virtual void CalcSize();
		static int cx;	// cursor position while editing
};

class TGFrame:
	public TGText
{
		typedef TGText super;
	public:
		TGFrame() {}
		TGFrame(TGadgetWindow *parent, int x,int y,int w, int h, const string &text="") {
			parent->gadgets.push_back(this);
			this->text = text;
			SetShape(x,y,w,h);
		};
		void paint(TPen &, EPaintType);

		void getShape(TRectangle&);
		double distance(int x, int y);
		unsigned stop(TGadgetEditor*);
		unsigned keyDown(TGadgetEditor*, TKey, char*, unsigned);
		bool getHandle(unsigned n, TPoint &p);
		unsigned mouseLDown(TGadgetEditor *e, int x, int y, unsigned m);
		unsigned mouseMove(TGadgetEditor *e, int x, int y, unsigned m);
		unsigned mouseLUp(TGadgetEditor *e, int x, int y, unsigned m);
		
		TCloneable* clone() { return new TGFrame(*this); }

		void CalcSize();
};

class TGWindow:
	public TGRectangle
{
		typedef TGRectangle super;
	public:
		TGWindow();
	
		TCloneable* clone() { return new TGWindow(*this); }

		void paint(TPen&, EPaintType);
		double distance(int x, int y);
		void translate(int dx, int dy);
		void translateHandle(unsigned handle, int x, int y);
		
		void store(TOutObjectStream&, ulong);
		void restore(TInObjectStream&, ulong);
		
		string title;
		string label;
		unsigned taborder;
		TWindow *window;
};

class TGGroup:
	public TGRectangle
{
	public:
		TGGroup() {
		}
		TGGroup(const TGGroup &g) {
			gadgets.insert(gadgets.begin(), 
										 g.gadgets.begin(), 
										 g.gadgets.end());
		}
		~TGGroup();
		void paint(TPen&, EPaintType);
		double distance(int x, int y);
		void translate(int dx, int dy);
		bool getHandle(unsigned n, TPoint &p);
		
		void CalcSize();
		typedef TGadgetWindow::TGadgetStorage TGadgetStorage;
		TGadgetStorage gadgets;

		TCloneable* clone() { return new TGGroup(*this); }

		void store(TOutObjectStream&, ulong);
		void restore(TInObjectStream&, ulong);
};

#endif
