/*
 * File:	objects.cc
 * Purpose:	
 * Author:	Julian Smart
 * Created:	1993
 * Updated:	
 * Copyright:	(c) 1993, AIAI, University of Edinburgh
 */

static const char sccsid[] = "%W% %G%";

/*
 * Purpose:  Object graphics library demo for wxWindows.
 *           Defines a canvas which repaints its own graphics objects.
 */

/*
 *
 * This shows how the files graphics.h and graphics.cc can be used
 * to maintain objects on an ObjectCanvas. The OnPaint event is handled
 * automatically, and the OnEvent handler routes messages to specific
 * object handlers, e.g. OnLeftClick. Selection of objects (and resizing)
 * is provided, along with drawing lines/splines between objects, with optional 
 * arrows.
 *
 * To test out demo, type 'objects'. Create new objects from the Edit menu.
 * Move objects by dragging with left mouse button. SHIFT-left selects
 * (you can then resize) and deselects. Left click on its own displays the
 * object's ID on the main frame status line.
 *
 * See graphics.h for what other types of objects may be created, and other features
 * of CanvasObjects (e.g. clearing/formatting text, flashing etc.) and ObjectCanvases
 * (e.g. Redrawing).
 */

#include "wx.h"
#include <iostream.h>
#include "read.h"
#include "graphics.h"
#include "misc.h"
#include "drawn.h"
#include "objects.h"

// Declare a frame
MyFrame   *frame = NULL;
wxMenuBar *menu_bar = NULL;

// This statement initialises the whole application
MyApp     myApp;
float     scale = 1.0;

CanvasObject *currentObject = NULL;

// The `main program' equivalent, creating the windows and returning the
// main frame
wxFrame *MyApp::OnInit(void)
{
  // Create the main frame window
  frame = new MyFrame(NULL, "wxWindows Graphics Demo", 0, 0, 400, 400);

  // Give it a status line
  frame->CreateStatusLine();

  // Give it an icon
#ifdef wx_msw
  wxIcon *icon = new wxIcon("aiai_icn");
#endif
#ifdef wx_x
  wxIcon *icon = new wxIcon("aiai.xbm");
#endif
  frame->SetIcon(icon);

  // Make a menubar
  wxMenu *file_menu = new wxMenu;
  file_menu->Append(OBJECTS_EDIT_SCALE, "Edit Scale");
//  file_menu->Append(OBJECTS_TEST_BLACK, "Paint black rectangle");
//  file_menu->Append(OBJECTS_TEST_WHITE, "Paint white rectangle");
//  file_menu->Append(OBJECTS_TEST_ERASE, "Erase current object");
  file_menu->Append(OBJECTS_LOAD_OBJECT, "Load object");
  file_menu->Append(OBJECTS_SAVE_OBJECT, "Save object");
  file_menu->Append(OBJECTS_QUIT, "Quit");

  wxMenu *edit_menu = new wxMenu;
  edit_menu->Append(OBJECTS_ELLIPSE, "New Ellipse");
  edit_menu->Append(OBJECTS_RECTANGLE, "New Rectangle");
  edit_menu->Append(OBJECTS_DRAWN, "New Drawn");
  edit_menu->Append(OBJECTS_CONTAINER, "New Container");
  edit_menu->Append(OBJECTS_ROTATE, "Rotate Drawn");
  edit_menu->Append(OBJECTS_EDIT_TEXT, "Edit Text");
  edit_menu->Append(OBJECTS_CONNECT, "Connect");
  edit_menu->Append(OBJECTS_DELETE, "Delete");

  wxMenu *help_menu = new wxMenu;
  help_menu->Append(OBJECTS_ABOUT, "About");

  menu_bar = new wxMenuBar;

  menu_bar->Append(file_menu, "File");
  menu_bar->Append(edit_menu, "Edit");
  menu_bar->Append(help_menu, "Help");

  // Associate the menu bar with the frame
  frame->SetMenuBar(menu_bar);

  ObjectCanvas *canvas = new ObjectCanvas(frame, 0, 0, 300, 300, wxRETAINED);
  frame->canvas = canvas;
  wxCursor *cursor = new wxCursor(wxCURSOR_HAND);
  canvas->SetCursor(cursor);

  // Give it scrollbars
  canvas->SetScrollbars(20, 20, 50, 50, 4, 4);

  // Remember to initialize the graphics library!
  GraphicsInitialize();

  // Centre the main window
  frame->Centre();
  frame->SetStatusText("Shift-left to (de)select, drag left to move");
  frame->Show(TRUE);

  // Return the main frame window
  return frame;
}

// Define my frame constructor
MyFrame::MyFrame(wxFrame *frame, char *title, int x, int y, int w, int h):
  wxFrame(frame, title, x, y, w, h)
{
}

void MyFrame::LoadObject(void)
{
  char *s = wxFileSelector("Load image from file", NULL, NULL, NULL, "*.img");
  if (s && FileExists(s))
  {
    PrologDatabase database;
    if (!database.ReadProlog(s) || (database.Number() == 0))
    {
      wxMessageBox("Could not read file!");
      return;
    }

    PrologExpr *clause = (PrologExpr *)database.First()->Data();
    char *func = clause->Functor();
    if (func)
    {
      if (strcmp(func, "rectangle") == 0)
      {
        MyRectangle *image = new MyRectangle(100, 50);
        image->ReadPrologAttributes(clause);
        image->SetClientData((wxObject *)OBJECTS_RECTANGLE);
        // label is our own member, not a CanvasObject's
        image->label = copystring("Default text");
        image->SetPen(wxBLACK_PEN);
        image->SetBrush(red_brush);
        canvas->AddObject(image);
        image->FormatText(image->label);
        image->Show(TRUE);
        image->Move(image->xpos, image->ypos);
      }
      else if (strcmp(func, "ellipse") == 0)
      {
        MyEllipse *image = new MyEllipse(100, 50);
        image->ReadPrologAttributes(clause);
        image->SetClientData((wxObject *)OBJECTS_ELLIPSE);
        // label is our own member, not a CanvasObject's
        image->label = copystring("Default text");
        image->SetPen(wxBLACK_PEN);
        image->SetBrush(red_brush);
        canvas->AddObject(image);
        image->FormatText(image->label);
        image->Show(TRUE);
        image->Move(image->xpos, image->ypos);
      }
      else if (strcmp(func, "drawn") == 0)
      {
        MyDrawn *image = new MyDrawn;
        image->ReadPrologAttributes(clause);
        image->SetClientData((wxObject *)OBJECTS_DRAWN);
        // label is our own member, not a CanvasObject's
        image->label = copystring("Default text");
        image->SetPen(wxBLACK_PEN);
        image->SetBrush(red_brush);
        canvas->AddObject(image);
        image->FormatText(image->label);
        image->Show(TRUE);
        image->Move(image->xpos, image->ypos);
      }
    }
    delete[] s;
  }
}

void MyFrame::SaveObject(void)
{
  wxNode *node = canvas->object_list->First();
  CanvasObject *first = NULL;

  while (node)
  {
    CanvasObject *obj = (CanvasObject *)node->Data();
    long client_data = (long)obj->GetClientData();
    if (obj->Selected() && (client_data != OBJECTS_CONNECT))
    {
      first = obj;
      node = NULL;
    }
    else node = node->Next();
  }
  if (!first)
    return;
    
  char *s = wxFileSelector("Save image in file", NULL, NULL, NULL, "*.img");
  if (s)
  {
    ofstream str(s);
    char *functor = "default";
    switch ((int)first->GetClientData())
    {
      case OBJECTS_RECTANGLE:
        functor = "rectangle";
        break;
      case OBJECTS_ELLIPSE:
        functor = "ellipse";
        break;
      case OBJECTS_DRAWN:
        functor = "drawn";
        break;
    }
    PrologExpr *clause = new PrologExpr(functor);
    first->WritePrologAttributes(clause);
    clause->WritePrologClause(str);
    
    delete[] s;
  }
}

// Intercept menu commands
void MyFrame::OnMenuCommand(int id)
{
  switch (id)
  {
    case OBJECTS_QUIT:
    {
      OnClose();
      delete this;
      break;
    }
    case OBJECTS_SAVE_OBJECT:
    {
      SaveObject();
      break;
    }
    case OBJECTS_LOAD_OBJECT:
    {
      LoadObject();
      break;
    }
    case OBJECTS_TEST_BLACK:
    {
      canvas->GetDC()->SetPen(wxBLACK_PEN);
      canvas->GetDC()->SetBrush(wxBLACK_BRUSH);
      canvas->GetDC()->DrawRectangle(50, 50, 60, 60);
      break;
    }
    case OBJECTS_TEST_WHITE:
    {
      canvas->GetDC()->SetPen(wxWHITE_PEN);
      canvas->GetDC()->SetBrush(wxWHITE_BRUSH);
      canvas->GetDC()->DrawRectangle(50, 50, 60, 60);
      break;
    }
    case OBJECTS_TEST_ERASE:
    {
//      currentObject->OnEraseContents();
      if (!currentObject) return;
      
      float maxX, maxY, minX, minY;
      float xp = currentObject->GetX();
      float yp = currentObject->GetY();
      currentObject->GetBoundingBoxMin(&minX, &minY);
      currentObject->GetBoundingBoxMax(&maxX, &maxY);
      float topLeftX = (float)(xp - (maxX / 2.0));
      float topLeftY = (float)(yp - (maxY / 2.0));
      wxDC *dc = currentObject->dc;

      if (dc)
      {
        int penWidth = 0;
        if (currentObject->pen)
         penWidth = currentObject->pen->GetWidth();

        dc->SetPen(white_background_pen);
        dc->SetBrush(white_background_brush);
//        dc->SetPen(wxWHITE_PEN);
//        dc->SetBrush(wxWHITE_BRUSH);

        dc->DrawRectangle((float)(topLeftX - penWidth), (float)(topLeftY - penWidth), 
                          (float)(maxX + penWidth*2.0), (float)(maxY + penWidth*2.0));
      }
      break;
    }
    case OBJECTS_EDIT_SCALE:
    {
      char buf[100];
      sprintf(buf, "%.2f", scale);
      char *s = wxGetTextFromUser("Enter scale", "Input", buf);
      if (s)
      {
        StringToFloat(s, &scale);
        canvas->GetDC()->SetUserScale(scale, scale);
        canvas->Redraw();
      }
      break;
    }

    case OBJECTS_ABOUT:
    {
      (void)wxMessageBox("wxWindows GUI library graphics demo Vsn 1.40\nAuthor: Julian Smart J.Smart@ed.ac.uk\nAIAI (c) 1993", "About Object Graphics Demo");
      break;
    }
    case OBJECTS_RECTANGLE:
    {
      MyRectangle *object = new MyRectangle(80, 60);
      object->centreResize = FALSE;

      // Spot the C++ kludge. It's difficult to have all
      // your derived objects to inherit from a new base
      // (my attempts at multiple inheritance were mostly 
      // doomed), but at least we can have this bit of data
      // in common.
      object->SetClientData((wxObject *)OBJECTS_RECTANGLE);

      // label is our own member, not a CanvasObject's
      object->label = copystring("Default text");

      // Easy to colour the border and background
      object->SetPen(wxBLACK_PEN);
      object->SetBrush(red_brush);

      // Add the object to the canvas
      canvas->AddObject(object);

      // Format the initial text, centring it in the object
      object->FormatText(object->label);
      // You need to do this to display it
      object->Show(TRUE);

      // Move it to an appropriate position
      object->Move(150, 150);

      currentObject = object;
      break;
    }
    case OBJECTS_CONTAINER:
    {
      MyComposite *object = new MyComposite;
      MyRectangle *rectangle = new MyRectangle(150, 80);
      rectangle->SetSensitivityFilter(0);
      
      object->AddChild(rectangle);
      object->SetClientData((wxObject *)OBJECTS_CONTAINER);

      // label is our own member, not a CanvasObject's
      rectangle->label = copystring("Default text");

      // Easy to colour the border and background
      rectangle->SetPen(wxBLACK_PEN);
      rectangle->SetBrush(red_brush);

      // Add the object to the canvas
      object->AddToCanvas(canvas);

      object->Recompute();
      object->MakeContainer();

      // You need to do this to display it
      object->Show(TRUE);

      // Move it to an appropriate position
      object->Move(150, 150);

      currentObject = object;
      break;
    }
    case OBJECTS_DRAWN:
    {
      MyDrawn *object = new MyDrawn;
      char *s = wxFileSelector("Load metafile", NULL, NULL, NULL, "*.wmf");
      if (s)
      {
        
        if (!object->LoadFromMetaFile(s))
        {
          wxMessageBox("Could not load metafile into drawn object!");
          delete object;
          return;
        }
      }

      // Spot the C++ kludge. It's difficult to have all
      // your derived objects to inherit from a new base
      // (my attempts at multiple inheritance were mostly 
      // doomed), but at least we can have this bit of data
      // in common.
      object->SetClientData((wxObject *)OBJECTS_DRAWN);

      // label is our own member, not a CanvasObject's
      object->label = copystring("Default text");

      // Easy to colour the border and background
      object->SetPen(wxBLACK_PEN);
      object->SetBrush(red_brush);

      // Add the object to the canvas
      canvas->AddObject(object);

      // Format the initial text, centring it in the object
      object->FormatText(object->label);
      // You need to do this to display it
      object->Show(TRUE);

      // Move it to an appropriate position
      object->Move(150, 150);

      currentObject = object;
      break;
    }
    case OBJECTS_ELLIPSE:
    {
      MyEllipse *object = new MyEllipse(90, 60);
      object->SetClientData((wxObject *)OBJECTS_ELLIPSE);

      object->label = copystring("Default text");
      object->SetPen(wxBLACK_PEN);
      object->SetBrush(cyan_brush);

      canvas->AddObject(object);

      object->FormatText(object->label);
      object->Show(TRUE);

      object->Move(150, 150);
      currentObject = object;
      break;
    }
    case OBJECTS_CONNECT:
    {
      wxNode *node = canvas->object_list->First();
      CanvasObject *first = NULL;
      CanvasObject *second = NULL;
      // Collect a couple of nodes (arbitrary order!)
      while (node)
      {
        CanvasObject *obj = (CanvasObject *)node->Data();
        long client_data = (long)obj->GetClientData();
        if (obj->Selected() && (client_data != OBJECTS_CONNECT))
	{
          if (first)
            second = obj;
          else first = obj;
	}
        node = node->Next();
      }
      if (first && second)
      {
        // Ok, add the line
        MyLine *line_object = new MyLine;

        // Yes, you can have more than 2 control points! In which case
        // it becomes a multi-segment line, and very pretty it
        // is too. The same for SplineObjects.
        // Try out the line straightening feature sometime, it's
        // rather good if I say so myself.
        line_object->MakeLineControlPoints(2);
        line_object->SetClientData((wxObject *)OBJECTS_CONNECT);

        // Add the connection between first and second
        // (the library knows about such things)
        first->AddLine(line_object, second);
        canvas->AddObject(line_object);
        line_object->Show(TRUE);

        // It won't get drawn properly unless you move both
        // connected images
        first->Move(first->GetX(), first->GetY());
        second->Move(second->GetX(), second->GetY());
        canvas->Redraw();
        break;
      }
      break;
    }
    case OBJECTS_ROTATE:
    {
      char *s = wxGetTextFromUser("Enter angle");
      if (s)
      {
        float theta = atof(s);
        
        wxNode *node = canvas->object_list->First();
        while (node)
        {
          CanvasObject *image = (CanvasObject *)node->Data();
          long client_data = (long)image->GetClientData();
          if (image->Selected())
          {
            switch (client_data)
            {
              case OBJECTS_DRAWN:
              {
                DrawnObject *drawn = (DrawnObject *)image;
                drawn->Erase();
                drawn->Rotate(0.0, 0.0, theta);
                drawn->Draw();
                break;
              }
              default:
                break;
            }
          }
          node = node->Next();
        }
      }
      break;
    }
    case OBJECTS_DELETE:
    {
      // Slightly subtle. Always delete a node's arcs first.
      // When you've deleted a selected image, go to the start of
      // the object list, or you'll be accessing dead objects.
      wxNode *node = canvas->object_list->First();
      while (node)
      {
        CanvasObject *image = (CanvasObject *)node->Data();
        long client_data = (long)image->GetClientData();
        if (image->Selected())
        {
          image->Select(FALSE);
          image->Erase();

          switch (client_data)
	  {
            case OBJECTS_ELLIPSE:
            case OBJECTS_RECTANGLE:
            case OBJECTS_DRAWN:
            {
              wxNode *node1 = image->lines.First();
              while (node1)
	      {
                LineObject *line_object = (LineObject *)node1->Data();
                line_object->Erase();
                line_object->Unlink();
                delete line_object;
                node1 = image->lines.First();
	      }
              delete image;
              break;
            }
            case OBJECTS_CONNECT:
	    {
              LineObject *line_object = (LineObject *)image;
              line_object->Erase();
              line_object->Unlink();
              delete line_object;
              break;
	    }
	  }

          node = canvas->object_list->First();
        } else node = node->Next();
      }

      canvas->Redraw();
      break;
    }

    case OBJECTS_EDIT_TEXT:
    {
      wxNode *node = canvas->object_list->First();
      while (node)
      {
        CanvasObject *image = (CanvasObject *)node->Data();
        long client_data = (long)image->GetClientData();
        if (image->Selected())
        {
          image->Select(FALSE);
          switch (client_data)
	  {
            case OBJECTS_RECTANGLE:
	    {
              MyRectangle *object = (MyRectangle *)image;
              char *s = wxGetTextFromUser("Edit text", "Edit", object->label);
              if (s)
	      {
                delete object->label;
                object->label = copystring(s);
                object->FormatText(s);
                object->Draw();
	      }
              break;
	    }
            case OBJECTS_DRAWN:
	    {
              MyDrawn *object = (MyDrawn *)image;
              char *s = wxGetTextFromUser("Edit text", "Edit", object->label);
              if (s)
	      {
                delete object->label;
                object->label = copystring(s);
                object->FormatText(s);
                object->Draw();
	      }
              break;
	    }
            case OBJECTS_ELLIPSE:
	    {
              MyEllipse *object = (MyEllipse *)image;
              char *s = wxGetTextFromUser("Edit text", "Edit", object->label);
              if (s)
	      {
                delete object->label;
                object->label = copystring(s);
                object->FormatText(s);
                object->Draw();
	      }
              break;
	    }
	  }
          return;
	}
        node = node->Next();
      }
      break;
    }
  }
}

// Define the behaviour for the frame closing
// - must delete all frames except for the main one.
Bool MyFrame::OnClose(void)
{
  return TRUE;
}

void GenericOk(wxButton& but, wxCommandEvent& event)
{
  wxDialogBox *dialog = (wxDialogBox *)but.GetParent();

  dialog->Show(FALSE);
}

MyRectangle::MyRectangle(float width, float height):RectangleObject(width, height)
{
  SetPen(wxBLACK_PEN);
  SetBrush(green_brush);
  SetFont(swiss_font_10);
}

void MyRectangle::OnLeftClick(float x, float y, int keys, int attachment)
{
  if (keys & KEY_SHIFT)
  {
    // Selection is a concept the library knows about
    if (Selected())
    {
      Select(FALSE);
      canvas->Redraw(); // Redraw because bits of objects will be are missing
    }
    else
    {
      Select(TRUE);
    }
  }
  else if (keys & KEY_CTRL)
  {
    // Do something for CONTROL
  }
  else
  {
    frame->SetStatusText(label);
  }
}

MyDrawn::MyDrawn(void)
{
  SetPen(wxBLACK_PEN);
  SetBrush(green_brush);
  SetFont(swiss_font_10);
}

void MyDrawn::OnLeftClick(float x, float y, int keys, int attachment)
{
  if (keys & KEY_SHIFT)
  {
    // Selection is a concept the library knows about
    if (Selected())
    {
      Select(FALSE);
      canvas->Redraw(); // Redraw because bits of objects will be are missing
    }
    else
    {
      Select(TRUE);
    }
  }
  else if (keys & KEY_CTRL)
  {
    // Do something for CONTROL
  }
  else
  {
    frame->SetStatusText(label);
  }
}

MyEllipse::MyEllipse(float width, float height):EllipseObject(width, height)
{
  SetPen(wxBLACK_PEN);
  SetBrush(green_brush);
  SetFont(swiss_font_10);
}

void MyEllipse::OnLeftClick(float x, float y, int keys, int attachment)
{
  if (keys & KEY_SHIFT)
  {
    // Selection is a concept the library knows about
    if (Selected())
    {
      Select(FALSE);
      canvas->Redraw(); // Redraw because bits of objects will be are missing
    }
    else
    {
      Select(TRUE);
    }
  }
  else if (keys & KEY_CTRL)
  {
    // Do something for CONTROL
  }
  else
  {
    frame->SetStatusText(label);
  }
}

MyLine::MyLine(void)
{
  SetPen(wxBLACK_PEN);
  SetBrush(wxBLACK_BRUSH);
  SetArrowSize(10, 4);
  SetEndArrow(ARROW_END);
}

void MyLine::OnLeftClick(float x, float y, int keys, int attachment)
{
  // If more than one link for this item, choose a link
  if (keys & KEY_SHIFT)
  {
    if (Selected())
    {
      Select(FALSE);
      canvas->Redraw(); // Redraw because bits of objects will be are missing
    }
    else
    {
      Select(TRUE);
    }
  }
  else if (keys & KEY_CTRL)
  {
    // Do something for CONTROL
  }
}

MyComposite::MyComposite(void)
{
  label = NULL;
}

void MyComposite::OnLeftClick(float x, float y, int keys, int attachment)
{
  if (keys & KEY_SHIFT)
  {
    if (Selected())
    {
      Select(FALSE);
      canvas->Redraw(); // Redraw because bits of objects will be are missing
    }
    else
    {
      Select(TRUE);
    }
  }
  else if (keys & KEY_CTRL)
  {
    // Do something for CONTROL
  }
}

