package Mojolicious::Plugin::ViewBuilder;
our $VERSION = "0.04";

use Mojo::Base 'Mojolicious::Plugin';

sub register {
    my ( $self, $app, $conf ) = @_;
    push @{ $app->renderer->classes }, __PACKAGE__;

    $app->helper(
        add_view => sub {
            shift;
            push( @{ $app->{views}->{ +shift() } }, shift );
        }
    );

    $app->helper(
        pluggable_view => sub {
            my $c=shift;
            die "No view defined" unless defined $_[0];
            my $output;
            $output .= ref $_ eq 'CODE' ? $_->( $c, $app,@_ ) : $app->$_($c,@_)
                for ( @{ $app->{views}->{ +shift() } } );
            return $output;
        }
    );

}

1;

__END__

=encoding utf-8

=head1 NAME

Mojolicious::Plugin::ViewBuilder - a Mojolicious plugin that allows to chain templates generated by other plugins

=head1 SYNOPSIS


  #Load the plugin

  # - Mojolicious
  $self->plugin('ViewBuilder');

  # - Mojolicious::Lite
  plugin 'ViewBuilder';

  # then, use it in your template, declare a new area, labeled "profile"
  <%= pluggable_view 'profile' %>

  # now, in the plugin, that will attach the new helper to the view

  sub register {
      my ( $self, $app, $conf ) = @_;

      $app->helper(
          test => sub {
              shift->render_to_string( "test", some_data => 1 );
          }
      );

      #attach the test helper to the view "profile"
      $app->add_view( "profile", "test" );

  }
  1;

  # or you can append a callback

  sub register {
      my ( $self, $app, $conf ) = @_;
      $app->add_view(
          profile => sub {
              shift->render_to_string( "test", some_data => 42 );
          }
      );
  }
  1;
  __DATA__
  @@ test.html.ep
  huuuray!
  % if( stash("some_data") and stash("some_data")==42){
   double it!
  % }

=head1 DESCRIPTION

L<Mojolicious::Plugin::ViewBuilder> is a L<Mojolicious> plugin. Chains templates generated by other plugins, associating them within the view. This is useful only if your workflow is to reap into little pieces the webapp (e.g. in plugins)

=head1 METHODS

L<Mojolicious::Plugin::ViewBuilder> inherits all methods from
L<Mojolicious::Plugin> and implements the following new ones.

=head2 pluggable_view

  <%=pluggable_view "profile" %>

Will render all the attached plugin associated within the view

=head2 add_view

  $app->add_view("view","helper");
  $app->add_view(view => sub{shift->render_to_string("Hello mojolicious!")});

Attach the "helper" within the "view"

=head1 LICENSE

Copyright (C) mudler.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHOR

mudler E<lt>mudler@dark-lab.netE<gt>

=cut

