/*	$NetBSD: ds1307.c,v 1.44 2025/10/14 09:19:48 tnn Exp $	*/

/*
 * Copyright (c) 2003 Wasabi Systems, Inc.
 * All rights reserved.
 *
 * Written by Steve C. Woodford and Jason R. Thorpe for Wasabi Systems, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed for the NetBSD Project by
 *      Wasabi Systems, Inc.
 * 4. The name of Wasabi Systems, Inc. may not be used to endorse
 *    or promote products derived from this software without specific prior
 *    written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY WASABI SYSTEMS, INC. ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL WASABI SYSTEMS, INC
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: ds1307.c,v 1.44 2025/10/14 09:19:48 tnn Exp $");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/device.h>
#include <sys/kernel.h>
#include <sys/fcntl.h>
#include <sys/uio.h>
#include <sys/conf.h>
#include <sys/event.h>

#include <dev/clock_subr.h>

#include <dev/i2c/i2cvar.h>
#include <dev/i2c/ds1307reg.h>
#include <dev/sysmon/sysmonvar.h>

#include "ioconf.h"

struct dsrtc_model {
	const i2c_addr_t *dm_valid_addrs;
	uint16_t dm_model;
	uint8_t dm_ch_reg;
	uint8_t dm_ch_value;
	uint8_t dm_vbaten_reg;
	uint8_t dm_vbaten_value;
	uint8_t dm_rtc_start;
	uint8_t dm_rtc_size;
	uint8_t dm_nvram_start;
	uint8_t dm_nvram_size;
	uint8_t dm_flags;
#define	DSRTC_FLAG_CLOCK_HOLD		0x01
#define	DSRTC_FLAG_BCD			0x02
#define	DSRTC_FLAG_TEMP			0x04
#define DSRTC_FLAG_VBATEN		0x08
#define	DSRTC_FLAG_CLOCK_HOLD_REVERSED	0x20
};

static const i2c_addr_t ds1307_valid_addrs[] = { DS1307_ADDR, 0 };
static const struct dsrtc_model ds1307_model = {
	.dm_valid_addrs = ds1307_valid_addrs,
	.dm_model = 1307,
	.dm_ch_reg = DSXXXX_SECONDS,
	.dm_ch_value = DS1307_SECONDS_CH,
	.dm_rtc_start = DS1307_RTC_START,
	.dm_rtc_size = DS1307_RTC_SIZE,
	.dm_nvram_start = DS1307_NVRAM_START,
	.dm_nvram_size = DS1307_NVRAM_SIZE,
	.dm_flags = DSRTC_FLAG_BCD | DSRTC_FLAG_CLOCK_HOLD,
};

static const struct dsrtc_model ds1339_model = {
	.dm_valid_addrs = ds1307_valid_addrs,
	.dm_model = 1339,
	.dm_rtc_start = DS1339_RTC_START,
	.dm_rtc_size = DS1339_RTC_SIZE,
	.dm_flags = DSRTC_FLAG_BCD,
};

static const struct dsrtc_model ds1340_model = {
	.dm_valid_addrs = ds1307_valid_addrs,
	.dm_model = 1340,
	.dm_ch_reg = DSXXXX_SECONDS,
	.dm_ch_value = DS1340_SECONDS_EOSC,
	.dm_rtc_start = DS1340_RTC_START,
	.dm_rtc_size = DS1340_RTC_SIZE,
	.dm_flags = DSRTC_FLAG_BCD,
};

static const struct dsrtc_model ds1672_model = {
	.dm_valid_addrs = ds1307_valid_addrs,
	.dm_model = 1672,
	.dm_rtc_start = DS1672_RTC_START,
	.dm_rtc_size = DS1672_RTC_SIZE,
	.dm_ch_reg = DS1672_CONTROL,
	.dm_ch_value = DS1672_CONTROL_CH,
	.dm_flags = 0,
};

static const struct dsrtc_model ds3231_model = {
	.dm_valid_addrs = ds1307_valid_addrs,
	.dm_model = 3231,
	.dm_rtc_start = DS3232_RTC_START,
	.dm_rtc_size = DS3232_RTC_SIZE,
	.dm_flags = DSRTC_FLAG_BCD | DSRTC_FLAG_TEMP,
};

static const struct dsrtc_model ds3232_model = {
	.dm_valid_addrs = ds1307_valid_addrs,
	.dm_model = 3232,
	.dm_rtc_start = DS3232_RTC_START,
	.dm_rtc_size = DS3232_RTC_SIZE,
	.dm_nvram_start = DS3232_NVRAM_START,
	.dm_nvram_size = DS3232_NVRAM_SIZE,
	/*
	 * XXX
	 * the DS3232 likely has the temperature sensor too but I can't
	 * easily verify or test that right now
	 */
	.dm_flags = DSRTC_FLAG_BCD,
};

static const i2c_addr_t mcp7940_valid_addrs[] = { MCP7940_ADDR, 0 };
static const struct dsrtc_model mcp7940_model = {
	.dm_valid_addrs = mcp7940_valid_addrs,
	.dm_model = 7940,
	.dm_rtc_start = DS1307_RTC_START,
	.dm_rtc_size = DS1307_RTC_SIZE,
	.dm_ch_reg = DSXXXX_SECONDS,
	.dm_ch_value = DS1307_SECONDS_CH,
	.dm_vbaten_reg = DSXXXX_DAY,
	.dm_vbaten_value = MCP7940_TOD_DAY_VBATEN,
	.dm_nvram_start = MCP7940_NVRAM_START,
	.dm_nvram_size = MCP7940_NVRAM_SIZE,
	.dm_flags = DSRTC_FLAG_BCD | DSRTC_FLAG_CLOCK_HOLD |
		DSRTC_FLAG_VBATEN | DSRTC_FLAG_CLOCK_HOLD_REVERSED,
};

static const struct device_compatible_entry compat_data[] = {
	{ .compat = "dallas,ds1307",		.data = &ds1307_model },
	{ .compat = "maxim,ds1307",		.data = &ds1307_model },
	{ .compat = "i2c-ds1307",		.data = &ds1307_model },

	{ .compat = "dallas,ds1339",		.data = &ds1339_model },
	{ .compat = "maxim,ds1339",		.data = &ds1339_model },

	{ .compat = "dallas,ds1340",		.data = &ds1340_model },
	{ .compat = "maxim,ds1340",		.data = &ds1340_model },

	{ .compat = "dallas,ds1672",		.data = &ds1672_model },
	{ .compat = "maxim,ds1672",		.data = &ds1672_model },

	{ .compat = "dallas,ds3231",		.data = &ds3231_model },
	{ .compat = "maxim,ds3231",		.data = &ds3231_model },

	{ .compat = "dallas,ds3232",		.data = &ds3232_model },
	{ .compat = "maxim,ds3232",		.data = &ds3232_model },

	{ .compat = "microchip,mcp7940",	.data = &mcp7940_model },

	DEVICE_COMPAT_EOL
};

struct dsrtc_softc {
	device_t sc_dev;
	i2c_tag_t sc_tag;
	uint8_t sc_address;
	bool sc_open;
	struct dsrtc_model sc_model;
	struct todr_chip_handle sc_todr;
	struct sysmon_envsys *sc_sme;
	envsys_data_t sc_sensor;
	unsigned int sc_base_year;
};

static void	dsrtc_attach(device_t, device_t, void *);
static int	dsrtc_match(device_t, cfdata_t, void *);

CFATTACH_DECL_NEW(dsrtc, sizeof(struct dsrtc_softc),
    dsrtc_match, dsrtc_attach, NULL, NULL);

dev_type_open(dsrtc_open);
dev_type_close(dsrtc_close);
dev_type_read(dsrtc_read);
dev_type_write(dsrtc_write);

const struct cdevsw dsrtc_cdevsw = {
	.d_open = dsrtc_open,
	.d_close = dsrtc_close,
	.d_read = dsrtc_read,
	.d_write = dsrtc_write,
	.d_ioctl = noioctl,
	.d_stop = nostop,
	.d_tty = notty,
	.d_poll = nopoll,
	.d_mmap = nommap,
	.d_kqfilter = nokqfilter,
	.d_discard = nodiscard,
	.d_flag = D_OTHER
};

static int dsrtc_gettime_ymdhms(struct todr_chip_handle *, struct clock_ymdhms *);
static int dsrtc_settime_ymdhms(struct todr_chip_handle *, struct clock_ymdhms *);
static int dsrtc_clock_read_ymdhms(struct dsrtc_softc *, struct clock_ymdhms *);
static int dsrtc_clock_write_ymdhms(struct dsrtc_softc *, struct clock_ymdhms *);

static int dsrtc_gettime_timeval(struct todr_chip_handle *, struct timeval *);
static int dsrtc_settime_timeval(struct todr_chip_handle *, struct timeval *);
static int dsrtc_clock_read_timeval(struct dsrtc_softc *, time_t *);
static int dsrtc_clock_write_timeval(struct dsrtc_softc *, time_t);

static int dsrtc_read_temp(struct dsrtc_softc *, uint32_t *);
static void dsrtc_refresh(struct sysmon_envsys *, envsys_data_t *);

static const struct dsrtc_model *
dsrtc_model_by_number(u_int model)
{
	const struct device_compatible_entry *dce;
	const struct dsrtc_model *dm;

	/* no model given, assume it's a DS1307 */
	if (model == 0)
		return &ds1307_model;

	for (dce = compat_data; dce->compat != NULL; dce++) {
		dm = dce->data;
		if (dm->dm_model == model)
			return dm;
	}
	return NULL;
}

static const struct dsrtc_model *
dsrtc_model_by_compat(const struct i2c_attach_args *ia)
{
	const struct dsrtc_model *dm = NULL;
	const struct device_compatible_entry *dce;

	if ((dce = iic_compatible_lookup(ia, compat_data)) != NULL)
		dm = dce->data;

	return dm;
}

static bool
dsrtc_is_valid_addr_for_model(const struct dsrtc_model *dm, i2c_addr_t addr)
{

	for (int i = 0; dm->dm_valid_addrs[i] != 0; i++) {
		if (addr == dm->dm_valid_addrs[i])
			return true;
	}
	return false;
}

static int
dsrtc_match(device_t parent, cfdata_t cf, void *arg)
{
	struct i2c_attach_args *ia = arg;
	const struct dsrtc_model *dm;
	int match_result;

	if (iic_use_direct_match(ia, cf, compat_data, &match_result))
		return match_result;

	dm = dsrtc_model_by_number(cf->cf_flags & 0xffff);
	if (dm == NULL)
		return 0;

	if (dsrtc_is_valid_addr_for_model(dm, ia->ia_addr))
		return I2C_MATCH_ADDRESS_ONLY;

	return 0;
}

static void
dsrtc_attach(device_t parent, device_t self, void *arg)
{
	struct dsrtc_softc *sc = device_private(self);
	struct i2c_attach_args *ia = arg;
	const struct dsrtc_model *dm;

	if ((dm = dsrtc_model_by_compat(ia)) == NULL)
		dm = dsrtc_model_by_number(device_cfdata(self)->cf_flags);

	if (dm == NULL) {
		aprint_error(": unable to determine model!\n");
		return;
	}

	aprint_naive(": Real-time Clock%s\n",
	    dm->dm_nvram_size > 0 ? "/NVRAM" : "");
	aprint_normal(": DS%u Real-time Clock%s\n", dm->dm_model,
	    dm->dm_nvram_size > 0 ? "/NVRAM" : "");

	sc->sc_tag = ia->ia_tag;
	sc->sc_address = ia->ia_addr;
	sc->sc_model = *dm;
	sc->sc_dev = self;
	sc->sc_open = 0;
	sc->sc_todr.todr_dev = self;
	
	if (dm->dm_flags & DSRTC_FLAG_BCD) {
		sc->sc_todr.todr_gettime_ymdhms = dsrtc_gettime_ymdhms;
		sc->sc_todr.todr_settime_ymdhms = dsrtc_settime_ymdhms;
	} else {
		sc->sc_todr.todr_gettime = dsrtc_gettime_timeval;
		sc->sc_todr.todr_settime = dsrtc_settime_timeval;
	}

	sc->sc_base_year = device_getprop_uint_default(self, "start-year",
	    POSIX_BASE_YEAR);

	todr_attach(&sc->sc_todr);
	if ((sc->sc_model.dm_flags & DSRTC_FLAG_TEMP) != 0) {
		int error;

		sc->sc_sme = sysmon_envsys_create();
		sc->sc_sme->sme_name = device_xname(self);
		sc->sc_sme->sme_cookie = sc;
		sc->sc_sme->sme_refresh = dsrtc_refresh;

		sc->sc_sensor.units =  ENVSYS_STEMP;
		sc->sc_sensor.state = ENVSYS_SINVALID;
		sc->sc_sensor.flags = 0;
		(void)strlcpy(sc->sc_sensor.desc, "temperature",
		    sizeof(sc->sc_sensor.desc));

		if (sysmon_envsys_sensor_attach(sc->sc_sme, &sc->sc_sensor)) {
			aprint_error_dev(self, "unable to attach sensor\n");
			goto bad;
		}

		error = sysmon_envsys_register(sc->sc_sme);
		if (error) {
			aprint_error_dev(self, 
			    "error %d registering with sysmon\n", error);
			goto bad;
		}
	}
	return;
bad:
	sysmon_envsys_destroy(sc->sc_sme);
}

/*ARGSUSED*/
int
dsrtc_open(dev_t dev, int flag, int fmt, struct lwp *l)
{
	struct dsrtc_softc *sc;

	if ((sc = device_lookup_private(&dsrtc_cd, minor(dev))) == NULL)
		return ENXIO;

	/* XXX: Locking */
	if (sc->sc_open)
		return EBUSY;

	sc->sc_open = true;
	return 0;
}

/*ARGSUSED*/
int
dsrtc_close(dev_t dev, int flag, int fmt, struct lwp *l)
{
	struct dsrtc_softc *sc;

	if ((sc = device_lookup_private(&dsrtc_cd, minor(dev))) == NULL)
		return ENXIO;

	sc->sc_open = false;
	return 0;
}

/*ARGSUSED*/
int
dsrtc_read(dev_t dev, struct uio *uio, int flags)
{
	struct dsrtc_softc *sc;
	int error;

	if ((sc = device_lookup_private(&dsrtc_cd, minor(dev))) == NULL)
		return ENXIO;

	const struct dsrtc_model * const dm = &sc->sc_model;
	if (uio->uio_offset < 0 || uio->uio_offset >= dm->dm_nvram_size)
		return EINVAL;

	if ((error = iic_acquire_bus(sc->sc_tag, 0)) != 0)
		return error;

	while (uio->uio_resid && uio->uio_offset < dm->dm_nvram_size) {
		uint8_t ch, cmd;
		const u_int a = uio->uio_offset;
		cmd = a + dm->dm_nvram_start;
		if ((error = iic_exec(sc->sc_tag,
		    uio->uio_resid > 1 ? I2C_OP_READ : I2C_OP_READ_WITH_STOP,
		    sc->sc_address, &cmd, 1, &ch, 1, 0)) != 0) {
			iic_release_bus(sc->sc_tag, 0);
			aprint_error_dev(sc->sc_dev,
			    "%s: read failed at 0x%x: %d\n",
			    __func__, a, error);
			return error;
		}
		if ((error = uiomove(&ch, 1, uio)) != 0) {
			iic_release_bus(sc->sc_tag, 0);
			return error;
		}
	}

	iic_release_bus(sc->sc_tag, 0);

	return 0;
}

/*ARGSUSED*/
int
dsrtc_write(dev_t dev, struct uio *uio, int flags)
{
	struct dsrtc_softc *sc;
	int error;

	if ((sc = device_lookup_private(&dsrtc_cd, minor(dev))) == NULL)
		return ENXIO;

	const struct dsrtc_model * const dm = &sc->sc_model;
	if (uio->uio_offset >= dm->dm_nvram_size)
		return EINVAL;

	if ((error = iic_acquire_bus(sc->sc_tag, 0)) != 0)
		return error;

	while (uio->uio_resid && uio->uio_offset < dm->dm_nvram_size) {
		uint8_t cmdbuf[2];
		const u_int a = (int)uio->uio_offset;
		cmdbuf[0] = a + dm->dm_nvram_start;
		if ((error = uiomove(&cmdbuf[1], 1, uio)) != 0)
			break;

		if ((error = iic_exec(sc->sc_tag,
		    uio->uio_resid ? I2C_OP_WRITE : I2C_OP_WRITE_WITH_STOP,
		    sc->sc_address, cmdbuf, 1, &cmdbuf[1], 1, 0)) != 0) {
			aprint_error_dev(sc->sc_dev,
			    "%s: write failed at 0x%x: %d\n",
			    __func__, a, error);
			break;
		}
	}

	iic_release_bus(sc->sc_tag, 0);

	return error;
}

static int
dsrtc_gettime_ymdhms(struct todr_chip_handle *ch, struct clock_ymdhms *dt)
{
	struct dsrtc_softc *sc = device_private(ch->todr_dev);
	struct clock_ymdhms check;
	int retries;

	memset(dt, 0, sizeof(*dt));
	memset(&check, 0, sizeof(check));

	/*
	 * Since we don't support Burst Read, we have to read the clock twice
	 * until we get two consecutive identical results.
	 */
	retries = 5;
	do {
		dsrtc_clock_read_ymdhms(sc, dt);
		dsrtc_clock_read_ymdhms(sc, &check);
	} while (memcmp(dt, &check, sizeof(check)) != 0 && --retries);

	return 0;
}

static int
dsrtc_settime_ymdhms(struct todr_chip_handle *ch, struct clock_ymdhms *dt)
{
	struct dsrtc_softc *sc = device_private(ch->todr_dev);

	if (dsrtc_clock_write_ymdhms(sc, dt) == 0)
		return -1;

	return 0;
}

static int
dsrtc_clock_read_ymdhms(struct dsrtc_softc *sc, struct clock_ymdhms *dt)
{
	struct dsrtc_model * const dm = &sc->sc_model;
	uint8_t bcd[DSXXXX_RTC_SIZE], cmdbuf[1];
	int error;

	KASSERT(DSXXXX_RTC_SIZE >= dm->dm_rtc_size);

	if ((error = iic_acquire_bus(sc->sc_tag, 0)) != 0) {
		aprint_error_dev(sc->sc_dev,
		    "%s: failed to acquire I2C bus: %d\n",
		    __func__, error);
		return 0;
	}

	/* Read each RTC register in order. */
	for (u_int i = 0; !error && i < dm->dm_rtc_size; i++) {
		cmdbuf[0] = dm->dm_rtc_start + i;

		error = iic_exec(sc->sc_tag, I2C_OP_READ_WITH_STOP,
		    sc->sc_address, cmdbuf, 1, &bcd[i], 1, 0);
	}

	/* Done with I2C */
	iic_release_bus(sc->sc_tag, 0);

	if (error != 0) {
		aprint_error_dev(sc->sc_dev,
		    "%s: failed to read rtc at 0x%x: %d\n", 
		    __func__, cmdbuf[0], error);
		return 0;
	}

	/*
	 * Convert the RTC's register values into something useable
	 */
	dt->dt_sec = bcdtobin(bcd[DSXXXX_SECONDS] & DSXXXX_SECONDS_MASK);
	dt->dt_min = bcdtobin(bcd[DSXXXX_MINUTES] & DSXXXX_MINUTES_MASK);

	if ((bcd[DSXXXX_HOURS] & DSXXXX_HOURS_12HRS_MODE) != 0) {
		dt->dt_hour = bcdtobin(bcd[DSXXXX_HOURS] &
		    DSXXXX_HOURS_12MASK) % 12; /* 12AM -> 0, 12PM -> 12 */
		if (bcd[DSXXXX_HOURS] & DSXXXX_HOURS_12HRS_PM)
			dt->dt_hour += 12;
	} else
		dt->dt_hour = bcdtobin(bcd[DSXXXX_HOURS] &
		    DSXXXX_HOURS_24MASK);

	dt->dt_day = bcdtobin(bcd[DSXXXX_DATE] & DSXXXX_DATE_MASK);
	dt->dt_mon = bcdtobin(bcd[DSXXXX_MONTH] & DSXXXX_MONTH_MASK);

	dt->dt_year = bcdtobin(bcd[DSXXXX_YEAR]) + sc->sc_base_year;
	if (bcd[DSXXXX_MONTH] & DSXXXX_MONTH_CENTURY) {
		dt->dt_year += 100;
	}

	return 1;
}

static int
dsrtc_clock_write_ymdhms(struct dsrtc_softc *sc, struct clock_ymdhms *dt)
{
	struct dsrtc_model * const dm = &sc->sc_model;
	uint8_t bcd[DSXXXX_RTC_SIZE], cmdbuf[2];
	int error;

	KASSERT(DSXXXX_RTC_SIZE >= dm->dm_rtc_size);

	/*
	 * Convert our time representation into something the DSXXXX
	 * can understand.
	 */
	bcd[DSXXXX_SECONDS] = bintobcd(dt->dt_sec);
	bcd[DSXXXX_MINUTES] = bintobcd(dt->dt_min);
	bcd[DSXXXX_HOURS] = bintobcd(dt->dt_hour); /* DSXXXX_HOURS_12HRS_MODE=0 */
	bcd[DSXXXX_DATE] = bintobcd(dt->dt_day);
	bcd[DSXXXX_DAY] = bintobcd(dt->dt_wday);
	bcd[DSXXXX_MONTH] = bintobcd(dt->dt_mon);
	
	bcd[DSXXXX_YEAR] = bintobcd((dt->dt_year - sc->sc_base_year) % 100);
	if (dt->dt_year - sc->sc_base_year >= 100)
		bcd[DSXXXX_MONTH] |= DSXXXX_MONTH_CENTURY;

	if ((error = iic_acquire_bus(sc->sc_tag, 0)) != 0) {
		aprint_error_dev(sc->sc_dev,
		    "%s: failed to acquire I2C bus: %d\n",
		    __func__, error);
		return 0;
	}

	/* Stop the clock */
	cmdbuf[0] = dm->dm_ch_reg;

	if ((error = iic_exec(sc->sc_tag, I2C_OP_READ, sc->sc_address,
	    cmdbuf, 1, &cmdbuf[1], 1, 0)) != 0) {
		iic_release_bus(sc->sc_tag, 0);
		aprint_error_dev(sc->sc_dev,
		    "%s: failed to read Hold Clock: %d\n",
		    __func__, error);
		return 0;
	}

	if (sc->sc_model.dm_flags & DSRTC_FLAG_CLOCK_HOLD_REVERSED)
		cmdbuf[1] &= ~dm->dm_ch_value;
	else
		cmdbuf[1] |= dm->dm_ch_value;

	if ((error = iic_exec(sc->sc_tag, I2C_OP_WRITE, sc->sc_address,
	    cmdbuf, 1, &cmdbuf[1], 1, 0)) != 0) {
		iic_release_bus(sc->sc_tag, 0);
		aprint_error_dev(sc->sc_dev,
		    "%s: failed to write Hold Clock: %d\n",
		    __func__, error);
		return 0;
	}

	/*
	 * Write registers in reverse order. The last write (to the Seconds
	 * register) will undo the Clock Hold, above.
	 */
	uint8_t op = I2C_OP_WRITE;
	for (signed int i = dm->dm_rtc_size - 1; i >= 0; i--) {
		cmdbuf[0] = dm->dm_rtc_start + i;
		if ((dm->dm_flags & DSRTC_FLAG_VBATEN) &&
				dm->dm_rtc_start + i == dm->dm_vbaten_reg)
			bcd[i] |= dm->dm_vbaten_value;
		if (dm->dm_rtc_start + i == dm->dm_ch_reg) {
			op = I2C_OP_WRITE_WITH_STOP;
			if (dm->dm_flags & DSRTC_FLAG_CLOCK_HOLD_REVERSED)
				bcd[i] |= dm->dm_ch_value;
		}
		if ((error = iic_exec(sc->sc_tag, op, sc->sc_address,
		    cmdbuf, 1, &bcd[i], 1, 0)) != 0) {
			iic_release_bus(sc->sc_tag, 0);
			aprint_error_dev(sc->sc_dev,
			    "%s: failed to write rtc at 0x%x: %d\n",
			    __func__, i, error);
			/* XXX: Clock Hold is likely still asserted! */
			return 0;
		}
	}
	/*
	 * If the clock hold register isn't the same register as seconds,
	 * we need to reenable the clock.
	 */
	if (op != I2C_OP_WRITE_WITH_STOP) {
		cmdbuf[0] = dm->dm_ch_reg;
		if (dm->dm_flags & DSRTC_FLAG_CLOCK_HOLD_REVERSED)
			cmdbuf[1] |= dm->dm_ch_value;
		else
			cmdbuf[1] &= ~dm->dm_ch_value;

		if ((error = iic_exec(sc->sc_tag, I2C_OP_WRITE_WITH_STOP,
		    sc->sc_address, cmdbuf, 1, &cmdbuf[1], 1, 0)) != 0) {
			iic_release_bus(sc->sc_tag, 0);
			aprint_error_dev(sc->sc_dev,
			    "%s: failed to Hold Clock: %d\n",
			    __func__, error);
			return 0;
		}
	}

	iic_release_bus(sc->sc_tag, 0);

	return 1;
}

static int
dsrtc_gettime_timeval(struct todr_chip_handle *ch, struct timeval *tv)
{
	struct dsrtc_softc *sc = device_private(ch->todr_dev);
	struct timeval check;
	int retries;

	memset(tv, 0, sizeof(*tv));
	memset(&check, 0, sizeof(check));

	/*
	 * Since we don't support Burst Read, we have to read the clock twice
	 * until we get two consecutive identical results.
	 */
	retries = 5;
	do {
		dsrtc_clock_read_timeval(sc, &tv->tv_sec);
		dsrtc_clock_read_timeval(sc, &check.tv_sec);
	} while (memcmp(tv, &check, sizeof(check)) != 0 && --retries);

	return 0;
}

static int
dsrtc_settime_timeval(struct todr_chip_handle *ch, struct timeval *tv)
{
	struct dsrtc_softc *sc = device_private(ch->todr_dev);

	if (dsrtc_clock_write_timeval(sc, tv->tv_sec) == 0)
		return -1;

	return 0;
}

/*
 * The RTC probably has a nice Clock Burst Read/Write command, but we can't use
 * it, since some I2C controllers don't support anything other than single-byte
 * transfers.
 */
static int
dsrtc_clock_read_timeval(struct dsrtc_softc *sc, time_t *tp)
{
	const struct dsrtc_model * const dm = &sc->sc_model;
	uint8_t buf[4];
	int error;

	if ((error = iic_acquire_bus(sc->sc_tag, 0)) != 0) {
		aprint_error_dev(sc->sc_dev,
		    "%s: failed to acquire I2C bus: %d\n",
		    __func__, error);
		return 0;
	}

	/* read all registers: */
	uint8_t reg = dm->dm_rtc_start;
	error = iic_exec(sc->sc_tag, I2C_OP_READ_WITH_STOP, sc->sc_address,
	     &reg, 1, buf, 4, 0);

	/* Done with I2C */
	iic_release_bus(sc->sc_tag, 0);

	if (error != 0) {
		aprint_error_dev(sc->sc_dev,
		    "%s: failed to read rtc at 0x%x: %d\n",
		    __func__, reg, error);
		return 0;
	}

	uint32_t v = (buf[3] << 24) | (buf[2] << 16) | (buf[1] << 8) | buf[0];
	*tp = v;

	aprint_debug_dev(sc->sc_dev, "%s: cntr=0x%08"PRIx32"\n",
	    __func__, v);

	return 1;
}

static int
dsrtc_clock_write_timeval(struct dsrtc_softc *sc, time_t t)
{
	const struct dsrtc_model * const dm = &sc->sc_model;
	size_t buflen = dm->dm_rtc_size + 2;
	/* XXX: the biggest dm_rtc_size we have now is 7, so we should be ok */ 
	uint8_t buf[16];
	int error;

	KASSERT((dm->dm_flags & DSRTC_FLAG_CLOCK_HOLD) == 0);
	KASSERT(dm->dm_ch_reg == dm->dm_rtc_start + 4);

	buf[0] = dm->dm_rtc_start;
	buf[1] = (t >> 0) & 0xff;
	buf[2] = (t >> 8) & 0xff;
	buf[3] = (t >> 16) & 0xff;
	buf[4] = (t >> 24) & 0xff;
	buf[5] = 0;

	if ((error = iic_acquire_bus(sc->sc_tag, 0)) != 0) {
		aprint_error_dev(sc->sc_dev,
		    "%s: failed to acquire I2C bus: %d\n",
		    __func__, error);
		return 0;
	}

	error = iic_exec(sc->sc_tag, I2C_OP_WRITE_WITH_STOP, sc->sc_address,
	    &buf, buflen, NULL, 0, 0);

	/* Done with I2C */
	iic_release_bus(sc->sc_tag, 0);

	/* send data */
	if (error != 0) {
		aprint_error_dev(sc->sc_dev,
		    "%s: failed to set time: %d\n",
		    __func__, error);
		return 0;
	}

	return 1;
}

static int
dsrtc_read_temp(struct dsrtc_softc *sc, uint32_t *temp)
{
	int error, tc;
	uint8_t reg = DS3232_TEMP_MSB;
	uint8_t buf[2];

	if ((sc->sc_model.dm_flags & DSRTC_FLAG_TEMP) == 0)
		return ENOTSUP;

	if ((error = iic_acquire_bus(sc->sc_tag, 0)) != 0) {
		aprint_error_dev(sc->sc_dev,
		    "%s: failed to acquire I2C bus: %d\n",
		    __func__, error);
		return 0;
	}

	/* read temperature registers: */
	error = iic_exec(sc->sc_tag, I2C_OP_READ_WITH_STOP, sc->sc_address,
	     &reg, 1, buf, 2, 0);

	/* Done with I2C */
	iic_release_bus(sc->sc_tag, 0);

	if (error != 0) {
		aprint_error_dev(sc->sc_dev,
		    "%s: failed to read temperature: %d\n",
		    __func__, error);
		return 0;
	}

	/* convert to microkelvin */
	tc = buf[0] * 1000000 + (buf[1] >> 6) * 250000;
	*temp = tc + 273150000;
	return 1;
}

static void
dsrtc_refresh(struct sysmon_envsys *sme, envsys_data_t *edata)
{
	struct dsrtc_softc *sc = sme->sme_cookie;
	uint32_t temp = 0;	/* XXX gcc */

	if (dsrtc_read_temp(sc, &temp) == 0) {
		edata->state = ENVSYS_SINVALID;
		return;
	}

	edata->value_cur = temp;

	edata->state = ENVSYS_SVALID;
}
