/*	$NetBSD: if_loop.c,v 1.119 2025/09/21 15:11:52 christos Exp $	*/

/*
 * Copyright (C) 1995, 1996, 1997, and 1998 WIDE Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * Copyright (c) 1982, 1986, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)if_loop.c	8.2 (Berkeley) 1/9/95
 */

/*
 * Loopback interface driver for protocol testing and timing.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: if_loop.c,v 1.119 2025/09/21 15:11:52 christos Exp $");

#ifdef _KERNEL_OPT
#include "opt_inet.h"
#include "opt_atalk.h"
#include "opt_mbuftrace.h"
#include "opt_mpls.h"
#include "opt_net_mpsafe.h"
#endif

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/errno.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/device.h>
#include <sys/module.h>

#include <sys/cpu.h>

#include <net/if.h>
#include <net/if_types.h>
#include <net/route.h>

#ifdef	INET
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/in_offload.h>
#include <netinet/ip.h>
#include <netinet/ip_var.h>
#endif

#ifdef INET6
#ifndef INET
#include <netinet/in.h>
#endif
#include <netinet6/in6_var.h>
#include <netinet6/in6_offload.h>
#include <netinet/ip6.h>
#endif

#ifdef MPLS
#include <netmpls/mpls.h>
#include <netmpls/mpls_var.h>
#endif

#ifdef NETATALK
#include <netatalk/at.h>
#include <netatalk/at_var.h>
#endif

#include <net/bpf.h>

#if defined(LARGE_LOMTU)
#define LOMTU	(131072 +  MHLEN + MLEN)
#define LOMTU_MAX LOMTU
#else
#define	LOMTU	(32768 +  MHLEN + MLEN)
#define	LOMTU_MAX	(65536 +  MHLEN + MLEN)
#endif

#ifdef ALTQ
static void	lostart(struct ifnet *);
#endif

static int	loop_clone_create(struct if_clone *, int);
static int	loop_clone_destroy(struct ifnet *);

static void	loop_rtrequest(int, struct rtentry *, const struct rt_addrinfo *);

static struct if_clone loop_cloner =
    IF_CLONE_INITIALIZER("lo", loop_clone_create, loop_clone_destroy);

void
loopattach(int n)
{

#ifndef _MODULE
	loop_clone_create(&loop_cloner, 0);	/* lo0 always exists */
#endif
}

void
loopinit(void)
{

	if (lo0ifp != NULL)	/* can happen in rump kernel */
		return;

#ifdef _MODULE
	loop_clone_create(&loop_cloner, 0);	/* lo0 always exists */
#endif
	if_clone_attach(&loop_cloner);
}

static int
loopdetach(void)
{
	/* no detach for now; we don't allow lo0 to be deleted */
	return EBUSY;
}

static int
loop_clone_create(struct if_clone *ifc, int unit)
{
	struct ifnet *ifp;

	ifp = if_alloc(IFT_LOOP);

	if_initname(ifp, ifc->ifc_name, unit);

	ifp->if_mtu = LOMTU;
	ifp->if_flags = IFF_LOOPBACK | IFF_MULTICAST;
#ifdef NET_MPSAFE
	ifp->if_extflags = IFEF_MPSAFE;
#endif
	ifp->if_ioctl = loioctl;
	ifp->if_output = looutput;
#ifdef ALTQ
	ifp->if_start = lostart;
#endif
	ifp->if_type = IFT_LOOP;
	ifp->if_hdrlen = 0;
	ifp->if_addrlen = 0;
	ifp->if_dlt = DLT_NULL;
	IFQ_SET_READY(&ifp->if_snd);
	if (unit == 0)
		lo0ifp = ifp;
	if_initialize(ifp);
	ifp->if_link_state = LINK_STATE_UP;
	if_alloc_sadl(ifp);
	bpf_attach(ifp, DLT_NULL, sizeof(u_int));
#ifdef MBUFTRACE
	ifp->if_mowner = malloc(sizeof(struct mowner), M_DEVBUF,
	    M_WAITOK | M_ZERO);
	strlcpy(ifp->if_mowner->mo_name, ifp->if_xname,
	    sizeof(ifp->if_mowner->mo_name));
	MOWNER_ATTACH(ifp->if_mowner);
#endif

	ifp->if_flags |= IFF_RUNNING;
	if_register(ifp);

	return (0);
}

static int
loop_clone_destroy(struct ifnet *ifp)
{

	if (ifp == lo0ifp)
		return (EPERM);

	ifp->if_flags &= ~IFF_RUNNING;

#ifdef MBUFTRACE
	MOWNER_DETACH(ifp->if_mowner);
	free(ifp->if_mowner, M_DEVBUF);
#endif

	bpf_detach(ifp);
	if_detach(ifp);

	if_free(ifp);

	return (0);
}

int
looutput(struct ifnet *ifp, struct mbuf *m, const struct sockaddr *dst,
    const struct rtentry *rt)
{
	pktqueue_t *pktq = NULL;
	int s;
	int csum_flags;
	int error = 0;
	size_t pktlen;

	MCLAIM(m, ifp->if_mowner);

	KERNEL_LOCK_UNLESS_NET_MPSAFE();

	if ((m->m_flags & M_PKTHDR) == 0)
		panic("looutput: no header mbuf");
	if (ifp->if_flags & IFF_LOOPBACK)
		bpf_mtap_af(ifp, dst->sa_family, m, BPF_D_OUT);
	m_set_rcvif(m, ifp);

	if (rt && rt->rt_flags & (RTF_REJECT|RTF_BLACKHOLE)) {
		m_freem(m);
		error = (rt->rt_flags & RTF_BLACKHOLE ? 0 :
			rt->rt_flags & RTF_HOST ? EHOSTUNREACH : ENETUNREACH);
		goto out;
	}

	pktlen = m->m_pkthdr.len;

	if_statadd2(ifp, if_opackets, 1, if_obytes, pktlen);

#ifdef ALTQ
	/*
	 * ALTQ on the loopback interface is just for debugging.  It's
	 * used only for loopback interfaces, not for a simplex interface.
	 */
	if ((ALTQ_IS_ENABLED(&ifp->if_snd) || TBR_IS_ENABLED(&ifp->if_snd)) &&
	    ifp->if_start == lostart) {
		/*
		 * If the queueing discipline needs packet classification,
		 * do it before prepending the link headers.
		 */
		IFQ_CLASSIFY(&ifp->if_snd, m, dst->sa_family);

		M_PREPEND(m, sizeof(uint32_t), M_DONTWAIT);
		if (m == NULL) {
			if_statinc(ifp, if_oerrors);
			error = ENOBUFS;
			goto out;
		}
		*(mtod(m, uint32_t *)) = dst->sa_family;

		error = if_transmit_lock(ifp, m);
		goto out;
	}
#endif /* ALTQ */

	m_tag_delete_chain(m);

#ifdef MPLS
	bool is_mpls = false;
	if (rt != NULL && rt_gettag(rt) != NULL &&
	    rt_gettag(rt)->sa_family == AF_MPLS &&
	    (m->m_flags & (M_MCAST | M_BCAST)) == 0) {
		union mpls_shim msh;
		msh.s_addr = MPLS_GETSADDR(rt);
		if (msh.shim.label != MPLS_LABEL_IMPLNULL) {
			is_mpls = true;
			pktq = mpls_pktq;
		}
	}
	if (!is_mpls)
#endif
	switch (dst->sa_family) {

#ifdef INET
	case AF_INET:
		csum_flags = m->m_pkthdr.csum_flags;
		KASSERT((csum_flags & ~(M_CSUM_IPv4|M_CSUM_UDPv4)) == 0);
		if (csum_flags != 0 && IN_LOOPBACK_NEED_CHECKSUM(csum_flags)) {
			in_undefer_cksum(m, 0, csum_flags);
			m->m_pkthdr.csum_flags = 0;
		} else {
			/*
			 * Do nothing. Pass M_CSUM_IPv4 and M_CSUM_UDPv4 as
			 * they are to tell those are calculated and good.
			 */
		}
		pktq = ip_pktq;
		break;
#endif
#ifdef INET6
	case AF_INET6:
		csum_flags = m->m_pkthdr.csum_flags;
		KASSERT((csum_flags & ~M_CSUM_UDPv6) == 0);
		if (csum_flags != 0 &&
		    IN6_LOOPBACK_NEED_CHECKSUM(csum_flags)) {
			in6_undefer_cksum(m, 0, csum_flags);
			m->m_pkthdr.csum_flags = 0;
		} else {
			/*
			 * Do nothing. Pass M_CSUM_UDPv6 as
			 * they are to tell those are calculated and good.
			 */
		}
		m->m_flags |= M_LOOP;
		pktq = ip6_pktq;
		break;
#endif
#ifdef NETATALK
	case AF_APPLETALK:
		pktq = at_pktq2;
		break;
#endif
	default:
		rt_unhandled(__func__, ifp, dst);
		m_freem(m);
		error = EAFNOSUPPORT;
		goto out;
	}

	KASSERT(pktq != NULL);

	error = 0;
	s = splnet();
	if (__predict_true(pktq_enqueue(pktq, m, 0))) {
		if_statadd2(ifp, if_ipackets, 1, if_ibytes, pktlen);
	} else {
		m_freem(m);
		if_statinc(ifp, if_oerrors);
		error = ENOBUFS;
	}
	splx(s);
out:
	KERNEL_UNLOCK_UNLESS_NET_MPSAFE();
	return error;
}

#ifdef ALTQ
static void
lostart(struct ifnet *ifp)
{
	for (;;) {
		pktqueue_t *pktq = NULL;
		struct mbuf *m;
		struct sockaddr sa;
		size_t pktlen;
		uint32_t af;
		int s;

		IFQ_DEQUEUE(&ifp->if_snd, m);
		if (m == NULL)
			return;

		af = *(mtod(m, uint32_t *));
		m_adj(m, sizeof(uint32_t));

		switch (af) {
#ifdef INET
		case AF_INET:
			pktq = ip_pktq;
			break;
#endif
#ifdef INET6
		case AF_INET6:
			m->m_flags |= M_LOOP;
			pktq = ip6_pktq;
			break;
#endif
#ifdef NETATALK
		case AF_APPLETALK:
			pktq = at_pktq2;
			break;
#endif
		default:
			memset(&sa, 0, sizeof(sa));
			sa.sa_family = af;
			rt_unhandled(__func__, ifp, &sa);
			m_freem(m);
			return;
		}
		pktlen = m->m_pkthdr.len;

		KASSERT(pktq != NULL);

		s = splnet();
		if (__predict_false(pktq_enqueue(pktq, m, 0))) {
			m_freem(m);
			splx(s);
			return;
		}
		if_statadd2(ifp, if_ipackets, 1, if_ibytes, pktlen);
		splx(s);
	}
}
#endif /* ALTQ */

/* ARGSUSED */
static void
loop_rtrequest(int cmd, struct rtentry *rt,
    const struct rt_addrinfo *info)
{

	if (rt)
		rt->rt_rmx.rmx_mtu = lo0ifp->if_mtu;
}

/*
 * Process an ioctl request.
 */
/* ARGSUSED */
int
loioctl(struct ifnet *ifp, u_long cmd, void *data)
{
	struct ifaddr *ifa;
	struct ifreq *ifr = data;
	int error = 0;

	switch (cmd) {

	case SIOCINITIFADDR:
		ifp->if_flags |= IFF_UP;
		ifa = (struct ifaddr *)data;
		if (ifa != NULL)
			ifa->ifa_rtrequest = loop_rtrequest;
		/*
		 * Everything else is done at a higher level.
		 */
		break;

	case SIOCSIFMTU:
		if ((unsigned)ifr->ifr_mtu > LOMTU_MAX)
			error = EINVAL;
		else if ((error = ifioctl_common(ifp, cmd, data)) == ENETRESET){
			error = 0;
		}
		break;

	case SIOCADDMULTI:
	case SIOCDELMULTI:
		if (ifr == NULL) {
			error = EAFNOSUPPORT;		/* XXX */
			break;
		}
		switch (ifreq_getaddr(cmd, ifr)->sa_family) {

#ifdef INET
		case AF_INET:
			break;
#endif
#ifdef INET6
		case AF_INET6:
			break;
#endif

		default:
			error = EAFNOSUPPORT;
			break;
		}
		break;

	default:
		error = ifioctl_common(ifp, cmd, data);
	}
	return (error);
}

/*
 * Module infrastructure
 */
#include "if_module.h"

IF_MODULE(MODULE_CLASS_DRIVER, loop, NULL)
