/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *   * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */

package org.apache.synapse.commons.evaluators.source;

import org.apache.synapse.commons.evaluators.EvaluatorContext;
import org.apache.synapse.commons.evaluators.EvaluatorException;
import org.apache.axiom.soap.SOAPEnvelope;
import org.apache.axiom.om.xpath.AXIOMXPath;
import org.apache.axiom.om.OMAttribute;
import org.apache.axiom.om.OMDocument;
import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMText;
import org.jaxen.JaxenException;

import java.util.List;

public class SOAPEnvelopeTextRetriever implements SourceTextRetriever {

    private String source;
    private AXIOMXPath compiledXPath;

    public SOAPEnvelopeTextRetriever(String source) {
        this.source = source;
    }

    @Override
    public String getSourceText(EvaluatorContext context) throws EvaluatorException {
        SOAPEnvelope envelope = context.getMessageContext().getEnvelope();
        Object result;
        
        try {
            if (compiledXPath == null) {
                compiledXPath = new AXIOMXPath(source);
            }
            result = compiledXPath.evaluate(envelope);
        } catch (JaxenException e) {
            throw new EvaluatorException("Error while parsing the XPath expression: " + source, e);
        }

        if (result instanceof List) {
            List list = (List) result;
            if (list.size() == 1 && list.get(0) == null) {
                return null;
            }

            StringBuffer textValue = new StringBuffer();
            for (Object o : list) {
                if (o instanceof OMText) {
                    textValue.append(((OMText) o).getText());

                } else if (o instanceof OMElement) {
                    String s = ((OMElement) o).getText();
                    if (s.trim().length() == 0) {
                        s = o.toString();
                    }
                    textValue.append(s);

                } else if (o instanceof OMDocument) {
                    textValue.append(((OMDocument) o).getOMDocumentElement().toString());
                } else if (o instanceof OMAttribute) {
                    textValue.append(((OMAttribute) o).getAttributeValue());
                }
            }

            return textValue.toString();

        } else {
            return result.toString();
        }
    }

    @Override
    public String getSource() {
        return source;
    }
}
