# Copyright 2015 Google Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

require 'google/apis/core/base_service'
require 'google/apis/core/json_representation'
require 'google/apis/core/hashable'
require 'google/apis/errors'

module Google
  module Apis
    module ComputeBeta
      # Compute Engine API
      #
      # Creates and runs virtual machines on Google Cloud Platform.
      #
      # @example
      #    require 'google/apis/compute_beta'
      #
      #    Compute = Google::Apis::ComputeBeta # Alias the module
      #    service = Compute::ComputeService.new
      #
      # @see https://developers.google.com/compute/docs/reference/latest/
      class ComputeService < Google::Apis::Core::BaseService
        # @return [String]
        #  API key. Your API key identifies your project and provides you with API access,
        #  quota, and reports. Required unless you provide an OAuth 2.0 token.
        attr_accessor :key

        # @return [String]
        #  An opaque string that represents a user for quota purposes. Must not exceed 40
        #  characters.
        attr_accessor :quota_user

        # @return [String]
        #  Deprecated. Please use quotaUser instead.
        attr_accessor :user_ip

        def initialize
          super('https://www.googleapis.com/', 'compute/beta/projects/')
          @batch_path = 'batch/compute/beta'
        end
        
        # Retrieves an aggregated list of accelerator types.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::AcceleratorTypeAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::AcceleratorTypeAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_accelerator_type_list(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/acceleratorTypes', options)
          command.response_representation = Google::Apis::ComputeBeta::AcceleratorTypeAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::AcceleratorTypeAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified accelerator type.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] accelerator_type
        #   Name of the accelerator type to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::AcceleratorType] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::AcceleratorType]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_accelerator_type(project, zone, accelerator_type, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/acceleratorTypes/{acceleratorType}', options)
          command.response_representation = Google::Apis::ComputeBeta::AcceleratorType::Representation
          command.response_class = Google::Apis::ComputeBeta::AcceleratorType
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['acceleratorType'] = accelerator_type unless accelerator_type.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of accelerator types available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::AcceleratorTypeList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::AcceleratorTypeList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_accelerator_types(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/acceleratorTypes', options)
          command.response_representation = Google::Apis::ComputeBeta::AcceleratorTypeList::Representation
          command.response_class = Google::Apis::ComputeBeta::AcceleratorTypeList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of addresses.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::AddressAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::AddressAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_addresses(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/addresses', options)
          command.response_representation = Google::Apis::ComputeBeta::AddressAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::AddressAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified address resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] address
        #   Name of the address resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_address(project, region, address, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/regions/{region}/addresses/{address}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['address'] = address unless address.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified address resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] address
        #   Name of the address resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Address] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Address]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_address(project, region, address, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/addresses/{address}', options)
          command.response_representation = Google::Apis::ComputeBeta::Address::Representation
          command.response_class = Google::Apis::ComputeBeta::Address
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['address'] = address unless address.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an address resource in the specified project using the data included
        # in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [Google::Apis::ComputeBeta::Address] address_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_address(project, region, address_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/addresses', options)
          command.request_representation = Google::Apis::ComputeBeta::Address::Representation
          command.request_object = address_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of addresses contained within the specified region.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::AddressList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::AddressList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_addresses(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/addresses', options)
          command.response_representation = Google::Apis::ComputeBeta::AddressList::Representation
          command.response_class = Google::Apis::ComputeBeta::AddressList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on an Address. To learn more about labels, read the Labeling
        # Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The region for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::RegionSetLabelsRequest] region_set_labels_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_address_labels(project, region, resource, region_set_labels_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/addresses/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeBeta::RegionSetLabelsRequest::Representation
          command.request_object = region_set_labels_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_address_iam_permissions(project, region, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/addresses/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of autoscalers.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::AutoscalerAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::AutoscalerAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_autoscalers(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/autoscalers', options)
          command.response_representation = Google::Apis::ComputeBeta::AutoscalerAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::AutoscalerAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified autoscaler.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [String] autoscaler
        #   Name of the autoscaler to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_autoscaler(project, zone, autoscaler, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/zones/{zone}/autoscalers/{autoscaler}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified autoscaler resource. Gets a list of available
        # autoscalers by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [String] autoscaler
        #   Name of the autoscaler to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Autoscaler] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Autoscaler]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_autoscaler(project, zone, autoscaler, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/autoscalers/{autoscaler}', options)
          command.response_representation = Google::Apis::ComputeBeta::Autoscaler::Representation
          command.response_class = Google::Apis::ComputeBeta::Autoscaler
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an autoscaler in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [Google::Apis::ComputeBeta::Autoscaler] autoscaler_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_autoscaler(project, zone, autoscaler_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/autoscalers', options)
          command.request_representation = Google::Apis::ComputeBeta::Autoscaler::Representation
          command.request_object = autoscaler_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of autoscalers contained within the specified zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::AutoscalerList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::AutoscalerList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_autoscalers(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/autoscalers', options)
          command.response_representation = Google::Apis::ComputeBeta::AutoscalerList::Representation
          command.response_class = Google::Apis::ComputeBeta::AutoscalerList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an autoscaler in the specified project using the data included in the
        # request. This method supports PATCH semantics and uses the JSON merge patch
        # format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [Google::Apis::ComputeBeta::Autoscaler] autoscaler_object
        # @param [String] autoscaler
        #   Name of the autoscaler to patch.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_autoscaler(project, zone, autoscaler_object = nil, autoscaler: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:patch, '{project}/zones/{zone}/autoscalers', options)
          command.request_representation = Google::Apis::ComputeBeta::Autoscaler::Representation
          command.request_object = autoscaler_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_autoscaler_iam_permissions(project, zone, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/autoscalers/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an autoscaler in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [Google::Apis::ComputeBeta::Autoscaler] autoscaler_object
        # @param [String] autoscaler
        #   Name of the autoscaler to update.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_autoscaler(project, zone, autoscaler_object = nil, autoscaler: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:put, '{project}/zones/{zone}/autoscalers', options)
          command.request_representation = Google::Apis::ComputeBeta::Autoscaler::Representation
          command.request_object = autoscaler_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds the given Signed URL Key to the backend bucket.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to which the Signed URL Key should be added.
        #   The name should conform to RFC1035.
        # @param [Google::Apis::ComputeBeta::SignedUrlKey] signed_url_key_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_backend_bucket_signed_url_key(project, backend_bucket, signed_url_key_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/backendBuckets/{backendBucket}/addSignedUrlKey', options)
          command.request_representation = Google::Apis::ComputeBeta::SignedUrlKey::Representation
          command.request_object = signed_url_key_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified BackendBucket resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_backend_bucket(project, backend_bucket, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/global/backendBuckets/{backendBucket}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the given Signed URL Key from the backend bucket.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to which the Signed URL Key should be added.
        #   The name should conform to RFC1035.
        # @param [String] key_name
        #   The name of the Signed URL Key to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_backend_bucket_signed_url_key(project, backend_bucket, key_name, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/backendBuckets/{backendBucket}/deleteSignedUrlKey', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['keyName'] = key_name unless key_name.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified BackendBucket resource. Gets a list of available backend
        # buckets by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::BackendBucket] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::BackendBucket]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_backend_bucket(project, backend_bucket, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/backendBuckets/{backendBucket}', options)
          command.response_representation = Google::Apis::ComputeBeta::BackendBucket::Representation
          command.response_class = Google::Apis::ComputeBeta::BackendBucket
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a BackendBucket resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::BackendBucket] backend_bucket_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_backend_bucket(project, backend_bucket_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/backendBuckets', options)
          command.request_representation = Google::Apis::ComputeBeta::BackendBucket::Representation
          command.request_object = backend_bucket_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of BackendBucket resources available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::BackendBucketList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::BackendBucketList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_backend_buckets(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/backendBuckets', options)
          command.response_representation = Google::Apis::ComputeBeta::BackendBucketList::Representation
          command.response_class = Google::Apis::ComputeBeta::BackendBucketList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified BackendBucket resource with the data included in the
        # request. This method supports PATCH semantics and uses the JSON merge patch
        # format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to patch.
        # @param [Google::Apis::ComputeBeta::BackendBucket] backend_bucket_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_backend_bucket(project, backend_bucket, backend_bucket_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:patch, '{project}/global/backendBuckets/{backendBucket}', options)
          command.request_representation = Google::Apis::ComputeBeta::BackendBucket::Representation
          command.request_object = backend_bucket_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified BackendBucket resource with the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to update.
        # @param [Google::Apis::ComputeBeta::BackendBucket] backend_bucket_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_backend_bucket(project, backend_bucket, backend_bucket_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:put, '{project}/global/backendBuckets/{backendBucket}', options)
          command.request_representation = Google::Apis::ComputeBeta::BackendBucket::Representation
          command.request_object = backend_bucket_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds the given Signed URL Key to the specified backend service.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to which the Signed URL Key should be
        #   added. The name should conform to RFC1035.
        # @param [Google::Apis::ComputeBeta::SignedUrlKey] signed_url_key_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_backend_service_signed_url_key(project, backend_service, signed_url_key_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/backendServices/{backendService}/addSignedUrlKey', options)
          command.request_representation = Google::Apis::ComputeBeta::SignedUrlKey::Representation
          command.request_object = signed_url_key_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of all BackendService resources, regional and global,
        # available to the specified project.
        # @param [String] project
        #   Name of the project scoping this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::BackendServiceAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::BackendServiceAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_backend_service_list(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/backendServices', options)
          command.response_representation = Google::Apis::ComputeBeta::BackendServiceAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::BackendServiceAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified BackendService resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_backend_service(project, backend_service, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/global/backendServices/{backendService}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the given Signed URL Key from the specified backend service.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to which the Signed URL Key should be
        #   added. The name should conform to RFC1035.
        # @param [String] key_name
        #   The name of the Signed URL Key to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_backend_service_signed_url_key(project, backend_service, key_name, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/backendServices/{backendService}/deleteSignedUrlKey', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['keyName'] = key_name unless key_name.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified BackendService resource. Gets a list of available
        # backend services by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::BackendService] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::BackendService]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_backend_service(project, backend_service, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/backendServices/{backendService}', options)
          command.response_representation = Google::Apis::ComputeBeta::BackendService::Representation
          command.response_class = Google::Apis::ComputeBeta::BackendService
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the most recent health check results for this BackendService.
        # @param [String] project
        # @param [String] backend_service
        #   Name of the BackendService resource to which the queried instance belongs.
        # @param [Google::Apis::ComputeBeta::ResourceGroupReference] resource_group_reference_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::BackendServiceGroupHealth] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::BackendServiceGroupHealth]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_backend_service_health(project, backend_service, resource_group_reference_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/backendServices/{backendService}/getHealth', options)
          command.request_representation = Google::Apis::ComputeBeta::ResourceGroupReference::Representation
          command.request_object = resource_group_reference_object
          command.response_representation = Google::Apis::ComputeBeta::BackendServiceGroupHealth::Representation
          command.response_class = Google::Apis::ComputeBeta::BackendServiceGroupHealth
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a BackendService resource in the specified project using the data
        # included in the request. There are several restrictions and guidelines to keep
        # in mind when creating a backend service. Read  Restrictions and Guidelines for
        # more information.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::BackendService] backend_service_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_backend_service(project, backend_service_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/backendServices', options)
          command.request_representation = Google::Apis::ComputeBeta::BackendService::Representation
          command.request_object = backend_service_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of BackendService resources available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::BackendServiceList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::BackendServiceList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_backend_services(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/backendServices', options)
          command.response_representation = Google::Apis::ComputeBeta::BackendServiceList::Representation
          command.response_class = Google::Apis::ComputeBeta::BackendServiceList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Patches the specified BackendService resource with the data included in the
        # request. There are several restrictions and guidelines to keep in mind when
        # updating a backend service. Read  Restrictions and Guidelines for more
        # information. This method supports PATCH semantics and uses the JSON merge
        # patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to patch.
        # @param [Google::Apis::ComputeBeta::BackendService] backend_service_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_backend_service(project, backend_service, backend_service_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:patch, '{project}/global/backendServices/{backendService}', options)
          command.request_representation = Google::Apis::ComputeBeta::BackendService::Representation
          command.request_object = backend_service_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the security policy for the specified backend service.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to which the security policy should be set.
        #   The name should conform to RFC1035.
        # @param [Google::Apis::ComputeBeta::SecurityPolicyReference] security_policy_reference_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_backend_service_security_policy(project, backend_service, security_policy_reference_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/backendServices/{backendService}/setSecurityPolicy', options)
          command.request_representation = Google::Apis::ComputeBeta::SecurityPolicyReference::Representation
          command.request_object = security_policy_reference_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_backend_service_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/backendServices/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified BackendService resource with the data included in the
        # request. There are several restrictions and guidelines to keep in mind when
        # updating a backend service. Read  Restrictions and Guidelines for more
        # information.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to update.
        # @param [Google::Apis::ComputeBeta::BackendService] backend_service_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_backend_service(project, backend_service, backend_service_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:put, '{project}/global/backendServices/{backendService}', options)
          command.request_representation = Google::Apis::ComputeBeta::BackendService::Representation
          command.request_object = backend_service_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of disk types.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::DiskTypeAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::DiskTypeAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_disk_types(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/diskTypes', options)
          command.response_representation = Google::Apis::ComputeBeta::DiskTypeAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::DiskTypeAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified disk type. Gets a list of available disk types by making
        # a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk_type
        #   Name of the disk type to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::DiskType] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::DiskType]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_disk_type(project, zone, disk_type, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/diskTypes/{diskType}', options)
          command.response_representation = Google::Apis::ComputeBeta::DiskType::Representation
          command.response_class = Google::Apis::ComputeBeta::DiskType
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['diskType'] = disk_type unless disk_type.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of disk types available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::DiskTypeList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::DiskTypeList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_disk_types(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/diskTypes', options)
          command.response_representation = Google::Apis::ComputeBeta::DiskTypeList::Representation
          command.response_class = Google::Apis::ComputeBeta::DiskTypeList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of persistent disks.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::DiskAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::DiskAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_disk(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/disks', options)
          command.response_representation = Google::Apis::ComputeBeta::DiskAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::DiskAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a snapshot of a specified persistent disk.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   Name of the persistent disk to snapshot.
        # @param [Google::Apis::ComputeBeta::Snapshot] snapshot_object
        # @param [Boolean] guest_flush
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_disk_snapshot(project, zone, disk, snapshot_object = nil, guest_flush: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/disks/{disk}/createSnapshot', options)
          command.request_representation = Google::Apis::ComputeBeta::Snapshot::Representation
          command.request_object = snapshot_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['guestFlush'] = guest_flush unless guest_flush.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified persistent disk. Deleting a disk removes its data
        # permanently and is irreversible. However, deleting a disk does not delete any
        # snapshots previously made from the disk. You must separately delete snapshots.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   Name of the persistent disk to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_disk(project, zone, disk, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/zones/{zone}/disks/{disk}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns a specified persistent disk. Gets a list of available persistent disks
        # by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   Name of the persistent disk to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Disk] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Disk]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_disk(project, zone, disk, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/disks/{disk}', options)
          command.response_representation = Google::Apis::ComputeBeta::Disk::Representation
          command.response_class = Google::Apis::ComputeBeta::Disk
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a persistent disk in the specified project using the data in the
        # request. You can create a disk with a sourceImage, a sourceSnapshot, or create
        # an empty 500 GB data disk by omitting all properties. You can also create a
        # disk that is larger than the default size by specifying the sizeGb property.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [Google::Apis::ComputeBeta::Disk] disk_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] source_image
        #   Optional. Source image to restore onto a disk.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_disk(project, zone, disk_object = nil, request_id: nil, source_image: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/disks', options)
          command.request_representation = Google::Apis::ComputeBeta::Disk::Representation
          command.request_object = disk_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['sourceImage'] = source_image unless source_image.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of persistent disks contained within the specified zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::DiskList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::DiskList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_disks(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/disks', options)
          command.response_representation = Google::Apis::ComputeBeta::DiskList::Representation
          command.response_class = Google::Apis::ComputeBeta::DiskList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Resizes the specified persistent disk. You can only increase the size of the
        # disk.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   The name of the persistent disk.
        # @param [Google::Apis::ComputeBeta::DisksResizeRequest] disks_resize_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def resize_disk(project, zone, disk, disks_resize_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/disks/{disk}/resize', options)
          command.request_representation = Google::Apis::ComputeBeta::DisksResizeRequest::Representation
          command.request_object = disks_resize_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on a disk. To learn more about labels, read the Labeling
        # Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::ZoneSetLabelsRequest] zone_set_labels_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_disk_labels(project, zone, resource, zone_set_labels_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/disks/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeBeta::ZoneSetLabelsRequest::Representation
          command.request_object = zone_set_labels_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_disk_iam_permissions(project, zone, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/disks/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified firewall.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] firewall
        #   Name of the firewall rule to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_firewall(project, firewall, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/global/firewalls/{firewall}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['firewall'] = firewall unless firewall.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified firewall.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] firewall
        #   Name of the firewall rule to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Firewall] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Firewall]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_firewall(project, firewall, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/firewalls/{firewall}', options)
          command.response_representation = Google::Apis::ComputeBeta::Firewall::Representation
          command.response_class = Google::Apis::ComputeBeta::Firewall
          command.params['project'] = project unless project.nil?
          command.params['firewall'] = firewall unless firewall.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a firewall rule in the specified project using the data included in
        # the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::Firewall] firewall_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_firewall(project, firewall_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/firewalls', options)
          command.request_representation = Google::Apis::ComputeBeta::Firewall::Representation
          command.request_object = firewall_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of firewall rules available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::FirewallList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::FirewallList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_firewalls(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/firewalls', options)
          command.response_representation = Google::Apis::ComputeBeta::FirewallList::Representation
          command.response_class = Google::Apis::ComputeBeta::FirewallList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified firewall rule with the data included in the request.
        # This method supports PATCH semantics and uses the JSON merge patch format and
        # processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] firewall
        #   Name of the firewall rule to patch.
        # @param [Google::Apis::ComputeBeta::Firewall] firewall_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_firewall(project, firewall, firewall_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:patch, '{project}/global/firewalls/{firewall}', options)
          command.request_representation = Google::Apis::ComputeBeta::Firewall::Representation
          command.request_object = firewall_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['firewall'] = firewall unless firewall.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_firewall_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/firewalls/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified firewall rule with the data included in the request. The
        # PUT method can only update the following fields of firewall rule: allowed,
        # description, sourceRanges, sourceTags, targetTags.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] firewall
        #   Name of the firewall rule to update.
        # @param [Google::Apis::ComputeBeta::Firewall] firewall_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_firewall(project, firewall, firewall_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:put, '{project}/global/firewalls/{firewall}', options)
          command.request_representation = Google::Apis::ComputeBeta::Firewall::Representation
          command.request_object = firewall_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['firewall'] = firewall unless firewall.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of forwarding rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::ForwardingRuleAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::ForwardingRuleAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_forwarding_rules(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/forwardingRules', options)
          command.response_representation = Google::Apis::ComputeBeta::ForwardingRuleAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::ForwardingRuleAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified ForwardingRule resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_forwarding_rule(project, region, forwarding_rule, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/regions/{region}/forwardingRules/{forwardingRule}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified ForwardingRule resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::ForwardingRule] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::ForwardingRule]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_forwarding_rule(project, region, forwarding_rule, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/forwardingRules/{forwardingRule}', options)
          command.response_representation = Google::Apis::ComputeBeta::ForwardingRule::Representation
          command.response_class = Google::Apis::ComputeBeta::ForwardingRule
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a ForwardingRule resource in the specified project and region using
        # the data included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [Google::Apis::ComputeBeta::ForwardingRule] forwarding_rule_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_forwarding_rule(project, region, forwarding_rule_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/forwardingRules', options)
          command.request_representation = Google::Apis::ComputeBeta::ForwardingRule::Representation
          command.request_object = forwarding_rule_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of ForwardingRule resources available to the specified
        # project and region.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::ForwardingRuleList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::ForwardingRuleList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_forwarding_rules(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/forwardingRules', options)
          command.response_representation = Google::Apis::ComputeBeta::ForwardingRuleList::Representation
          command.response_class = Google::Apis::ComputeBeta::ForwardingRuleList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on the specified resource. To learn more about labels, read
        # the Labeling Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The region for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::RegionSetLabelsRequest] region_set_labels_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_forwarding_rule_labels(project, region, resource, region_set_labels_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/forwardingRules/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeBeta::RegionSetLabelsRequest::Representation
          command.request_object = region_set_labels_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Changes target URL for forwarding rule. The new target should be of the same
        # type as the old target.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource in which target is to be set.
        # @param [Google::Apis::ComputeBeta::TargetReference] target_reference_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_forwarding_rule_target(project, region, forwarding_rule, target_reference_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/forwardingRules/{forwardingRule}/setTarget', options)
          command.request_representation = Google::Apis::ComputeBeta::TargetReference::Representation
          command.request_object = target_reference_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_forwarding_rule_iam_permissions(project, region, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/forwardingRules/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified address resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] address
        #   Name of the address resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_global_address(project, address, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/global/addresses/{address}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['address'] = address unless address.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified address resource. Gets a list of available addresses by
        # making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] address
        #   Name of the address resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Address] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Address]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_global_address(project, address, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/addresses/{address}', options)
          command.response_representation = Google::Apis::ComputeBeta::Address::Representation
          command.response_class = Google::Apis::ComputeBeta::Address
          command.params['project'] = project unless project.nil?
          command.params['address'] = address unless address.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an address resource in the specified project using the data included
        # in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::Address] address_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_global_address(project, address_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/addresses', options)
          command.request_representation = Google::Apis::ComputeBeta::Address::Representation
          command.request_object = address_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of global addresses.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::AddressList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::AddressList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_global_addresses(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/addresses', options)
          command.response_representation = Google::Apis::ComputeBeta::AddressList::Representation
          command.response_class = Google::Apis::ComputeBeta::AddressList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on a GlobalAddress. To learn more about labels, read the
        # Labeling Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::GlobalSetLabelsRequest] global_set_labels_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_global_address_labels(project, resource, global_set_labels_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/addresses/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeBeta::GlobalSetLabelsRequest::Representation
          command.request_object = global_set_labels_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_global_address_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/addresses/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified GlobalForwardingRule resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_global_forwarding_rule(project, forwarding_rule, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/global/forwardingRules/{forwardingRule}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified GlobalForwardingRule resource. Gets a list of available
        # forwarding rules by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::ForwardingRule] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::ForwardingRule]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_global_forwarding_rule(project, forwarding_rule, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/forwardingRules/{forwardingRule}', options)
          command.response_representation = Google::Apis::ComputeBeta::ForwardingRule::Representation
          command.response_class = Google::Apis::ComputeBeta::ForwardingRule
          command.params['project'] = project unless project.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a GlobalForwardingRule resource in the specified project using the
        # data included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::ForwardingRule] forwarding_rule_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_global_forwarding_rule(project, forwarding_rule_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/forwardingRules', options)
          command.request_representation = Google::Apis::ComputeBeta::ForwardingRule::Representation
          command.request_object = forwarding_rule_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of GlobalForwardingRule resources available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::ForwardingRuleList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::ForwardingRuleList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_global_forwarding_rules(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/forwardingRules', options)
          command.response_representation = Google::Apis::ComputeBeta::ForwardingRuleList::Representation
          command.response_class = Google::Apis::ComputeBeta::ForwardingRuleList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on the specified resource. To learn more about labels, read
        # the Labeling Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::GlobalSetLabelsRequest] global_set_labels_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_global_forwarding_rule_labels(project, resource, global_set_labels_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/forwardingRules/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeBeta::GlobalSetLabelsRequest::Representation
          command.request_object = global_set_labels_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Changes target URL for the GlobalForwardingRule resource. The new target
        # should be of the same type as the old target.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource in which target is to be set.
        # @param [Google::Apis::ComputeBeta::TargetReference] target_reference_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_global_forwarding_rule_target(project, forwarding_rule, target_reference_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/forwardingRules/{forwardingRule}/setTarget', options)
          command.request_representation = Google::Apis::ComputeBeta::TargetReference::Representation
          command.request_object = target_reference_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_global_forwarding_rule_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/forwardingRules/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of all operations.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::OperationAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::OperationAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_global_operation(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/operations', options)
          command.response_representation = Google::Apis::ComputeBeta::OperationAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::OperationAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified Operations resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] operation
        #   Name of the Operations resource to delete.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [NilClass] No result returned for this method
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [void]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_global_operation(project, operation, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/global/operations/{operation}', options)
          command.params['project'] = project unless project.nil?
          command.params['operation'] = operation unless operation.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the specified Operations resource. Gets a list of operations by
        # making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] operation
        #   Name of the Operations resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_global_operation(project, operation, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/operations/{operation}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['operation'] = operation unless operation.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of Operation resources contained within the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::OperationList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::OperationList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_global_operations(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/operations', options)
          command.response_representation = Google::Apis::ComputeBeta::OperationList::Representation
          command.response_class = Google::Apis::ComputeBeta::OperationList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified HealthCheck resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] health_check
        #   Name of the HealthCheck resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_health_check(project, health_check, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/global/healthChecks/{healthCheck}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['healthCheck'] = health_check unless health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified HealthCheck resource. Gets a list of available health
        # checks by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] health_check
        #   Name of the HealthCheck resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::HealthCheck] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::HealthCheck]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_health_check(project, health_check, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/healthChecks/{healthCheck}', options)
          command.response_representation = Google::Apis::ComputeBeta::HealthCheck::Representation
          command.response_class = Google::Apis::ComputeBeta::HealthCheck
          command.params['project'] = project unless project.nil?
          command.params['healthCheck'] = health_check unless health_check.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a HealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::HealthCheck] health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_health_check(project, health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/healthChecks', options)
          command.request_representation = Google::Apis::ComputeBeta::HealthCheck::Representation
          command.request_object = health_check_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of HealthCheck resources available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::HealthCheckList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::HealthCheckList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_health_checks(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/healthChecks', options)
          command.response_representation = Google::Apis::ComputeBeta::HealthCheckList::Representation
          command.response_class = Google::Apis::ComputeBeta::HealthCheckList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HealthCheck resource in the specified project using the data
        # included in the request. This method supports PATCH semantics and uses the
        # JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] health_check
        #   Name of the HealthCheck resource to patch.
        # @param [Google::Apis::ComputeBeta::HealthCheck] health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_health_check(project, health_check, health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:patch, '{project}/global/healthChecks/{healthCheck}', options)
          command.request_representation = Google::Apis::ComputeBeta::HealthCheck::Representation
          command.request_object = health_check_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['healthCheck'] = health_check unless health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_health_check_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/healthChecks/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] health_check
        #   Name of the HealthCheck resource to update.
        # @param [Google::Apis::ComputeBeta::HealthCheck] health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_health_check(project, health_check, health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:put, '{project}/global/healthChecks/{healthCheck}', options)
          command.request_representation = Google::Apis::ComputeBeta::HealthCheck::Representation
          command.request_object = health_check_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['healthCheck'] = health_check unless health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified HttpHealthCheck resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] http_health_check
        #   Name of the HttpHealthCheck resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_http_health_check(project, http_health_check, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/global/httpHealthChecks/{httpHealthCheck}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpHealthCheck'] = http_health_check unless http_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified HttpHealthCheck resource. Gets a list of available HTTP
        # health checks by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] http_health_check
        #   Name of the HttpHealthCheck resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::HttpHealthCheck] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::HttpHealthCheck]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_http_health_check(project, http_health_check, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/httpHealthChecks/{httpHealthCheck}', options)
          command.response_representation = Google::Apis::ComputeBeta::HttpHealthCheck::Representation
          command.response_class = Google::Apis::ComputeBeta::HttpHealthCheck
          command.params['project'] = project unless project.nil?
          command.params['httpHealthCheck'] = http_health_check unless http_health_check.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a HttpHealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::HttpHealthCheck] http_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_http_health_check(project, http_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/httpHealthChecks', options)
          command.request_representation = Google::Apis::ComputeBeta::HttpHealthCheck::Representation
          command.request_object = http_health_check_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of HttpHealthCheck resources available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::HttpHealthCheckList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::HttpHealthCheckList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_http_health_checks(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/httpHealthChecks', options)
          command.response_representation = Google::Apis::ComputeBeta::HttpHealthCheckList::Representation
          command.response_class = Google::Apis::ComputeBeta::HttpHealthCheckList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HttpHealthCheck resource in the specified project using the data
        # included in the request. This method supports PATCH semantics and uses the
        # JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] http_health_check
        #   Name of the HttpHealthCheck resource to patch.
        # @param [Google::Apis::ComputeBeta::HttpHealthCheck] http_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_http_health_check(project, http_health_check, http_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:patch, '{project}/global/httpHealthChecks/{httpHealthCheck}', options)
          command.request_representation = Google::Apis::ComputeBeta::HttpHealthCheck::Representation
          command.request_object = http_health_check_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpHealthCheck'] = http_health_check unless http_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_http_health_check_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/httpHealthChecks/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HttpHealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] http_health_check
        #   Name of the HttpHealthCheck resource to update.
        # @param [Google::Apis::ComputeBeta::HttpHealthCheck] http_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_http_health_check(project, http_health_check, http_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:put, '{project}/global/httpHealthChecks/{httpHealthCheck}', options)
          command.request_representation = Google::Apis::ComputeBeta::HttpHealthCheck::Representation
          command.request_object = http_health_check_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpHealthCheck'] = http_health_check unless http_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified HttpsHealthCheck resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] https_health_check
        #   Name of the HttpsHealthCheck resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_https_health_check(project, https_health_check, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/global/httpsHealthChecks/{httpsHealthCheck}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpsHealthCheck'] = https_health_check unless https_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified HttpsHealthCheck resource. Gets a list of available
        # HTTPS health checks by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] https_health_check
        #   Name of the HttpsHealthCheck resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::HttpsHealthCheck] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::HttpsHealthCheck]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_https_health_check(project, https_health_check, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/httpsHealthChecks/{httpsHealthCheck}', options)
          command.response_representation = Google::Apis::ComputeBeta::HttpsHealthCheck::Representation
          command.response_class = Google::Apis::ComputeBeta::HttpsHealthCheck
          command.params['project'] = project unless project.nil?
          command.params['httpsHealthCheck'] = https_health_check unless https_health_check.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a HttpsHealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::HttpsHealthCheck] https_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_https_health_check(project, https_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/httpsHealthChecks', options)
          command.request_representation = Google::Apis::ComputeBeta::HttpsHealthCheck::Representation
          command.request_object = https_health_check_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of HttpsHealthCheck resources available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::HttpsHealthCheckList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::HttpsHealthCheckList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_https_health_checks(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/httpsHealthChecks', options)
          command.response_representation = Google::Apis::ComputeBeta::HttpsHealthCheckList::Representation
          command.response_class = Google::Apis::ComputeBeta::HttpsHealthCheckList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HttpsHealthCheck resource in the specified project using the data
        # included in the request. This method supports PATCH semantics and uses the
        # JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] https_health_check
        #   Name of the HttpsHealthCheck resource to patch.
        # @param [Google::Apis::ComputeBeta::HttpsHealthCheck] https_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_https_health_check(project, https_health_check, https_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:patch, '{project}/global/httpsHealthChecks/{httpsHealthCheck}', options)
          command.request_representation = Google::Apis::ComputeBeta::HttpsHealthCheck::Representation
          command.request_object = https_health_check_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpsHealthCheck'] = https_health_check unless https_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_https_health_check_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/httpsHealthChecks/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HttpsHealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] https_health_check
        #   Name of the HttpsHealthCheck resource to update.
        # @param [Google::Apis::ComputeBeta::HttpsHealthCheck] https_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_https_health_check(project, https_health_check, https_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:put, '{project}/global/httpsHealthChecks/{httpsHealthCheck}', options)
          command.request_representation = Google::Apis::ComputeBeta::HttpsHealthCheck::Representation
          command.request_object = https_health_check_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpsHealthCheck'] = https_health_check unless https_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified image.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] image
        #   Name of the image resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_image(project, image, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/global/images/{image}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['image'] = image unless image.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the deprecation status of an image.
        # If an empty request body is given, clears the deprecation status instead.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] image
        #   Image name.
        # @param [Google::Apis::ComputeBeta::DeprecationStatus] deprecation_status_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def deprecate_image(project, image, deprecation_status_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/images/{image}/deprecate', options)
          command.request_representation = Google::Apis::ComputeBeta::DeprecationStatus::Representation
          command.request_object = deprecation_status_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['image'] = image unless image.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified image. Gets a list of available images by making a list()
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] image
        #   Name of the image resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Image] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Image]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_image(project, image, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/images/{image}', options)
          command.response_representation = Google::Apis::ComputeBeta::Image::Representation
          command.response_class = Google::Apis::ComputeBeta::Image
          command.params['project'] = project unless project.nil?
          command.params['image'] = image unless image.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the latest image that is part of an image family and is not deprecated.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] family
        #   Name of the image family to search for.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Image] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Image]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_image_from_family(project, family, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/images/family/{family}', options)
          command.response_representation = Google::Apis::ComputeBeta::Image::Representation
          command.response_class = Google::Apis::ComputeBeta::Image
          command.params['project'] = project unless project.nil?
          command.params['family'] = family unless family.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an image in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::Image] image_object
        # @param [Boolean] force_create
        #   Force image creation if true.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_image(project, image_object = nil, force_create: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/images', options)
          command.request_representation = Google::Apis::ComputeBeta::Image::Representation
          command.request_object = image_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['forceCreate'] = force_create unless force_create.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of custom images available to the specified project. Custom
        # images are images you create that belong to your project. This method does not
        # get any images that belong to other projects, including publicly-available
        # images, like Debian 8. If you want to get a list of publicly-available images,
        # use this method to make a request to the respective image project, such as
        # debian-cloud or windows-cloud.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::ImageList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::ImageList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_images(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/images', options)
          command.response_representation = Google::Apis::ComputeBeta::ImageList::Representation
          command.response_class = Google::Apis::ComputeBeta::ImageList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on an image. To learn more about labels, read the Labeling
        # Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::GlobalSetLabelsRequest] global_set_labels_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_image_labels(project, resource, global_set_labels_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/images/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeBeta::GlobalSetLabelsRequest::Representation
          command.request_object = global_set_labels_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_image_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/images/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Schedules a group action to remove the specified instances from the managed
        # instance group. Abandoning an instance does not delete the instance, but it
        # does remove the instance from any target pools that are applied by the managed
        # instance group. This method reduces the targetSize of the managed instance
        # group by the number of instances that you abandon. This operation is marked as
        # DONE when the action is scheduled even if the instances have not yet been
        # removed from the group. You must separately verify the status of the
        # abandoning action with the listmanagedinstances method.
        # If the group is part of a backend service that has enabled connection draining,
        # it can take up to 60 seconds after the connection draining duration has
        # elapsed before the VM instance is removed or deleted.
        # You can specify a maximum of 1000 instances with this method per request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeBeta::InstanceGroupManagersAbandonInstancesRequest] instance_group_managers_abandon_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def abandon_instance_group_manager_instances(project, zone, instance_group_manager, instance_group_managers_abandon_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/abandonInstances', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceGroupManagersAbandonInstancesRequest::Representation
          command.request_object = instance_group_managers_abandon_instances_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of managed instance groups and groups them by zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InstanceGroupManagerAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InstanceGroupManagerAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_instance_group_managers(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/instanceGroupManagers', options)
          command.response_representation = Google::Apis::ComputeBeta::InstanceGroupManagerAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::InstanceGroupManagerAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified managed instance group and all of the instances in that
        # group. Note that the instance group must not belong to a backend service. Read
        # Deleting an instance group for more information.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance_group_manager(project, zone, instance_group_manager, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Schedules a group action to delete the specified instances in the managed
        # instance group. The instances are also removed from any target pools of which
        # they were a member. This method reduces the targetSize of the managed instance
        # group by the number of instances that you delete. This operation is marked as
        # DONE when the action is scheduled even if the instances are still being
        # deleted. You must separately verify the status of the deleting action with the
        # listmanagedinstances method.
        # If the group is part of a backend service that has enabled connection draining,
        # it can take up to 60 seconds after the connection draining duration has
        # elapsed before the VM instance is removed or deleted.
        # You can specify a maximum of 1000 instances with this method per request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeBeta::InstanceGroupManagersDeleteInstancesRequest] instance_group_managers_delete_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance_group_manager_instances(project, zone, instance_group_manager, instance_group_managers_delete_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/deleteInstances', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceGroupManagersDeleteInstancesRequest::Representation
          command.request_object = instance_group_managers_delete_instances_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns all of the details about the specified managed instance group. Gets a
        # list of available managed instance groups by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InstanceGroupManager] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InstanceGroupManager]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_group_manager(project, zone, instance_group_manager, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}', options)
          command.response_representation = Google::Apis::ComputeBeta::InstanceGroupManager::Representation
          command.response_class = Google::Apis::ComputeBeta::InstanceGroupManager
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a managed instance group using the information that you specify in the
        # request. After the group is created, it schedules an action to create
        # instances in the group using the specified instance template. This operation
        # is marked as DONE when the group is created even if the instances in the group
        # have not yet been created. You must separately verify the status of the
        # individual instances with the listmanagedinstances method.
        # A managed instance group can have up to 1000 VM instances per group. Please
        # contact Cloud Support if you need an increase in this limit.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where you want to create the managed instance group.
        # @param [Google::Apis::ComputeBeta::InstanceGroupManager] instance_group_manager_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_instance_group_manager(project, zone, instance_group_manager_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceGroupManager::Representation
          command.request_object = instance_group_manager_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of managed instance groups that are contained within the
        # specified project and zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InstanceGroupManagerList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InstanceGroupManagerList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_group_managers(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/instanceGroupManagers', options)
          command.response_representation = Google::Apis::ComputeBeta::InstanceGroupManagerList::Representation
          command.response_class = Google::Apis::ComputeBeta::InstanceGroupManagerList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all of the instances in the managed instance group. Each instance in the
        # list has a currentAction, which indicates the action that the managed instance
        # group is performing on the instance. For example, if the group is still
        # creating an instance, the currentAction is CREATING. If a previous action
        # failed, the list displays the errors for that failed action.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [String] filter
        # @param [Fixnum] max_results
        # @param [String] order_by
        # @param [String] page_token
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InstanceGroupManagersListManagedInstancesResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InstanceGroupManagersListManagedInstancesResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_group_manager_managed_instances(project, zone, instance_group_manager, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/listManagedInstances', options)
          command.response_representation = Google::Apis::ComputeBeta::InstanceGroupManagersListManagedInstancesResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::InstanceGroupManagersListManagedInstancesResponse
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['order_by'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a managed instance group using the information that you specify in the
        # request. This operation is marked as DONE when the group is patched even if
        # the instances in the group are still in the process of being patched. You must
        # separately verify the status of the individual instances with the
        # listManagedInstances method. This method supports PATCH semantics and uses the
        # JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where you want to create the managed instance group.
        # @param [String] instance_group_manager
        #   The name of the instance group manager.
        # @param [Google::Apis::ComputeBeta::InstanceGroupManager] instance_group_manager_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_instance_group_manager(project, zone, instance_group_manager, instance_group_manager_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:patch, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceGroupManager::Representation
          command.request_object = instance_group_manager_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Schedules a group action to recreate the specified instances in the managed
        # instance group. The instances are deleted and recreated using the current
        # instance template for the managed instance group. This operation is marked as
        # DONE when the action is scheduled even if the instances have not yet been
        # recreated. You must separately verify the status of the recreating action with
        # the listmanagedinstances method.
        # If the group is part of a backend service that has enabled connection draining,
        # it can take up to 60 seconds after the connection draining duration has
        # elapsed before the VM instance is removed or deleted.
        # You can specify a maximum of 1000 instances with this method per request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeBeta::InstanceGroupManagersRecreateInstancesRequest] instance_group_managers_recreate_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def recreate_instance_group_manager_instances(project, zone, instance_group_manager, instance_group_managers_recreate_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/recreateInstances', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceGroupManagersRecreateInstancesRequest::Representation
          command.request_object = instance_group_managers_recreate_instances_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Resizes the managed instance group. If you increase the size, the group
        # creates new instances using the current instance template. If you decrease the
        # size, the group deletes instances. The resize operation is marked DONE when
        # the resize actions are scheduled even if the group has not yet added or
        # deleted any instances. You must separately verify the status of the creating
        # or deleting actions with the listmanagedinstances method.
        # If the group is part of a backend service that has enabled connection draining,
        # it can take up to 60 seconds after the connection draining duration has
        # elapsed before the VM instance is removed or deleted.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Fixnum] size
        #   The number of running instances that the managed instance group should
        #   maintain at any given time. The group automatically adds or removes instances
        #   to maintain the number of instances specified by this parameter.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def resize_instance_group_manager(project, zone, instance_group_manager, size, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/resize', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['size'] = size unless size.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Resizes the managed instance group with advanced configuration options like
        # disabling creation retries. This is an extended version of the resize method.
        # If you increase the size of the instance group, the group creates new
        # instances using the current instance template. If you decrease the size, the
        # group deletes instances. The resize operation is marked DONE when the resize
        # actions are scheduled even if the group has not yet added or deleted any
        # instances. You must separately verify the status of the creating,
        # creatingWithoutRetries, or deleting actions with the get or
        # listmanagedinstances method.
        # If the group is part of a backend service that has enabled connection draining,
        # it can take up to 60 seconds after the connection draining duration has
        # elapsed before the VM instance is removed or deleted.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeBeta::InstanceGroupManagersResizeAdvancedRequest] instance_group_managers_resize_advanced_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def resize_instance_group_manager_advanced(project, zone, instance_group_manager, instance_group_managers_resize_advanced_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/resizeAdvanced', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceGroupManagersResizeAdvancedRequest::Representation
          command.request_object = instance_group_managers_resize_advanced_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Modifies the autohealing policies.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the instance group manager.
        # @param [Google::Apis::ComputeBeta::InstanceGroupManagersSetAutoHealingRequest] instance_group_managers_set_auto_healing_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_group_manager_auto_healing_policies(project, zone, instance_group_manager, instance_group_managers_set_auto_healing_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/setAutoHealingPolicies', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceGroupManagersSetAutoHealingRequest::Representation
          command.request_object = instance_group_managers_set_auto_healing_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Specifies the instance template to use when creating new instances in this
        # group. The templates for existing instances in the group do not change unless
        # you recreate them.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeBeta::InstanceGroupManagersSetInstanceTemplateRequest] instance_group_managers_set_instance_template_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_group_manager_instance_template(project, zone, instance_group_manager, instance_group_managers_set_instance_template_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/setInstanceTemplate', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceGroupManagersSetInstanceTemplateRequest::Representation
          command.request_object = instance_group_managers_set_instance_template_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Modifies the target pools to which all instances in this managed instance
        # group are assigned. The target pools automatically apply to all of the
        # instances in the managed instance group. This operation is marked DONE when
        # you make the request even if the instances have not yet been added to their
        # target pools. The change might take some time to apply to all of the instances
        # in the group depending on the size of the group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeBeta::InstanceGroupManagersSetTargetPoolsRequest] instance_group_managers_set_target_pools_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_group_manager_target_pools(project, zone, instance_group_manager, instance_group_managers_set_target_pools_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/setTargetPools', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceGroupManagersSetTargetPoolsRequest::Representation
          command.request_object = instance_group_managers_set_target_pools_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_instance_group_manager_iam_permissions(project, zone, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a managed instance group using the information that you specify in the
        # request. This operation is marked as DONE when the group is updated even if
        # the instances in the group have not yet been updated. You must separately
        # verify the status of the individual instances with the listManagedInstances
        # method.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where you want to create the managed instance group.
        # @param [String] instance_group_manager
        #   The name of the instance group manager.
        # @param [Google::Apis::ComputeBeta::InstanceGroupManager] instance_group_manager_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_instance_group_manager(project, zone, instance_group_manager, instance_group_manager_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:put, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceGroupManager::Representation
          command.request_object = instance_group_manager_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds a list of instances to the specified instance group. All of the instances
        # in the instance group must be in the same network/subnetwork. Read  Adding
        # instances for more information.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group where you are adding instances.
        # @param [Google::Apis::ComputeBeta::InstanceGroupsAddInstancesRequest] instance_groups_add_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_instance_group_instances(project, zone, instance_group, instance_groups_add_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instanceGroups/{instanceGroup}/addInstances', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceGroupsAddInstancesRequest::Representation
          command.request_object = instance_groups_add_instances_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of instance groups and sorts them by zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InstanceGroupAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InstanceGroupAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_instance_groups(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/instanceGroups', options)
          command.response_representation = Google::Apis::ComputeBeta::InstanceGroupAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::InstanceGroupAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified instance group. The instances in the group are not
        # deleted. Note that instance group must not belong to a backend service. Read
        # Deleting an instance group for more information.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance_group(project, zone, instance_group, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/zones/{zone}/instanceGroups/{instanceGroup}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified instance group. Gets a list of available instance groups
        # by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InstanceGroup] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InstanceGroup]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_group(project, zone, instance_group, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/instanceGroups/{instanceGroup}', options)
          command.response_representation = Google::Apis::ComputeBeta::InstanceGroup::Representation
          command.response_class = Google::Apis::ComputeBeta::InstanceGroup
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an instance group in the specified project using the parameters that
        # are included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where you want to create the instance group.
        # @param [Google::Apis::ComputeBeta::InstanceGroup] instance_group_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_instance_group(project, zone, instance_group_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instanceGroups', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceGroup::Representation
          command.request_object = instance_group_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of instance groups that are located in the specified
        # project and zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InstanceGroupList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InstanceGroupList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_groups(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/instanceGroups', options)
          command.response_representation = Google::Apis::ComputeBeta::InstanceGroupList::Representation
          command.response_class = Google::Apis::ComputeBeta::InstanceGroupList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists the instances in the specified instance group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group from which you want to generate a list of
        #   included instances.
        # @param [Google::Apis::ComputeBeta::InstanceGroupsListInstancesRequest] instance_groups_list_instances_request_object
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InstanceGroupsListInstances] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InstanceGroupsListInstances]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_group_instances(project, zone, instance_group, instance_groups_list_instances_request_object = nil, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instanceGroups/{instanceGroup}/listInstances', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceGroupsListInstancesRequest::Representation
          command.request_object = instance_groups_list_instances_request_object
          command.response_representation = Google::Apis::ComputeBeta::InstanceGroupsListInstances::Representation
          command.response_class = Google::Apis::ComputeBeta::InstanceGroupsListInstances
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Removes one or more instances from the specified instance group, but does not
        # delete those instances.
        # If the group is part of a backend service that has enabled connection draining,
        # it can take up to 60 seconds after the connection draining duration before
        # the VM instance is removed or deleted.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group where the specified instances will be removed.
        # @param [Google::Apis::ComputeBeta::InstanceGroupsRemoveInstancesRequest] instance_groups_remove_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def remove_instance_group_instances(project, zone, instance_group, instance_groups_remove_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instanceGroups/{instanceGroup}/removeInstances', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceGroupsRemoveInstancesRequest::Representation
          command.request_object = instance_groups_remove_instances_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the named ports for the specified instance group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group where the named ports are updated.
        # @param [Google::Apis::ComputeBeta::InstanceGroupsSetNamedPortsRequest] instance_groups_set_named_ports_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_group_named_ports(project, zone, instance_group, instance_groups_set_named_ports_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instanceGroups/{instanceGroup}/setNamedPorts', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceGroupsSetNamedPortsRequest::Representation
          command.request_object = instance_groups_set_named_ports_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_instance_group_iam_permissions(project, zone, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instanceGroups/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified instance template. Deleting an instance template is
        # permanent and cannot be undone. It's not possible to delete templates which
        # are in use by an instance group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] instance_template
        #   The name of the instance template to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance_template(project, instance_template, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/global/instanceTemplates/{instanceTemplate}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['instanceTemplate'] = instance_template unless instance_template.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified instance template. Gets a list of available instance
        # templates by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] instance_template
        #   The name of the instance template.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InstanceTemplate] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InstanceTemplate]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_template(project, instance_template, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/instanceTemplates/{instanceTemplate}', options)
          command.response_representation = Google::Apis::ComputeBeta::InstanceTemplate::Representation
          command.response_class = Google::Apis::ComputeBeta::InstanceTemplate
          command.params['project'] = project unless project.nil?
          command.params['instanceTemplate'] = instance_template unless instance_template.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an instance template in the specified project using the data that is
        # included in the request. If you are creating a new template to update an
        # existing instance group, your new instance template must use the same network
        # or, if applicable, the same subnetwork as the original template.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::InstanceTemplate] instance_template_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_instance_template(project, instance_template_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/instanceTemplates', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceTemplate::Representation
          command.request_object = instance_template_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of instance templates that are contained within the specified
        # project and zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InstanceTemplateList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InstanceTemplateList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_templates(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/instanceTemplates', options)
          command.response_representation = Google::Apis::ComputeBeta::InstanceTemplateList::Representation
          command.response_class = Google::Apis::ComputeBeta::InstanceTemplateList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_instance_template_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/instanceTemplates/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds an access config to an instance's network interface.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [String] network_interface
        #   The name of the network interface to add to this instance.
        # @param [Google::Apis::ComputeBeta::AccessConfig] access_config_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_instance_access_config(project, zone, instance, network_interface, access_config_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/addAccessConfig', options)
          command.request_representation = Google::Apis::ComputeBeta::AccessConfig::Representation
          command.request_object = access_config_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['networkInterface'] = network_interface unless network_interface.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves aggregated list of instances.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InstanceAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InstanceAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_instances(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/instances', options)
          command.response_representation = Google::Apis::ComputeBeta::InstanceAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::InstanceAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Attaches an existing Disk resource to an instance. You must first create the
        # disk before you can attach it. It is not possible to create and attach a disk
        # at the same time. For more information, read Adding a persistent disk to your
        # instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [Google::Apis::ComputeBeta::AttachedDisk] attached_disk_object
        # @param [Boolean] force_attach
        #   Whether to force attach the disk even if it's currently attached to another
        #   instance. This is only available for regional disks.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def attach_disk(project, zone, instance, attached_disk_object = nil, force_attach: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/attachDisk', options)
          command.request_representation = Google::Apis::ComputeBeta::AttachedDisk::Representation
          command.request_object = attached_disk_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['forceAttach'] = force_attach unless force_attach.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified Instance resource. For more information, see Stopping or
        # Deleting an Instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance(project, zone, instance, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/zones/{zone}/instances/{instance}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes an access config from an instance's network interface.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [String] access_config
        #   The name of the access config to delete.
        # @param [String] network_interface
        #   The name of the network interface.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance_access_config(project, zone, instance, access_config, network_interface, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/deleteAccessConfig', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['accessConfig'] = access_config unless access_config.nil?
          command.query['networkInterface'] = network_interface unless network_interface.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Detaches a disk from an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Instance name.
        # @param [String] device_name
        #   Disk device name to detach.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def detach_disk(project, zone, instance, device_name, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/detachDisk', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['deviceName'] = device_name unless device_name.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified Instance resource. Gets a list of available instances by
        # making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Instance] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Instance]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance(project, zone, instance, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/instances/{instance}', options)
          command.response_representation = Google::Apis::ComputeBeta::Instance::Representation
          command.response_class = Google::Apis::ComputeBeta::Instance
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified instance's serial port output.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Fixnum] port
        #   Specifies which COM or serial port to retrieve data from.
        # @param [Fixnum] start
        #   Returns output starting from a specific byte position. Use this to page
        #   through output when the output is too large to return in a single request. For
        #   the initial request, leave this field unspecified. For subsequent calls, this
        #   field should be set to the next value returned in the previous call.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::SerialPortOutput] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::SerialPortOutput]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_serial_port_output(project, zone, instance, port: nil, start: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/instances/{instance}/serialPort', options)
          command.response_representation = Google::Apis::ComputeBeta::SerialPortOutput::Representation
          command.response_class = Google::Apis::ComputeBeta::SerialPortOutput
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['port'] = port unless port.nil?
          command.query['start'] = start unless start.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an instance resource in the specified project using the data included
        # in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [Google::Apis::ComputeBeta::Instance] instance_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] source_instance_template
        #   Specifies instance template to create the instance.
        #   This field is optional. It can be a full or partial URL. For example, the
        #   following are all valid URLs to an instance template:
        #   - https://www.googleapis.com/compute/v1/projects/project/global/global/
        #   instanceTemplates/instanceTemplate
        #   - projects/project/global/global/instanceTemplates/instanceTemplate
        #   - global/instancesTemplates/instanceTemplate
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_instance(project, zone, instance_object = nil, request_id: nil, source_instance_template: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances', options)
          command.request_representation = Google::Apis::ComputeBeta::Instance::Representation
          command.request_object = instance_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['sourceInstanceTemplate'] = source_instance_template unless source_instance_template.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of instances contained within the specified zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InstanceList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InstanceList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instances(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/instances', options)
          command.response_representation = Google::Apis::ComputeBeta::InstanceList::Representation
          command.response_class = Google::Apis::ComputeBeta::InstanceList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of referrers to instances contained within the specified
        # zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the target instance scoping this request, or '-' if the request should
        #   span over all instances in the container.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InstanceListReferrers] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InstanceListReferrers]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_referrers(project, zone, instance, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/instances/{instance}/referrers', options)
          command.response_representation = Google::Apis::ComputeBeta::InstanceListReferrers::Representation
          command.response_class = Google::Apis::ComputeBeta::InstanceListReferrers
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Performs a reset on the instance. For more information, see Resetting an
        # instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def reset_instance(project, zone, instance, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/reset', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets deletion protection on the instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Boolean] deletion_protection
        #   Whether the resource should be protected against deletion.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_deletion_protection(project, zone, resource, deletion_protection: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{resource}/setDeletionProtection', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['deletionProtection'] = deletion_protection unless deletion_protection.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the auto-delete flag for a disk attached to an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name.
        # @param [Boolean] auto_delete
        #   Whether to auto-delete the disk when the instance is deleted.
        # @param [String] device_name
        #   The device name of the disk to modify.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_disk_auto_delete(project, zone, instance, auto_delete, device_name, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setDiskAutoDelete', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['autoDelete'] = auto_delete unless auto_delete.nil?
          command.query['deviceName'] = device_name unless device_name.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets labels on an instance. To learn more about labels, read the Labeling
        # Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeBeta::InstancesSetLabelsRequest] instances_set_labels_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_labels(project, zone, instance, instances_set_labels_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setLabels', options)
          command.request_representation = Google::Apis::ComputeBeta::InstancesSetLabelsRequest::Representation
          command.request_object = instances_set_labels_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Changes the number and/or type of accelerator for a stopped instance to the
        # values specified in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeBeta::InstancesSetMachineResourcesRequest] instances_set_machine_resources_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_machine_resources(project, zone, instance, instances_set_machine_resources_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setMachineResources', options)
          command.request_representation = Google::Apis::ComputeBeta::InstancesSetMachineResourcesRequest::Representation
          command.request_object = instances_set_machine_resources_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Changes the machine type for a stopped instance to the machine type specified
        # in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeBeta::InstancesSetMachineTypeRequest] instances_set_machine_type_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_machine_type(project, zone, instance, instances_set_machine_type_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setMachineType', options)
          command.request_representation = Google::Apis::ComputeBeta::InstancesSetMachineTypeRequest::Representation
          command.request_object = instances_set_machine_type_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets metadata for the specified instance to the data included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeBeta::Metadata] metadata_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_metadata(project, zone, instance, metadata_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setMetadata', options)
          command.request_representation = Google::Apis::ComputeBeta::Metadata::Representation
          command.request_object = metadata_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Changes the minimum CPU platform that this instance should use. This method
        # can only be called on a stopped instance. For more information, read
        # Specifying a Minimum CPU Platform.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeBeta::InstancesSetMinCpuPlatformRequest] instances_set_min_cpu_platform_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_min_cpu_platform(project, zone, instance, instances_set_min_cpu_platform_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setMinCpuPlatform', options)
          command.request_representation = Google::Apis::ComputeBeta::InstancesSetMinCpuPlatformRequest::Representation
          command.request_object = instances_set_min_cpu_platform_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets an instance's scheduling options.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Instance name.
        # @param [Google::Apis::ComputeBeta::Scheduling] scheduling_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_scheduling(project, zone, instance, scheduling_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setScheduling', options)
          command.request_representation = Google::Apis::ComputeBeta::Scheduling::Representation
          command.request_object = scheduling_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the service account on the instance. For more information, read Changing
        # the service account and access scopes for an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance resource to start.
        # @param [Google::Apis::ComputeBeta::InstancesSetServiceAccountRequest] instances_set_service_account_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_service_account(project, zone, instance, instances_set_service_account_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setServiceAccount', options)
          command.request_representation = Google::Apis::ComputeBeta::InstancesSetServiceAccountRequest::Representation
          command.request_object = instances_set_service_account_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the Shielded VM integrity policy for an instance. This method supports
        # PATCH semantics and uses the JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeBeta::ShieldedVmIntegrityPolicy] shielded_vm_integrity_policy_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_shielded_vm_integrity_policy(project, zone, instance, shielded_vm_integrity_policy_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:patch, '{project}/zones/{zone}/instances/{instance}/setShieldedVmIntegrityPolicy', options)
          command.request_representation = Google::Apis::ComputeBeta::ShieldedVmIntegrityPolicy::Representation
          command.request_object = shielded_vm_integrity_policy_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets tags for the specified instance to the data included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeBeta::Tags] tags_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_tags(project, zone, instance, tags_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setTags', options)
          command.request_representation = Google::Apis::ComputeBeta::Tags::Representation
          command.request_object = tags_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Simulates a maintenance event on the instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def simulate_instance_maintenance_event(project, zone, instance, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/simulateMaintenanceEvent', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Starts an instance that was stopped using the instances().stop method. For
        # more information, see Restart an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance resource to start.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def start_instance(project, zone, instance, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/start', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Starts an instance that was stopped using the instances().stop method. For
        # more information, see Restart an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance resource to start.
        # @param [Google::Apis::ComputeBeta::InstancesStartWithEncryptionKeyRequest] instances_start_with_encryption_key_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def start_instance_with_encryption_key(project, zone, instance, instances_start_with_encryption_key_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/startWithEncryptionKey', options)
          command.request_representation = Google::Apis::ComputeBeta::InstancesStartWithEncryptionKeyRequest::Representation
          command.request_object = instances_start_with_encryption_key_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Stops a running instance, shutting it down cleanly, and allows you to restart
        # the instance at a later time. Stopped instances do not incur VM usage charges
        # while they are stopped. However, resources that the VM is using, such as
        # persistent disks and static IP addresses, will continue to be charged until
        # they are deleted. For more information, see Stopping an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance resource to stop.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def stop_instance(project, zone, instance, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/stop', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_instance_iam_permissions(project, zone, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified access config from an instance's network interface with
        # the data included in the request. This method supports PATCH semantics and
        # uses the JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [String] network_interface
        #   The name of the network interface where the access config is attached.
        # @param [Google::Apis::ComputeBeta::AccessConfig] access_config_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_instance_access_config(project, zone, instance, network_interface, access_config_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/updateAccessConfig', options)
          command.request_representation = Google::Apis::ComputeBeta::AccessConfig::Representation
          command.request_object = access_config_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['networkInterface'] = network_interface unless network_interface.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an instance's network interface. This method follows PATCH semantics.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [String] network_interface
        #   The name of the network interface to update.
        # @param [Google::Apis::ComputeBeta::NetworkInterface] network_interface_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_instance_network_interface(project, zone, instance, network_interface, network_interface_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:patch, '{project}/zones/{zone}/instances/{instance}/updateNetworkInterface', options)
          command.request_representation = Google::Apis::ComputeBeta::NetworkInterface::Representation
          command.request_object = network_interface_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['networkInterface'] = network_interface unless network_interface.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the Shielded VM config for an instance. This method supports PATCH
        # semantics and uses the JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeBeta::ShieldedVmConfig] shielded_vm_config_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_instance_shielded_vm_config(project, zone, instance, shielded_vm_config_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:patch, '{project}/zones/{zone}/instances/{instance}/updateShieldedVmConfig', options)
          command.request_representation = Google::Apis::ComputeBeta::ShieldedVmConfig::Representation
          command.request_object = shielded_vm_config_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of interconnect attachments.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InterconnectAttachmentAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InterconnectAttachmentAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_interconnect_attachment_list(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/interconnectAttachments', options)
          command.response_representation = Google::Apis::ComputeBeta::InterconnectAttachmentAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::InterconnectAttachmentAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified interconnect attachment.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] interconnect_attachment
        #   Name of the interconnect attachment to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_interconnect_attachment(project, region, interconnect_attachment, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/regions/{region}/interconnectAttachments/{interconnectAttachment}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['interconnectAttachment'] = interconnect_attachment unless interconnect_attachment.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified interconnect attachment.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] interconnect_attachment
        #   Name of the interconnect attachment to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InterconnectAttachment] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InterconnectAttachment]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_interconnect_attachment(project, region, interconnect_attachment, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/interconnectAttachments/{interconnectAttachment}', options)
          command.response_representation = Google::Apis::ComputeBeta::InterconnectAttachment::Representation
          command.response_class = Google::Apis::ComputeBeta::InterconnectAttachment
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['interconnectAttachment'] = interconnect_attachment unless interconnect_attachment.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an InterconnectAttachment in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [Google::Apis::ComputeBeta::InterconnectAttachment] interconnect_attachment_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_interconnect_attachment(project, region, interconnect_attachment_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/interconnectAttachments', options)
          command.request_representation = Google::Apis::ComputeBeta::InterconnectAttachment::Representation
          command.request_object = interconnect_attachment_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of interconnect attachments contained within the specified
        # region.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InterconnectAttachmentList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InterconnectAttachmentList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_interconnect_attachments(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/interconnectAttachments', options)
          command.response_representation = Google::Apis::ComputeBeta::InterconnectAttachmentList::Representation
          command.response_class = Google::Apis::ComputeBeta::InterconnectAttachmentList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified interconnect attachment with the data included in the
        # request. This method supports PATCH semantics and uses the JSON merge patch
        # format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] interconnect_attachment
        #   Name of the interconnect attachment to patch.
        # @param [Google::Apis::ComputeBeta::InterconnectAttachment] interconnect_attachment_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_interconnect_attachment(project, region, interconnect_attachment, interconnect_attachment_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:patch, '{project}/regions/{region}/interconnectAttachments/{interconnectAttachment}', options)
          command.request_representation = Google::Apis::ComputeBeta::InterconnectAttachment::Representation
          command.request_object = interconnect_attachment_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['interconnectAttachment'] = interconnect_attachment unless interconnect_attachment.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on an InterconnectAttachment. To learn more about labels, read
        # the Labeling Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The region for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::RegionSetLabelsRequest] region_set_labels_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_interconnect_attachment_labels(project, region, resource, region_set_labels_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/interconnectAttachments/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeBeta::RegionSetLabelsRequest::Representation
          command.request_object = region_set_labels_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_interconnect_attachment_iam_permissions(project, region, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/interconnectAttachments/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the details for the specified interconnect location. Gets a list of
        # available interconnect locations by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] interconnect_location
        #   Name of the interconnect location to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InterconnectLocation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InterconnectLocation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_interconnect_location(project, interconnect_location, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/interconnectLocations/{interconnectLocation}', options)
          command.response_representation = Google::Apis::ComputeBeta::InterconnectLocation::Representation
          command.response_class = Google::Apis::ComputeBeta::InterconnectLocation
          command.params['project'] = project unless project.nil?
          command.params['interconnectLocation'] = interconnect_location unless interconnect_location.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of interconnect locations available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InterconnectLocationList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InterconnectLocationList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_interconnect_locations(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/interconnectLocations', options)
          command.response_representation = Google::Apis::ComputeBeta::InterconnectLocationList::Representation
          command.response_class = Google::Apis::ComputeBeta::InterconnectLocationList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified interconnect.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] interconnect
        #   Name of the interconnect to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_interconnect(project, interconnect, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/global/interconnects/{interconnect}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['interconnect'] = interconnect unless interconnect.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified interconnect. Get a list of available interconnects by
        # making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] interconnect
        #   Name of the interconnect to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Interconnect] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Interconnect]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_interconnect(project, interconnect, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/interconnects/{interconnect}', options)
          command.response_representation = Google::Apis::ComputeBeta::Interconnect::Representation
          command.response_class = Google::Apis::ComputeBeta::Interconnect
          command.params['project'] = project unless project.nil?
          command.params['interconnect'] = interconnect unless interconnect.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a Interconnect in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::Interconnect] interconnect_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_interconnect(project, interconnect_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/interconnects', options)
          command.request_representation = Google::Apis::ComputeBeta::Interconnect::Representation
          command.request_object = interconnect_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of interconnect available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InterconnectList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InterconnectList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_interconnects(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/interconnects', options)
          command.response_representation = Google::Apis::ComputeBeta::InterconnectList::Representation
          command.response_class = Google::Apis::ComputeBeta::InterconnectList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified interconnect with the data included in the request. This
        # method supports PATCH semantics and uses the JSON merge patch format and
        # processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] interconnect
        #   Name of the interconnect to update.
        # @param [Google::Apis::ComputeBeta::Interconnect] interconnect_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_interconnect(project, interconnect, interconnect_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:patch, '{project}/global/interconnects/{interconnect}', options)
          command.request_representation = Google::Apis::ComputeBeta::Interconnect::Representation
          command.request_object = interconnect_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['interconnect'] = interconnect unless interconnect.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on an Interconnect. To learn more about labels, read the
        # Labeling Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::GlobalSetLabelsRequest] global_set_labels_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_interconnect_labels(project, resource, global_set_labels_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/interconnects/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeBeta::GlobalSetLabelsRequest::Representation
          command.request_object = global_set_labels_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_interconnect_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/interconnects/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Return a specified license code. License codes are mirrored across all
        # projects that have permissions to read the License Code.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] license_code
        #   Number corresponding to the License code resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::LicenseCode] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::LicenseCode]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_license_code(project, license_code, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/licenseCodes/{licenseCode}', options)
          command.response_representation = Google::Apis::ComputeBeta::LicenseCode::Representation
          command.response_class = Google::Apis::ComputeBeta::LicenseCode
          command.params['project'] = project unless project.nil?
          command.params['licenseCode'] = license_code unless license_code.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified license.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] license
        #   Name of the license resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_license(project, license, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/global/licenses/{license}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['license'] = license unless license.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified License resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] license
        #   Name of the License resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::License] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::License]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_license(project, license, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/licenses/{license}', options)
          command.response_representation = Google::Apis::ComputeBeta::License::Representation
          command.response_class = Google::Apis::ComputeBeta::License
          command.params['project'] = project unless project.nil?
          command.params['license'] = license unless license.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Create a License resource in the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::License] license_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_license(project, license_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/licenses', options)
          command.request_representation = Google::Apis::ComputeBeta::License::Representation
          command.request_object = license_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of licenses available in the specified project. This method
        # does not get any licenses that belong to other projects, including licenses
        # attached to publicly-available images, like Debian 9. If you want to get a
        # list of publicly-available licenses, use this method to make a request to the
        # respective image project, such as debian-cloud or windows-cloud.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::LicensesListResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::LicensesListResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_licenses(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/licenses', options)
          command.response_representation = Google::Apis::ComputeBeta::LicensesListResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::LicensesListResponse
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of machine types.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::MachineTypeAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::MachineTypeAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_machine_types(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/machineTypes', options)
          command.response_representation = Google::Apis::ComputeBeta::MachineTypeAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::MachineTypeAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified machine type. Gets a list of available machine types by
        # making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] machine_type
        #   Name of the machine type to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::MachineType] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::MachineType]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_machine_type(project, zone, machine_type, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/machineTypes/{machineType}', options)
          command.response_representation = Google::Apis::ComputeBeta::MachineType::Representation
          command.response_class = Google::Apis::ComputeBeta::MachineType
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['machineType'] = machine_type unless machine_type.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of machine types available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::MachineTypeList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::MachineTypeList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_machine_types(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/machineTypes', options)
          command.response_representation = Google::Apis::ComputeBeta::MachineTypeList::Representation
          command.response_class = Google::Apis::ComputeBeta::MachineTypeList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of network endpoint groups and sorts them by zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::NetworkEndpointGroupAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::NetworkEndpointGroupAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_network_endpoint_group_list(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/networkEndpointGroups', options)
          command.response_representation = Google::Apis::ComputeBeta::NetworkEndpointGroupAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::NetworkEndpointGroupAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Attach a list of network endpoints to the specified network endpoint group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the network endpoint group is located. It should
        #   comply with RFC1035.
        # @param [String] network_endpoint_group
        #   The name of the network endpoint group where you are attaching network
        #   endpoints to. It should comply with RFC1035.
        # @param [Google::Apis::ComputeBeta::NetworkEndpointGroupsAttachEndpointsRequest] network_endpoint_groups_attach_endpoints_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def attach_network_endpoint_group_network_endpoints(project, zone, network_endpoint_group, network_endpoint_groups_attach_endpoints_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/networkEndpointGroups/{networkEndpointGroup}/attachNetworkEndpoints', options)
          command.request_representation = Google::Apis::ComputeBeta::NetworkEndpointGroupsAttachEndpointsRequest::Representation
          command.request_object = network_endpoint_groups_attach_endpoints_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['networkEndpointGroup'] = network_endpoint_group unless network_endpoint_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified network endpoint group. The network endpoints in the NEG
        # and the VM instances they belong to are not terminated when the NEG is deleted.
        # Note that the NEG cannot be deleted if there are backend services referencing
        # it.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the network endpoint group is located. It should
        #   comply with RFC1035.
        # @param [String] network_endpoint_group
        #   The name of the network endpoint group to delete. It should comply with
        #   RFC1035.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_network_endpoint_group(project, zone, network_endpoint_group, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/zones/{zone}/networkEndpointGroups/{networkEndpointGroup}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['networkEndpointGroup'] = network_endpoint_group unless network_endpoint_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Detach a list of network endpoints from the specified network endpoint group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the network endpoint group is located. It should
        #   comply with RFC1035.
        # @param [String] network_endpoint_group
        #   The name of the network endpoint group where you are removing network
        #   endpoints. It should comply with RFC1035.
        # @param [Google::Apis::ComputeBeta::NetworkEndpointGroupsDetachEndpointsRequest] network_endpoint_groups_detach_endpoints_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def detach_network_endpoint_group_network_endpoints(project, zone, network_endpoint_group, network_endpoint_groups_detach_endpoints_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/networkEndpointGroups/{networkEndpointGroup}/detachNetworkEndpoints', options)
          command.request_representation = Google::Apis::ComputeBeta::NetworkEndpointGroupsDetachEndpointsRequest::Representation
          command.request_object = network_endpoint_groups_detach_endpoints_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['networkEndpointGroup'] = network_endpoint_group unless network_endpoint_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified network endpoint group. Gets a list of available network
        # endpoint groups by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the network endpoint group is located. It should
        #   comply with RFC1035.
        # @param [String] network_endpoint_group
        #   The name of the network endpoint group. It should comply with RFC1035.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::NetworkEndpointGroup] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::NetworkEndpointGroup]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_network_endpoint_group(project, zone, network_endpoint_group, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/networkEndpointGroups/{networkEndpointGroup}', options)
          command.response_representation = Google::Apis::ComputeBeta::NetworkEndpointGroup::Representation
          command.response_class = Google::Apis::ComputeBeta::NetworkEndpointGroup
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['networkEndpointGroup'] = network_endpoint_group unless network_endpoint_group.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a network endpoint group in the specified project using the parameters
        # that are included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where you want to create the network endpoint group. It
        #   should comply with RFC1035.
        # @param [Google::Apis::ComputeBeta::NetworkEndpointGroup] network_endpoint_group_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_network_endpoint_group(project, zone, network_endpoint_group_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/networkEndpointGroups', options)
          command.request_representation = Google::Apis::ComputeBeta::NetworkEndpointGroup::Representation
          command.request_object = network_endpoint_group_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of network endpoint groups that are located in the
        # specified project and zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the network endpoint group is located. It should
        #   comply with RFC1035.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::NetworkEndpointGroupList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::NetworkEndpointGroupList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_network_endpoint_groups(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/networkEndpointGroups', options)
          command.response_representation = Google::Apis::ComputeBeta::NetworkEndpointGroupList::Representation
          command.response_class = Google::Apis::ComputeBeta::NetworkEndpointGroupList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists the network endpoints in the specified network endpoint group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the network endpoint group is located. It should
        #   comply with RFC1035.
        # @param [String] network_endpoint_group
        #   The name of the network endpoint group from which you want to generate a list
        #   of included network endpoints. It should comply with RFC1035.
        # @param [Google::Apis::ComputeBeta::NetworkEndpointGroupsListEndpointsRequest] network_endpoint_groups_list_endpoints_request_object
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::NetworkEndpointGroupsListNetworkEndpoints] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::NetworkEndpointGroupsListNetworkEndpoints]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_network_endpoint_group_network_endpoints(project, zone, network_endpoint_group, network_endpoint_groups_list_endpoints_request_object = nil, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/networkEndpointGroups/{networkEndpointGroup}/listNetworkEndpoints', options)
          command.request_representation = Google::Apis::ComputeBeta::NetworkEndpointGroupsListEndpointsRequest::Representation
          command.request_object = network_endpoint_groups_list_endpoints_request_object
          command.response_representation = Google::Apis::ComputeBeta::NetworkEndpointGroupsListNetworkEndpoints::Representation
          command.response_class = Google::Apis::ComputeBeta::NetworkEndpointGroupsListNetworkEndpoints
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['networkEndpointGroup'] = network_endpoint_group unless network_endpoint_group.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_network_endpoint_group_iam_permissions(project, zone, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/networkEndpointGroups/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds a peering to the specified network.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network
        #   Name of the network resource to add peering to.
        # @param [Google::Apis::ComputeBeta::NetworksAddPeeringRequest] networks_add_peering_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_network_peering(project, network, networks_add_peering_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/networks/{network}/addPeering', options)
          command.request_representation = Google::Apis::ComputeBeta::NetworksAddPeeringRequest::Representation
          command.request_object = networks_add_peering_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['network'] = network unless network.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified network.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network
        #   Name of the network to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_network(project, network, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/global/networks/{network}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['network'] = network unless network.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified network. Gets a list of available networks by making a
        # list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network
        #   Name of the network to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Network] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Network]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_network(project, network, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/networks/{network}', options)
          command.response_representation = Google::Apis::ComputeBeta::Network::Representation
          command.response_class = Google::Apis::ComputeBeta::Network
          command.params['project'] = project unless project.nil?
          command.params['network'] = network unless network.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a network in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::Network] network_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_network(project, network_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/networks', options)
          command.request_representation = Google::Apis::ComputeBeta::Network::Representation
          command.request_object = network_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of networks available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::NetworkList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::NetworkList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_networks(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/global/networks', options)
          command.response_representation = Google::Apis::ComputeBeta::NetworkList::Representation
          command.response_class = Google::Apis::ComputeBeta::NetworkList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Patches the specified network with the data included in the request. Only the
        # following fields can be modified: routingConfig.routingMode.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network
        #   Name of the network to update.
        # @param [Google::Apis::ComputeBeta::Network] network_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_network(project, network, network_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:patch, '{project}/global/networks/{network}', options)
          command.request_representation = Google::Apis::ComputeBeta::Network::Representation
          command.request_object = network_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['network'] = network unless network.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Removes a peering from the specified network.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network
        #   Name of the network resource to remove peering from.
        # @param [Google::Apis::ComputeBeta::NetworksRemovePeeringRequest] networks_remove_peering_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def remove_network_peering(project, network, networks_remove_peering_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/networks/{network}/removePeering', options)
          command.request_representation = Google::Apis::ComputeBeta::NetworksRemovePeeringRequest::Representation
          command.request_object = networks_remove_peering_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['network'] = network unless network.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Switches the network mode from auto subnet mode to custom subnet mode.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network
        #   Name of the network to be updated.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def switch_network_to_custom_mode(project, network, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/networks/{network}/switchToCustomMode', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['network'] = network unless network.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_network_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/global/networks/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds specified number of nodes to the node group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] node_group
        #   Name of the NodeGroup resource to delete.
        # @param [Google::Apis::ComputeBeta::NodeGroupsAddNodesRequest] node_groups_add_nodes_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_node_group_nodes(project, zone, node_group, node_groups_add_nodes_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/nodeGroups/{nodeGroup}/addNodes', options)
          command.request_representation = Google::Apis::ComputeBeta::NodeGroupsAddNodesRequest::Representation
          command.request_object = node_groups_add_nodes_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['nodeGroup'] = node_group unless node_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of node groups.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::NodeGroupAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::NodeGroupAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_node_group_list(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/nodeGroups', options)
          command.response_representation = Google::Apis::ComputeBeta::NodeGroupAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::NodeGroupAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified NodeGroup resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] node_group
        #   Name of the NodeGroup resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_node_group(project, zone, node_group, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/zones/{zone}/nodeGroups/{nodeGroup}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['nodeGroup'] = node_group unless node_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes specified nodes from the node group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] node_group
        #   Name of the NodeGroup resource to delete.
        # @param [Google::Apis::ComputeBeta::NodeGroupsDeleteNodesRequest] node_groups_delete_nodes_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_node_group_nodes(project, zone, node_group, node_groups_delete_nodes_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/nodeGroups/{nodeGroup}/deleteNodes', options)
          command.request_representation = Google::Apis::ComputeBeta::NodeGroupsDeleteNodesRequest::Representation
          command.request_object = node_groups_delete_nodes_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['nodeGroup'] = node_group unless node_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified NodeGroup. Get a list of available NodeGroups by making
        # a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] node_group
        #   Name of the node group to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::NodeGroup] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::NodeGroup]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_node_group(project, zone, node_group, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/nodeGroups/{nodeGroup}', options)
          command.response_representation = Google::Apis::ComputeBeta::NodeGroup::Representation
          command.response_class = Google::Apis::ComputeBeta::NodeGroup
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['nodeGroup'] = node_group unless node_group.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a NodeGroup resource in the specified project using the data included
        # in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [Fixnum] initial_node_count
        #   Initial count of nodes in the node group.
        # @param [Google::Apis::ComputeBeta::NodeGroup] node_group_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_node_group(project, zone, initial_node_count, node_group_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/nodeGroups', options)
          command.request_representation = Google::Apis::ComputeBeta::NodeGroup::Representation
          command.request_object = node_group_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['initialNodeCount'] = initial_node_count unless initial_node_count.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of node groups available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::NodeGroupList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::NodeGroupList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_node_groups(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/nodeGroups', options)
          command.response_representation = Google::Apis::ComputeBeta::NodeGroupList::Representation
          command.response_class = Google::Apis::ComputeBeta::NodeGroupList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the node template of the node group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] node_group
        #   Name of the NodeGroup resource to delete.
        # @param [Google::Apis::ComputeBeta::NodeGroupsSetNodeTemplateRequest] node_groups_set_node_template_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_node_group_node_template(project, zone, node_group, node_groups_set_node_template_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/zones/{zone}/nodeGroups/{nodeGroup}/setNodeTemplate', options)
          command.request_representation = Google::Apis::ComputeBeta::NodeGroupsSetNodeTemplateRequest::Representation
          command.request_object = node_groups_set_node_template_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['nodeGroup'] = node_group unless node_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of node templates.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::NodeTemplateAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::NodeTemplateAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_node_template_list(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/nodeTemplates', options)
          command.response_representation = Google::Apis::ComputeBeta::NodeTemplateAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::NodeTemplateAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified NodeTemplate resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] node_template
        #   Name of the NodeTemplate resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_node_template(project, region, node_template, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/regions/{region}/nodeTemplates/{nodeTemplate}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['nodeTemplate'] = node_template unless node_template.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified node template. Gets a list of available node templates
        # by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] node_template
        #   Name of the node template to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::NodeTemplate] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::NodeTemplate]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_node_template(project, region, node_template, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/nodeTemplates/{nodeTemplate}', options)
          command.response_representation = Google::Apis::ComputeBeta::NodeTemplate::Representation
          command.response_class = Google::Apis::ComputeBeta::NodeTemplate
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['nodeTemplate'] = node_template unless node_template.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a NodeTemplate resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [Google::Apis::ComputeBeta::NodeTemplate] node_template_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_node_template(project, region, node_template_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/nodeTemplates', options)
          command.request_representation = Google::Apis::ComputeBeta::NodeTemplate::Representation
          command.request_object = node_template_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of node templates available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::NodeTemplateList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::NodeTemplateList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_node_templates(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/nodeTemplates', options)
          command.response_representation = Google::Apis::ComputeBeta::NodeTemplateList::Representation
          command.response_class = Google::Apis::ComputeBeta::NodeTemplateList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of node types.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::NodeTypeAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::NodeTypeAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_node_type_list(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/nodeTypes', options)
          command.response_representation = Google::Apis::ComputeBeta::NodeTypeAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::NodeTypeAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified node type. Gets a list of available node types by making
        # a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] node_type
        #   Name of the node type to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::NodeType] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::NodeType]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_node_type(project, zone, node_type, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/nodeTypes/{nodeType}', options)
          command.response_representation = Google::Apis::ComputeBeta::NodeType::Representation
          command.response_class = Google::Apis::ComputeBeta::NodeType
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['nodeType'] = node_type unless node_type.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of node types available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::NodeTypeList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::NodeTypeList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_node_types(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/zones/{zone}/nodeTypes', options)
          command.response_representation = Google::Apis::ComputeBeta::NodeTypeList::Representation
          command.response_class = Google::Apis::ComputeBeta::NodeTypeList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Disable this project as a shared VPC host project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def disable_project_xpn_host(project, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/disableXpnHost', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Disable a serivce resource (a.k.a service project) associated with this host
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::ProjectsDisableXpnResourceRequest] projects_disable_xpn_resource_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def disable_project_xpn_resource(project, projects_disable_xpn_resource_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/disableXpnResource', options)
          command.request_representation = Google::Apis::ComputeBeta::ProjectsDisableXpnResourceRequest::Representation
          command.request_object = projects_disable_xpn_resource_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Enable this project as a shared VPC host project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def enable_project_xpn_host(project, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/enableXpnHost', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Enable service resource (a.k.a service project) for a host project, so that
        # subnets in the host project can be used by instances in the service project.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::ProjectsEnableXpnResourceRequest] projects_enable_xpn_resource_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def enable_project_xpn_resource(project, projects_enable_xpn_resource_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/enableXpnResource', options)
          command.request_representation = Google::Apis::ComputeBeta::ProjectsEnableXpnResourceRequest::Representation
          command.request_object = projects_enable_xpn_resource_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified Project resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Project] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Project]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project(project, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}', options)
          command.response_representation = Google::Apis::ComputeBeta::Project::Representation
          command.response_class = Google::Apis::ComputeBeta::Project
          command.params['project'] = project unless project.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the shared VPC host project that this project links to. May be empty if
        # no link exists.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Project] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Project]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_xpn_host(project, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/getXpnHost', options)
          command.response_representation = Google::Apis::ComputeBeta::Project::Representation
          command.response_class = Google::Apis::ComputeBeta::Project
          command.params['project'] = project unless project.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets service resources (a.k.a service project) associated with this host
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        # @param [Fixnum] max_results
        # @param [String] order_by
        # @param [String] page_token
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::ProjectsGetXpnResources] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::ProjectsGetXpnResources]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_xpn_resources(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/getXpnResources', options)
          command.response_representation = Google::Apis::ComputeBeta::ProjectsGetXpnResources::Representation
          command.response_class = Google::Apis::ComputeBeta::ProjectsGetXpnResources
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['order_by'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all shared VPC host projects visible to the user in an organization.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::ProjectsListXpnHostsRequest] projects_list_xpn_hosts_request_object
        # @param [String] filter
        # @param [Fixnum] max_results
        # @param [String] order_by
        # @param [String] page_token
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::XpnHostList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::XpnHostList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_project_xpn_hosts(project, projects_list_xpn_hosts_request_object = nil, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/listXpnHosts', options)
          command.request_representation = Google::Apis::ComputeBeta::ProjectsListXpnHostsRequest::Representation
          command.request_object = projects_list_xpn_hosts_request_object
          command.response_representation = Google::Apis::ComputeBeta::XpnHostList::Representation
          command.response_class = Google::Apis::ComputeBeta::XpnHostList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['order_by'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Moves a persistent disk from one zone to another.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::DiskMoveRequest] disk_move_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def move_disk(project, disk_move_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/moveDisk', options)
          command.request_representation = Google::Apis::ComputeBeta::DiskMoveRequest::Representation
          command.request_object = disk_move_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Moves an instance and its attached persistent disks from one zone to another.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::MoveInstanceRequest] move_instance_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def move_instance(project, move_instance_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/moveInstance', options)
          command.request_representation = Google::Apis::ComputeBeta::MoveInstanceRequest::Representation
          command.request_object = move_instance_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets metadata common to all instances within the specified project using the
        # data included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::Metadata] metadata_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_common_instance_metadata(project, metadata_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/setCommonInstanceMetadata', options)
          command.request_representation = Google::Apis::ComputeBeta::Metadata::Representation
          command.request_object = metadata_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the default network tier of the project. The default network tier is used
        # when an address/forwardingRule/instance is created without specifying the
        # network tier field.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::ProjectsSetDefaultNetworkTierRequest] projects_set_default_network_tier_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_project_default_network_tier(project, projects_set_default_network_tier_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/setDefaultNetworkTier', options)
          command.request_representation = Google::Apis::ComputeBeta::ProjectsSetDefaultNetworkTierRequest::Representation
          command.request_object = projects_set_default_network_tier_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Enables the usage export feature and sets the usage export bucket where
        # reports are stored. If you provide an empty request body using this method,
        # the usage export feature will be disabled.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeBeta::UsageExportLocation] usage_export_location_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_usage_export_bucket(project, usage_export_location_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/setUsageExportBucket', options)
          command.request_representation = Google::Apis::ComputeBeta::UsageExportLocation::Representation
          command.request_object = usage_export_location_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified autoscaler.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] autoscaler
        #   Name of the autoscaler to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_region_autoscaler(project, region, autoscaler, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/regions/{region}/autoscalers/{autoscaler}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified autoscaler.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] autoscaler
        #   Name of the autoscaler to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Autoscaler] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Autoscaler]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_region_autoscaler(project, region, autoscaler, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/autoscalers/{autoscaler}', options)
          command.response_representation = Google::Apis::ComputeBeta::Autoscaler::Representation
          command.response_class = Google::Apis::ComputeBeta::Autoscaler
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an autoscaler in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [Google::Apis::ComputeBeta::Autoscaler] autoscaler_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_region_autoscaler(project, region, autoscaler_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/autoscalers', options)
          command.request_representation = Google::Apis::ComputeBeta::Autoscaler::Representation
          command.request_object = autoscaler_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of autoscalers contained within the specified region.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::RegionAutoscalerList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::RegionAutoscalerList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_region_autoscalers(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/autoscalers', options)
          command.response_representation = Google::Apis::ComputeBeta::RegionAutoscalerList::Representation
          command.response_class = Google::Apis::ComputeBeta::RegionAutoscalerList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an autoscaler in the specified project using the data included in the
        # request. This method supports PATCH semantics and uses the JSON merge patch
        # format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [Google::Apis::ComputeBeta::Autoscaler] autoscaler_object
        # @param [String] autoscaler
        #   Name of the autoscaler to patch.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_region_autoscaler(project, region, autoscaler_object = nil, autoscaler: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:patch, '{project}/regions/{region}/autoscalers', options)
          command.request_representation = Google::Apis::ComputeBeta::Autoscaler::Representation
          command.request_object = autoscaler_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_region_autoscaler_iam_permissions(project, region, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/autoscalers/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an autoscaler in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [Google::Apis::ComputeBeta::Autoscaler] autoscaler_object
        # @param [String] autoscaler
        #   Name of the autoscaler to update.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_region_autoscaler(project, region, autoscaler_object = nil, autoscaler: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:put, '{project}/regions/{region}/autoscalers', options)
          command.request_representation = Google::Apis::ComputeBeta::Autoscaler::Representation
          command.request_object = autoscaler_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified regional BackendService resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_region_backend_service(project, region, backend_service, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/regions/{region}/backendServices/{backendService}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified regional BackendService resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::BackendService] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::BackendService]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_region_backend_service(project, region, backend_service, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/backendServices/{backendService}', options)
          command.response_representation = Google::Apis::ComputeBeta::BackendService::Representation
          command.response_class = Google::Apis::ComputeBeta::BackendService
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the most recent health check results for this regional BackendService.
        # @param [String] project
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] backend_service
        #   Name of the BackendService resource for which to get health.
        # @param [Google::Apis::ComputeBeta::ResourceGroupReference] resource_group_reference_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::BackendServiceGroupHealth] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::BackendServiceGroupHealth]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_region_backend_service_health(project, region, backend_service, resource_group_reference_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/backendServices/{backendService}/getHealth', options)
          command.request_representation = Google::Apis::ComputeBeta::ResourceGroupReference::Representation
          command.request_object = resource_group_reference_object
          command.response_representation = Google::Apis::ComputeBeta::BackendServiceGroupHealth::Representation
          command.response_class = Google::Apis::ComputeBeta::BackendServiceGroupHealth
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a regional BackendService resource in the specified project using the
        # data included in the request. There are several restrictions and guidelines to
        # keep in mind when creating a regional backend service. Read  Restrictions and
        # Guidelines for more information.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [Google::Apis::ComputeBeta::BackendService] backend_service_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_region_backend_service(project, region, backend_service_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/backendServices', options)
          command.request_representation = Google::Apis::ComputeBeta::BackendService::Representation
          command.request_object = backend_service_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of regional BackendService resources available to the
        # specified project in the given region.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::BackendServiceList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::BackendServiceList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_region_backend_services(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/backendServices', options)
          command.response_representation = Google::Apis::ComputeBeta::BackendServiceList::Representation
          command.response_class = Google::Apis::ComputeBeta::BackendServiceList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified regional BackendService resource with the data included
        # in the request. There are several restrictions and guidelines to keep in mind
        # when updating a backend service. Read  Restrictions and Guidelines for more
        # information. This method supports PATCH semantics and uses the JSON merge
        # patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to patch.
        # @param [Google::Apis::ComputeBeta::BackendService] backend_service_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_region_backend_service(project, region, backend_service, backend_service_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:patch, '{project}/regions/{region}/backendServices/{backendService}', options)
          command.request_representation = Google::Apis::ComputeBeta::BackendService::Representation
          command.request_object = backend_service_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_region_backend_service_iam_permissions(project, region, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/backendServices/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified regional BackendService resource with the data included
        # in the request. There are several restrictions and guidelines to keep in mind
        # when updating a backend service. Read  Restrictions and Guidelines for more
        # information.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to update.
        # @param [Google::Apis::ComputeBeta::BackendService] backend_service_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_region_backend_service(project, region, backend_service, backend_service_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:put, '{project}/regions/{region}/backendServices/{backendService}', options)
          command.request_representation = Google::Apis::ComputeBeta::BackendService::Representation
          command.request_object = backend_service_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of commitments.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::CommitmentAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::CommitmentAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_region_commitment_list(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/aggregated/commitments', options)
          command.response_representation = Google::Apis::ComputeBeta::CommitmentAggregatedList::Representation
          command.response_class = Google::Apis::ComputeBeta::CommitmentAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified commitment resource. Gets a list of available
        # commitments by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] commitment
        #   Name of the commitment to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Commitment] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Commitment]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_region_commitment(project, region, commitment, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/commitments/{commitment}', options)
          command.response_representation = Google::Apis::ComputeBeta::Commitment::Representation
          command.response_class = Google::Apis::ComputeBeta::Commitment
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['commitment'] = commitment unless commitment.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a commitment in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [Google::Apis::ComputeBeta::Commitment] commitment_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_region_commitment(project, region, commitment_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/commitments', options)
          command.request_representation = Google::Apis::ComputeBeta::Commitment::Representation
          command.request_object = commitment_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of commitments contained within the specified region.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::CommitmentList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::CommitmentList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_region_commitments(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/commitments', options)
          command.response_representation = Google::Apis::ComputeBeta::CommitmentList::Representation
          command.response_class = Google::Apis::ComputeBeta::CommitmentList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified regional disk type. Gets a list of available disk types
        # by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] disk_type
        #   Name of the disk type to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::DiskType] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::DiskType]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_region_disk_type(project, region, disk_type, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/diskTypes/{diskType}', options)
          command.response_representation = Google::Apis::ComputeBeta::DiskType::Representation
          command.response_class = Google::Apis::ComputeBeta::DiskType
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['diskType'] = disk_type unless disk_type.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of regional disk types available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::RegionDiskTypeList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::RegionDiskTypeList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_region_disk_types(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/diskTypes', options)
          command.response_representation = Google::Apis::ComputeBeta::RegionDiskTypeList::Representation
          command.response_class = Google::Apis::ComputeBeta::RegionDiskTypeList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a snapshot of this regional disk.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] disk
        #   Name of the regional persistent disk to snapshot.
        # @param [Google::Apis::ComputeBeta::Snapshot] snapshot_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_region_disk_snapshot(project, region, disk, snapshot_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/disks/{disk}/createSnapshot', options)
          command.request_representation = Google::Apis::ComputeBeta::Snapshot::Representation
          command.request_object = snapshot_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified regional persistent disk. Deleting a regional disk
        # removes all the replicas of its data permanently and is irreversible. However,
        # deleting a disk does not delete any snapshots previously made from the disk.
        # You must separately delete snapshots.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] disk
        #   Name of the regional persistent disk to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_region_disk(project, region, disk, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/regions/{region}/disks/{disk}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns a specified regional persistent disk.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] disk
        #   Name of the regional persistent disk to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Disk] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Disk]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_region_disk(project, region, disk, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/disks/{disk}', options)
          command.response_representation = Google::Apis::ComputeBeta::Disk::Representation
          command.response_class = Google::Apis::ComputeBeta::Disk
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a persistent regional disk in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [Google::Apis::ComputeBeta::Disk] disk_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] source_image
        #   Optional. Source image to restore onto a disk.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_region_disk(project, region, disk_object = nil, request_id: nil, source_image: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/disks', options)
          command.request_representation = Google::Apis::ComputeBeta::Disk::Representation
          command.request_object = disk_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['sourceImage'] = source_image unless source_image.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of persistent disks contained within the specified region.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::DiskList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::DiskList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_region_disks(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/disks', options)
          command.response_representation = Google::Apis::ComputeBeta::DiskList::Representation
          command.response_class = Google::Apis::ComputeBeta::DiskList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Resizes the specified regional persistent disk.
        # @param [String] project
        #   The project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] disk
        #   Name of the regional persistent disk.
        # @param [Google::Apis::ComputeBeta::RegionDisksResizeRequest] region_disks_resize_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def resize_region_disk(project, region, disk, region_disks_resize_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/disks/{disk}/resize', options)
          command.request_representation = Google::Apis::ComputeBeta::RegionDisksResizeRequest::Representation
          command.request_object = region_disks_resize_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on the target regional disk.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The region for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::RegionSetLabelsRequest] region_set_labels_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_region_disk_labels(project, region, resource, region_set_labels_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/disks/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeBeta::RegionSetLabelsRequest::Representation
          command.request_object = region_set_labels_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] resource
        #   Name of the resource for this request.
        # @param [Google::Apis::ComputeBeta::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_region_disk_iam_permissions(project, region, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/disks/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeBeta::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeBeta::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Schedules a group action to remove the specified instances from the managed
        # instance group. Abandoning an instance does not delete the instance, but it
        # does remove the instance from any target pools that are applied by the managed
        # instance group. This method reduces the targetSize of the managed instance
        # group by the number of instances that you abandon. This operation is marked as
        # DONE when the action is scheduled even if the instances have not yet been
        # removed from the group. You must separately verify the status of the
        # abandoning action with the listmanagedinstances method.
        # If the group is part of a backend service that has enabled connection draining,
        # it can take up to 60 seconds after the connection draining duration has
        # elapsed before the VM instance is removed or deleted.
        # You can specify a maximum of 1000 instances with this method per request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] instance_group_manager
        #   Name of the managed instance group.
        # @param [Google::Apis::ComputeBeta::RegionInstanceGroupManagersAbandonInstancesRequest] region_instance_group_managers_abandon_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def abandon_region_instance_group_manager_instances(project, region, instance_group_manager, region_instance_group_managers_abandon_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/abandonInstances', options)
          command.request_representation = Google::Apis::ComputeBeta::RegionInstanceGroupManagersAbandonInstancesRequest::Representation
          command.request_object = region_instance_group_managers_abandon_instances_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified managed instance group and all of the instances in that
        # group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] instance_group_manager
        #   Name of the managed instance group to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_region_instance_group_manager(project, region, instance_group_manager, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:delete, '{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Schedules a group action to delete the specified instances in the managed
        # instance group. The instances are also removed from any target pools of which
        # they were a member. This method reduces the targetSize of the managed instance
        # group by the number of instances that you delete. This operation is marked as
        # DONE when the action is scheduled even if the instances are still being
        # deleted. You must separately verify the status of the deleting action with the
        # listmanagedinstances method.
        # If the group is part of a backend service that has enabled connection draining,
        # it can take up to 60 seconds after the connection draining duration has
        # elapsed before the VM instance is removed or deleted.
        # You can specify a maximum of 1000 instances with this method per request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] instance_group_manager
        #   Name of the managed instance group.
        # @param [Google::Apis::ComputeBeta::RegionInstanceGroupManagersDeleteInstancesRequest] region_instance_group_managers_delete_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_region_instance_group_manager_instances(project, region, instance_group_manager, region_instance_group_managers_delete_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/deleteInstances', options)
          command.request_representation = Google::Apis::ComputeBeta::RegionInstanceGroupManagersDeleteInstancesRequest::Representation
          command.request_object = region_instance_group_managers_delete_instances_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns all of the details about the specified managed instance group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] instance_group_manager
        #   Name of the managed instance group to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::InstanceGroupManager] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::InstanceGroupManager]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_region_instance_group_manager(project, region, instance_group_manager, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}', options)
          command.response_representation = Google::Apis::ComputeBeta::InstanceGroupManager::Representation
          command.response_class = Google::Apis::ComputeBeta::InstanceGroupManager
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a managed instance group using the information that you specify in the
        # request. After the group is created, it schedules an action to create
        # instances in the group using the specified instance template. This operation
        # is marked as DONE when the group is created even if the instances in the group
        # have not yet been created. You must separately verify the status of the
        # individual instances with the listmanagedinstances method.
        # A regional managed instance group can contain up to 2000 instances.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [Google::Apis::ComputeBeta::InstanceGroupManager] instance_group_manager_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_region_instance_group_manager(project, region, instance_group_manager_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/instanceGroupManagers', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceGroupManager::Representation
          command.request_object = instance_group_manager_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of managed instance groups that are contained within the
        # specified region.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::RegionInstanceGroupManagerList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::RegionInstanceGroupManagerList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_region_instance_group_managers(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:get, '{project}/regions/{region}/instanceGroupManagers', options)
          command.response_representation = Google::Apis::ComputeBeta::RegionInstanceGroupManagerList::Representation
          command.response_class = Google::Apis::ComputeBeta::RegionInstanceGroupManagerList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists the instances in the managed instance group and instances that are
        # scheduled to be created. The list includes any current actions that the group
        # has scheduled for its instances.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [String] filter
        # @param [Fixnum] max_results
        # @param [String] order_by
        # @param [String] page_token
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::RegionInstanceGroupManagersListInstancesResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::RegionInstanceGroupManagersListInstancesResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_region_instance_group_manager_managed_instances(project, region, instance_group_manager, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/listManagedInstances', options)
          command.response_representation = Google::Apis::ComputeBeta::RegionInstanceGroupManagersListInstancesResponse::Representation
          command.response_class = Google::Apis::ComputeBeta::RegionInstanceGroupManagersListInstancesResponse
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['order_by'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a managed instance group using the information that you specify in the
        # request. This operation is marked as DONE when the group is patched even if
        # the instances in the group are still in the process of being patched. You must
        # separately verify the status of the individual instances with the
        # listmanagedinstances method. This method supports PATCH semantics and uses the
        # JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] instance_group_manager
        #   The name of the instance group manager.
        # @param [Google::Apis::ComputeBeta::InstanceGroupManager] instance_group_manager_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_region_instance_group_manager(project, region, instance_group_manager, instance_group_manager_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:patch, '{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}', options)
          command.request_representation = Google::Apis::ComputeBeta::InstanceGroupManager::Representation
          command.request_object = instance_group_manager_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Schedules a group action to recreate the specified instances in the managed
        # instance group. The instances are deleted and recreated using the current
        # instance template for the managed instance group. This operation is marked as
        # DONE when the action is scheduled even if the instances have not yet been
        # recreated. You must separately verify the status of the recreating action with
        # the listmanagedinstances method.
        # If the group is part of a backend service that has enabled connection draining,
        # it can take up to 60 seconds after the connection draining duration has
        # elapsed before the VM instance is removed or deleted.
        # You can specify a maximum of 1000 instances with this method per request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] instance_group_manager
        #   Name of the managed instance group.
        # @param [Google::Apis::ComputeBeta::RegionInstanceGroupManagersRecreateRequest] region_instance_group_managers_recreate_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def recreate_region_instance_group_manager_instances(project, region, instance_group_manager, region_instance_group_managers_recreate_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/recreateInstances', options)
          command.request_representation = Google::Apis::ComputeBeta::RegionInstanceGroupManagersRecreateRequest::Representation
          command.request_object = region_instance_group_managers_recreate_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Changes the intended size for the managed instance group. If you increase the
        # size, the group schedules actions to create new instances using the current
        # instance template. If you decrease the size, the group schedules delete
        # actions on one or more instances. The resize operation is marked DONE when the
        # resize actions are scheduled even if the group has not yet added or deleted
        # any instances. You must separately verify the status of the creating or
        # deleting actions with the listmanagedinstances method.
        # If the group is part of a backend service that has enabled connection draining,
        # it can take up to 60 seconds after the connection draining duration has
        # elapsed before the VM instance is removed or deleted.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] instance_group_manager
        #   Name of the managed instance group.
        # @param [Fixnum] size
        #   Number of instances that should exist in this instance group manager.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def resize_region_instance_group_manager(project, region, instance_group_manager, size, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/resize', options)
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['size'] = size unless size.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Modifies the autohealing policy for the instances in this managed instance
        # group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] instance_group_manager
        #   Name of the managed instance group.
        # @param [Google::Apis::ComputeBeta::RegionInstanceGroupManagersSetAutoHealingRequest] region_instance_group_managers_set_auto_healing_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeBeta::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeBeta::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_region_instance_group_manager_auto_healing_policies(project, region, instance_group_manager, region_instance_group_managers_set_auto_healing_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command =  make_simple_command(:post, '{project}/regions/{region}/instanceGroupManagers/{instanceGroupManager}/setAutoHealingPolicies', options)
          command.request_representation = Google::Apis::ComputeBeta::RegionInstanceGroupManagersSetAutoHealingRequest::Representation
          command.request_object = region_instance_group_managers_set_auto_healing_request_object
          command.response_representation = Google::Apis::ComputeBeta::Operation::Representation
          command.response_class = Google::Apis::ComputeBeta::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the instance template to use when creating new instances or recreating
        # instances in this group. Existing instances are not affected.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeBeta::RegionInstanceGroupManagersSetTemplateRequest] region_instance_group_managers_set_template_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characte