/*******************************************************************************
 * Copyright (c) 2010 Engineering Group.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Marco Cortella (Engineering Group) - initial API and implementation and/or initial documentation
 *******************************************************************************/
package org.eclipse.ebam.emf;

import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.FlushModeType;
import javax.persistence.NoResultException;
import javax.persistence.Query;

import org.eclipse.ebam.emf.interfaces.ICacheTransaction;
import org.eclipse.ebam.model.entities.Journal;
import org.eclipse.ebam.model.entities.MessageContent;
import org.eclipse.ebam.model.entities.SbiResources;
import org.eclipse.ebam.model.entities.ServiceInstance;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class CachingTransaction extends EMF implements ICacheTransaction {
    /**
     * Logger.
     */
    private static Logger logger = LoggerFactory
	    .getLogger(CachingTransaction.class);

    /**
     * Method that executes database query to extract
     * <code>ServiceInstance</code> by label.
     * 
     * @param label
     *            to search
     * @param serviceID
     *            service id
     * @return <code>ServiceInstance</code>
     * @throws Exception
     *             error
     */
    @SuppressWarnings("finally")
    private final ServiceInstance getServiceInstanceByLabel(
	    final EntityManager em, final String label, final Integer serviceID)
	    throws Exception {
	logger.debug("IN");
	ServiceInstance servInst = null;
	try {
	    Query query = em
		    .createQuery("select e from ServiceInstance "
			    + "e where e.label = :label and e.service.idService = :serviceID");
	    query.setParameter("label", label);
	    query.setParameter("serviceID", serviceID);
	    servInst = (ServiceInstance) query.getSingleResult();
	} catch (NoResultException e) {

	} catch (Exception e) {
	    logger.error(e.getMessage(), e);
	    throw e;
	} finally {
	    logger.debug("OUT");
	    return servInst;
	}
    }

    /**
     * Method that stores Journal, ServiceInstance and relevant data on db.
     * 
     * @param j
     *            <code>Journal</code> to be stored
     * @param serviceInstance
     *            <code>ServiceInstance</code> already present in DB
     * @param messageContents
     *            List of relevant data to be stored
     * @return operation result
     */
    public boolean saveJournal(final Journal j,
	    final List<MessageContent> messageContents) {
	logger.debug("IN");
	boolean isSaved = false;

	EntityManager em = getEMF().createEntityManager();
	em.getTransaction().begin();
	// Set the flush-mode so the manager query does not force a
	// write before the read.
	em.setFlushMode(FlushModeType.COMMIT);

	String siLabel = j.getServiceInstance().getLabel();
	Integer serviceID = j.getServiceInstance().getService().getIdService();
	try {
	    ServiceInstance serviceInstance = getServiceInstanceByLabel(em,
		    siLabel, serviceID);

	    if (serviceInstance == null) {

		em.persist(j.getServiceInstance());
		em.flush();
		logger.info("Service Instance "+ j.getServiceInstance().getLabel() + " saved");
	    } else {
		// IF the message contains the Service END TIME, I have to insert this 
		// information on DB
		logger.info("There is one Service Instance already on DB");
		if(j.getServiceInstance().getEnddate() != null) {
		    logger.info("j.getServiceInstance().getEnddate() != null.  SAVE END DATE");
		    serviceInstance.setEnddate(j.getServiceInstance().getEnddate());
		    j.setServiceInstance(serviceInstance);
		    em.merge(serviceInstance);
		    em.flush();
		}else{
		    j.setServiceInstance(serviceInstance); 
		}
		
		
	    }
	    em.persist(j);
	    em.flush();

	    logger.info("Journal " + j.getJurnalId() +" / "+j.getLabel() +" saved");
	    if (messageContents != null) {
		for (int i = 0; i < messageContents.size(); i++) {
		    em.persist(messageContents.get(i));
		    logger.info("Relevant Data Saved:"+((MessageContent)messageContents.get(i)).getName()+" - "+((MessageContent)messageContents.get(i)).getValue());
		}
		logger.info("Relevant data saved");
	    }
	    
	    em.getTransaction().commit();
	    isSaved = true;

	} catch (Exception e) {
	    if (em.getTransaction() != null && em.getTransaction().isActive()) {
		em.getTransaction().rollback();
	    }
	    logger.error("ROLL BACK....", e);
	    throw e;
	} finally {
	    em.close();
	    logger.debug("OUT.isSaved="+isSaved);
	    return isSaved;
	}
    }

    public SbiResources getSbiResourcesByName(final String name) {
	logger.debug("IN");
	SbiResources sbiResources = null;
	EntityManager em = getEMF().createEntityManager();
	try {
	    Query query = em
		    .createQuery("select r from SbiResources r where r.resourceName = :name ");
	    query.setParameter("name", name);
	    sbiResources = (SbiResources) query.getSingleResult();
	} catch (NoResultException e) {
	    // logger.debug(e.getMessage(), e);
	} catch (Exception e) {
	    logger.error(e.getMessage(), e);
	    throw e;
	} finally {
	    em.close();
	    logger.debug("OUT");
	    return sbiResources;
	}

    }
}
