/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: RDFXMLLoader.java,v 1.1 2007/03/18 08:39:03 lzhang Exp $
 */

package org.eclipse.eodm.rdf.resource.parser.impl;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;

import org.eclipse.eodm.rdf.rdfweb.Document;
import org.eclipse.eodm.rdf.resource.parser.RDFXMLParser;
import org.eclipse.eodm.rdf.resource.parser.exception.ParserException;
import org.eclipse.eodm.rdf.resource.parser.exception.ParserIOException;
import org.eclipse.eodm.rdf.resource.parser.xml.RDFXMLParserImpl;
import org.xml.sax.InputSource;

/**
 * RDFXMLLoader
 * 
 * Load RDF/XML Model from a file/stream in RDF/XML syntax to construct an ODM
 * model.
 * 
 */
public final class RDFXMLLoader {
    private RDFXMLLoader() {
    }

    /**
     * The default base uri when xml:base is absent from the document to be
     * parsed
     */
    private static String DEFAULT_BASE_URI = "http://default-base-uri";

    /**
     * To get the default base uri.
     * 
     * @return the default base uri
     */
    public static String getDefaultBaseURI() {
        return DEFAULT_BASE_URI;
    }

    /**
     * To set the value of the default base uri.
     * 
     * @param uri
     *            the default base uri
     */
    public static void setDefaultBaseURI(String uri) {
        if (uri != null)
            DEFAULT_BASE_URI = uri;
    }

    /**
     * Load RDF/XML Model from rdf file.
     * 
     * @param fileName
     *            The rdf file name
     * @return RDF Model
     * @throws ParserException
     *             exception when parsing rdf or wrapping it.
     */
    public static Document loadFromFile(String fileName)
            throws ParserIOException {
        return loadFromFile(fileName, null);
    }

    /**
     * Load RDF/XML Model from rdf file.
     * 
     * @param fileName
     *            The file name.
     * @param charset
     *            The encoding charset.
     * @return RDF Model
     * @throws ParserException
     *             exception when parsing rdf or wrapping it.
     */
    public static Document loadFromFile(String fileName, String charset)
            throws ParserIOException {
        FileInputStream input = null;

        try {
            input = new FileInputStream(fileName);
        } catch (FileNotFoundException e) {
            throw new ParserIOException(e.getMessage(), e);
        }

        return loadFromStream(input, charset);
    }

    /**
     * Load RDF/XML Model from stream.
     * 
     * @param input
     *            The input stream.
     * @return RDF Model
     * @throws ParserException
     *             exception when parsing rdf or wrapping it.
     */
    public static Document loadFromStream(InputStream input)
            throws ParserIOException {
        return loadFromStream(input, null);
    }

    /**
     * Load RDF/XML Model from stream.
     * 
     * @param input
     *            The input stream.
     * @param charset
     *            The encoding charset.
     * @return RDF Model
     * @throws ParserException
     *             exception when parsing rdf or wrapping it.
     */
    public static Document loadFromStream(InputStream input, String charset)
            throws ParserIOException {
        if (input == null)
            return null;

        InputSource source = new InputSource(input);
        if (charset != null)
            source.setEncoding(charset);

        /**
         * 1. Instantiate a triple analyser and initialize it. It will catch
         * triples produced by RDF Parser and wrap them into an RDF Model:
         * RDFDescription.
         */
        RDFTripleAnalyzer tripleAnalyser = new RDFTripleAnalyzer();
        tripleAnalyser.initialize(DEFAULT_BASE_URI);

        /**
         * 2. Create an RDF Parser instance, and call its parse method to parse
         * the RDF input stream into triples.
         * 
         * The triple analyser embbeded in the result handler will analyse all
         * the triples generated during the parsing process. Just those triples
         * with "rdf:type" predicate will be handled in the step. The analyser
         * identifies all the elements with an explicit type, eg. RDFSClass,
         * RDFAlt, RDFSeq, RDFBag, RDFProperty, RDFStatement, RDFXMLLiteral,
         * RDFSLiteral, RDFList(rdf:first, rdf:rest), RDFSContainer(rdf:_n),
         * RDFSDatatype, RDFSResource. And also it handles some axioms in this
         * step, eg. rdfs:subClassOf, rdfs:subPropertyOf, rdfs:domain,
         * rdfs:range, rdf:subject, rdf:predicate, rdf:object etc.
         */
        RDFXMLParser parser = new RDFXMLParserImpl();
        /** 2.1 TO set the namespace and statement handler. */
        RDFParserResultHandler resultHandler = new RDFParserResultHandler(
                tripleAnalyser);
        parser.setNamespaceHandler(resultHandler);
        parser.setStatementHandler(resultHandler);
        /** 2.2 Start the parsing process: Identify all the type elements */
        try {
            parser.parse(source, DEFAULT_BASE_URI);
        } catch (ParserIOException e) {
            throw new ParserIOException("RDF parsing IO error", e);
        } catch (ParserException e) {
            throw new ParserIOException("RDF parsing error", e);
        }
        return (tripleAnalyser.getDocument());
    }
}