/**
 * <copyright>
 * 
 * Copyright (c) 2014 itemis and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     itemis - Initial API and implementation
 * 
 * </copyright>
 */
package org.eclipse.sphinx.jdt.loaders;

import com.google.common.base.Objects;
import java.io.File;
import java.net.URI;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IPath;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IType;
import org.eclipse.sphinx.jdt.loaders.DelegatingCompositeBundleClassLoader;
import org.eclipse.sphinx.jdt.util.JavaExtensions;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.osgi.framework.Bundle;

@SuppressWarnings("all")
public class ProjectClassLoader extends URLClassLoader {
  private IJavaProject javaProject;
  
  protected static URL[] getJavaOutputURLs(final IJavaProject javaProject) {
    try {
      URL[] _xblockexpression = null;
      {
        Assert.isNotNull(javaProject);
        boolean _and = false;
        boolean _exists = javaProject.exists();
        if (!_exists) {
          _and = false;
        } else {
          boolean _isOpen = javaProject.isOpen();
          _and = _isOpen;
        }
        Assert.isLegal(_and);
        IPath _outputLocation = javaProject.getOutputLocation();
        IPath _location = JavaExtensions.getLocation(_outputLocation);
        File _file = null;
        if (_location!=null) {
          _file=_location.toFile();
        }
        URI _uRI = null;
        if (_file!=null) {
          _uRI=_file.toURI();
        }
        URL _uRL = null;
        if (_uRI!=null) {
          _uRL=_uRI.toURL();
        }
        final URL outputURL = _uRL;
        URL[] _xifexpression = null;
        boolean _notEquals = (!Objects.equal(outputURL, null));
        if (_notEquals) {
          _xifexpression = new URL[] { outputURL };
        } else {
          _xifexpression = new URL[] {};
        }
        _xblockexpression = _xifexpression;
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  protected static ClassLoader hookupDependencyClassLoaders(final IJavaProject javaProject, final ClassLoader parent) {
    try {
      ClassLoader _xblockexpression = null;
      {
        Assert.isNotNull(javaProject);
        boolean _and = false;
        boolean _exists = javaProject.exists();
        if (!_exists) {
          _and = false;
        } else {
          boolean _isOpen = javaProject.isOpen();
          _and = _isOpen;
        }
        Assert.isLegal(_and);
        ClassLoader lastClassLoader = parent;
        final IClasspathEntry[] entries = javaProject.getResolvedClasspath(true);
        final Function1<IClasspathEntry, Boolean> _function = new Function1<IClasspathEntry, Boolean>() {
          public Boolean apply(final IClasspathEntry it) {
            int _entryKind = it.getEntryKind();
            return Boolean.valueOf((_entryKind == IClasspathEntry.CPE_PROJECT));
          }
        };
        Iterable<IClasspathEntry> _filter = IterableExtensions.<IClasspathEntry>filter(((Iterable<IClasspathEntry>)Conversions.doWrapArray(entries)), _function);
        final Function1<IClasspathEntry, String> _function_1 = new Function1<IClasspathEntry, String>() {
          public String apply(final IClasspathEntry it) {
            IPath _path = it.getPath();
            return _path.segment(0);
          }
        };
        final Iterable<String> requiredProjectNames = IterableExtensions.<IClasspathEntry, String>map(_filter, _function_1);
        for (final String requiredProjectName : requiredProjectNames) {
          IJavaProject _javaProject = JavaExtensions.getJavaProject(requiredProjectName);
          ProjectClassLoader _projectClassLoader = new ProjectClassLoader(_javaProject, lastClassLoader);
          lastClassLoader = _projectClassLoader;
        }
        final Function1<IClasspathEntry, Boolean> _function_2 = new Function1<IClasspathEntry, Boolean>() {
          public Boolean apply(final IClasspathEntry it) {
            int _entryKind = it.getEntryKind();
            return Boolean.valueOf((_entryKind == IClasspathEntry.CPE_LIBRARY));
          }
        };
        Iterable<IClasspathEntry> _filter_1 = IterableExtensions.<IClasspathEntry>filter(((Iterable<IClasspathEntry>)Conversions.doWrapArray(entries)), _function_2);
        final Function1<IClasspathEntry, File> _function_3 = new Function1<IClasspathEntry, File>() {
          public File apply(final IClasspathEntry it) {
            return JavaExtensions.getFile(it);
          }
        };
        Iterable<File> _map = IterableExtensions.<IClasspathEntry, File>map(_filter_1, _function_3);
        final Function1<File, Boolean> _function_4 = new Function1<File, Boolean>() {
          public Boolean apply(final File it) {
            return Boolean.valueOf(it.exists());
          }
        };
        Iterable<File> _filter_2 = IterableExtensions.<File>filter(_map, _function_4);
        final Function1<File, URL> _function_5 = new Function1<File, URL>() {
          public URL apply(final File it) {
            try {
              URI _uRI = it.toURI();
              return _uRI.toURL();
            } catch (Throwable _e) {
              throw Exceptions.sneakyThrow(_e);
            }
          }
        };
        final Iterable<URL> libraryURLs = IterableExtensions.<File, URL>map(_filter_2, _function_5);
        final ArrayList<URL> javaLibraryURLs = CollectionLiterals.<URL>newArrayList();
        final List<Bundle> requiredBundles = CollectionLiterals.<Bundle>newArrayList();
        for (final URL libraryURL : libraryURLs) {
          {
            final Bundle requiredBundle = JavaExtensions.getBundle(libraryURL);
            boolean _notEquals = (!Objects.equal(requiredBundle, null));
            if (_notEquals) {
              requiredBundles.add(requiredBundle);
            } else {
              javaLibraryURLs.add(libraryURL);
            }
          }
        }
        boolean _isEmpty = requiredBundles.isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          DelegatingCompositeBundleClassLoader _delegatingCompositeBundleClassLoader = new DelegatingCompositeBundleClassLoader(lastClassLoader, requiredBundles);
          lastClassLoader = _delegatingCompositeBundleClassLoader;
        }
        boolean _isEmpty_1 = javaLibraryURLs.isEmpty();
        boolean _not_1 = (!_isEmpty_1);
        if (_not_1) {
          URLClassLoader _uRLClassLoader = new URLClassLoader(((URL[])Conversions.unwrapArray(javaLibraryURLs, URL.class)), lastClassLoader);
          lastClassLoader = _uRLClassLoader;
        }
        _xblockexpression = lastClassLoader;
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public ProjectClassLoader(final IJavaProject javaProject) {
    this(javaProject, Thread.currentThread().getContextClassLoader());
  }
  
  public ProjectClassLoader(final IJavaProject javaProject, final ClassLoader parent) {
    super(ProjectClassLoader.getJavaOutputURLs(javaProject), ProjectClassLoader.hookupDependencyClassLoaders(javaProject, parent));
    this.javaProject = javaProject;
  }
  
  protected Class<?> loadClass(final String name, final boolean resolve) throws ClassNotFoundException {
    try {
      Class<?> _xblockexpression = null;
      {
        final IType type = this.javaProject.findType(name);
        boolean _and = false;
        boolean _and_1 = false;
        boolean _notEquals = (!Objects.equal(type, null));
        if (!_notEquals) {
          _and_1 = false;
        } else {
          boolean _isBinary = type.isBinary();
          boolean _not = (!_isBinary);
          _and_1 = _not;
        }
        if (!_and_1) {
          _and = false;
        } else {
          IPath _path = this.javaProject.getPath();
          IPath _path_1 = type.getPath();
          boolean _isPrefixOf = _path.isPrefixOf(_path_1);
          _and = _isPrefixOf;
        }
        if (_and) {
          return this.findClass(name);
        }
        _xblockexpression = super.loadClass(name, resolve);
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public String toString() {
    Class<? extends ProjectClassLoader> _class = this.getClass();
    String _name = _class.getName();
    String _plus = (_name + " [project=");
    IProject _project = this.javaProject.getProject();
    String _name_1 = _project.getName();
    String _plus_1 = (_plus + _name_1);
    return (_plus_1 + "]");
  }
  
  public IJavaProject getProject() {
    return this.javaProject;
  }
}
