/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*-
 *
 * Copyright (C) 2007 David Zeuthen <davidz@redhat.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>

#include <devkit-gobject/devkit-gobject.h>

static char        **opt_subsys      = NULL;
static char        **opt_native_path = NULL;
static gboolean      opt_monitor     = FALSE;
static gboolean      opt_version     = FALSE;

static gboolean
print_prop (DevkitDevice *device, const char *key, const char *value, gpointer user_data)
{
        g_print ("    %s=%s\n", key, value);
        return FALSE;
}

static void
print_device (DevkitDevice *device)
{
        const char *device_file;
        const char **symlinks;
        int n;

        device_file = devkit_device_get_device_file (device);
        symlinks = devkit_device_get_device_file_symlinks (device);

        g_print ("device:           %s\n"
                 "  subsystem:      %s\n"
                 "  device file:    %s\n",
                 devkit_device_get_native_path (device),
                 devkit_device_get_subsystem (device),
                 device_file != NULL && strlen (device_file) > 0 ? device_file : "(none)");

        for (n = 0; symlinks != NULL && symlinks[n] != NULL; n++) {
                g_print ("    symlink:      %s\n", symlinks[n]);
        }
        g_print ("  properties:\n");
        devkit_device_properties_foreach (device, print_prop, NULL);
}

static const char *
get_time (void)
{
        struct timeval tnow;
        struct tm *tlocaltime;
        static char tbuf[256];
        static char buf[256];
        GTimeVal t;

        gettimeofday (&tnow, NULL);
        tlocaltime = localtime ((time_t *) &tnow.tv_sec);
        strftime (tbuf, sizeof (tbuf), "%H:%M:%S", tlocaltime);

        g_get_current_time (&t);
        g_snprintf (buf, sizeof(buf), "%s.%06d", tbuf, (int) (t.tv_usec));
        return buf;
}

static void
device_event_signal_handler (DevkitClient *client,
                             const char   *action,
                             DevkitDevice *device,
                             gpointer      user_data)
{
        g_print ("----------------------------------------------------------------------\n");
        g_print ("action=%s @ %s\n", action, get_time ());
        print_device (device);
        g_print ("\n");
}


int
main (int argc, char **argv)
{
        int                  ret;
        GOptionContext      *context;
        GError              *error = NULL;
        DevkitClient        *client;
        static GOptionEntry  entries []     = {
                { "subsystem", 's', 0, G_OPTION_ARG_STRING_ARRAY, &opt_subsys, "Enumerate by subsystem (can be used multiple times)", NULL },
                { "native-path", 'n', 0, G_OPTION_ARG_STRING_ARRAY, &opt_native_path, "Enumerate by native path (can be used multiple times)", NULL },
                { "monitor", 'm', 0, G_OPTION_ARG_NONE, &opt_monitor, "Monitor events from the DeviceKit daemon", NULL },
                { "version", 'v', 0, G_OPTION_ARG_NONE, &opt_version, "Print version of client and daemon", NULL },
                { NULL }
        };

        ret = 1;
        client = NULL;

        g_type_init ();

        context = g_option_context_new ("DeviceKit client");
        g_option_context_add_main_entries (context, entries, NULL);
        g_option_context_parse (context, &argc, &argv, NULL);

        client = devkit_client_new ((const char **) opt_subsys);
        if (!devkit_client_connect (client, &error)) {
                g_warning ("Cannot connect to DeviceKit daemon: %s", error->message);
                g_error_free (error);
                goto out;
        }

        if (opt_version) {
                g_print ("DeviceKit client version %s\n"
                         "DeviceKit daemon version %s\n",
                         PACKAGE_VERSION,
                         devkit_client_get_daemon_version (client));
                ret = 0;
                goto out;
        }

        if (opt_monitor) {
                GMainLoop *loop;
                loop = g_main_loop_new (NULL, FALSE);
                g_signal_connect (client, "device-event", G_CALLBACK (device_event_signal_handler), NULL);
                g_main_loop_run (loop);
                ret = 0;
                goto out;
        }

        /* just print */
        if (opt_subsys != NULL) {
                GList *l;
                GList *devices;

                devices = devkit_client_enumerate_by_subsystem (client, (const char **) opt_subsys, &error);
                if (error != NULL) {
                        g_warning ("Error calling EnumerateBySubsystem: %s", error->message);
                        g_error_free (error);
                        goto out;
                }
                for (l = devices; l != NULL; l = l->next) {
                        DevkitDevice *device = DEVKIT_DEVICE (l->data);

                        if (l != devices)
                                g_print ("\n");

                        print_device (device);
                }
                g_list_foreach (devices, (GFunc) g_object_unref, NULL);
                g_list_free (devices);

                ret = 0;
                goto out;

        /* just print */
        } else if (opt_native_path != NULL) {
                GList *l;
                GList *devices;

                devices = devkit_client_enumerate_by_native_path (client, (const char **) opt_native_path, &error);
                if (error != NULL) {
                        g_warning ("Error calling EnumerateByNativePath: %s", error->message);
                        g_error_free (error);
                        goto out;
                }
                for (l = devices; l != NULL; l = l->next) {
                        DevkitDevice *device = DEVKIT_DEVICE (l->data);

                        if (l != devices)
                                g_print ("\n");

                        print_device (device);
                }
                g_list_foreach (devices, (GFunc) g_object_unref, NULL);
                g_list_free (devices);

                ret = 0;
                goto out;

        } else {
                char *s;
                s = g_option_context_get_help (context, FALSE, NULL);
                g_print ("%s\n", s);
                goto out;
        }


out:
        g_option_context_free (context);
        if (client != NULL)
                g_object_unref (client);
        return ret;
}
