/*****************************************************************************
  Module: mpgrecorder.h - Defines the MPEGrecorder class.
  Copyright (C) 1999  Andrew L. Sandoval

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *****************************************************************************/
/* Module: mpgrecorder.h                                                     */
/* Developer: Andrew L. Sandoval                                             */
/* Purpose: Implements the MPEGrecorder class.  MPEGrecorder is invoked to   */
/* record MPEG content from an MPEG capture card (i.e. Antex, Snazzi, etc.)  */
/* Depending on the Constructor selected output can be sent directly to an   */
/* open file HANDLE for network streaming, etc., or a file will be opened.   */
/* By default C:\temp\date-time.mp3 will be used.                            */
/*****************************************************************************/
#include <iostream>
#include <string>
#include <sstream>
#include <map>
#include <vector>
#include <deque>
#include <windows.h>
#include <mmsystem.h>
#include <mmreg.h>

using namespace std;

HWAVEIN createWaveHandle(DWORD bitrate, DWORD sampFreq, UINT deviceId, 
						 MMRESULT* lastResult, HANDLE event, 
						 int *normalWave = NULL,
						 WORD layer = ACM_MPEG_LAYER3);
HANDLE createEncoderPipe(int normalWave, DWORD bitrate, DWORD sampFreq,
						 HANDLE *childReadPipe, PROCESS_INFORMATION *pi,
						 const char *filename);
HANDLE createUnnamedFile(int normalWave = 0);
HANDLE createNamedFile(const char *filename, int normalWave = 0);
const int mpegFrameSize = 1152 * 2 /* stereo */;
const int waveBufferSize = mpegFrameSize * 200;
const int mpegBufferSize =  (144 * 128000/44100) * 200;

class MPEGrecorder
{
private:
	HANDLE bufferFullEvent;		//Windows will post when it fills a buffer
	HANDLE notifyUser;			//Used to notify the caller that we are done
	MMRESULT lastResult;		//In case there is an error
	HWAVEIN hdev;				//Handle to the sound card device
	int normalWave;				//Set if the device does not support MP3 capture
	DWORD record_block_size;	//Size of the buffers handed to Windows
	HANDLE writeToHandle;		//File for direct write from an MP3 capable device
	HANDLE encoderPipe;			//pipe to "lame mp3encoder" which will then create the requested file
	bool closeOnDestroy;		//For API use -- not commandline
	bool recording;				//Flag for processing threads
	void (*callback)(MPEGrecorder&);	//User callback (progress indication)
	HANDLE dequeMutex;			//Keep the threads from stepping on the Buffer deque
	deque<WAVEHDR*> fullBuffers;//Full buffers are stored here for processing as the
								//as the processing thread can handle them.
	HANDLE childReadPipe;		//Before the encoder will shutdown it's read pipe needs to be closed
	PROCESS_INFORMATION child_pi;//Contains the thread & process handle to shutdown the encoder
								 //when we are done.  (Windows is apparently not smart enough
								 //to just shutdown when stdin is closed even though the child
								 //is through processing.  A 'lame' bug?
public:
	//Constructors:
	//Creates a new file in %TEMP%doy-hh-mm.mp3 which is closed on destruction
	MPEGrecorder() :  
	  bufferFullEvent( CreateEvent(NULL, FALSE, FALSE, NULL) ),
      notifyUser( CreateEvent(NULL, FALSE, FALSE, NULL) ),
	  lastResult(NULL),
	  hdev( createWaveHandle( 128000, 44100, WAVE_MAPPER, &lastResult, bufferFullEvent, &normalWave ) ),
	  record_block_size( normalWave ? waveBufferSize : mpegBufferSize ),
	  writeToHandle( createUnnamedFile(normalWave) ),
	  encoderPipe( createEncoderPipe(normalWave, 128000, 44100, &childReadPipe, &child_pi, NULL) ),
	  closeOnDestroy(TRUE),
	  recording(FALSE),
	  callback(NULL),
	  dequeMutex( CreateMutex( NULL, FALSE, NULL ) ) { };

	MPEGrecorder(DWORD bitrate, DWORD sampleFrequency = 44100, 
				 UINT deviceId = WAVE_MAPPER, WORD layer = ACM_MPEG_LAYER3,
				 void (*Callback)(MPEGrecorder&) = NULL ) :
	  bufferFullEvent( CreateEvent(NULL, FALSE, FALSE, NULL) ),
	  notifyUser( CreateEvent(NULL, FALSE, FALSE, NULL) ),
	  lastResult(NULL),
	  hdev( createWaveHandle( bitrate, sampleFrequency, deviceId, &lastResult, bufferFullEvent, &normalWave, layer ) ),
	  record_block_size( normalWave ? waveBufferSize : mpegBufferSize ),
	  writeToHandle( createUnnamedFile(normalWave) ),
	  encoderPipe( createEncoderPipe(normalWave, bitrate, sampleFrequency, &childReadPipe, &child_pi, NULL) ),
	  closeOnDestroy( TRUE ),
	  recording(FALSE),
	  callback(Callback),
	  dequeMutex( CreateMutex( NULL, FALSE, NULL ) ) { };

	MPEGrecorder(UINT deviceId, DWORD bitrate = 44100, 
		         DWORD sampleFrequency = 44100, WORD layer = ACM_MPEG_LAYER3,
				 void (*Callback)(MPEGrecorder&) = NULL) :
	  bufferFullEvent( CreateEvent(NULL, FALSE, FALSE, NULL) ),
      notifyUser( CreateEvent(NULL, FALSE, FALSE, NULL) ),
	  lastResult(NULL),
	  hdev( createWaveHandle( bitrate, sampleFrequency, deviceId, &lastResult, bufferFullEvent, &normalWave, layer ) ),
	  record_block_size( normalWave ? waveBufferSize : mpegBufferSize ),
	  writeToHandle( createUnnamedFile(normalWave) ),
	  encoderPipe( createEncoderPipe(normalWave, bitrate, sampleFrequency, &childReadPipe, &child_pi, NULL) ),
	  closeOnDestroy( TRUE ),
	  recording(FALSE),
	  callback(Callback),
	  dequeMutex( CreateMutex( NULL, FALSE, NULL ) ) { };

	//Create the file specified:
	MPEGrecorder(const char *filename, DWORD bitrate = 128000, 
		         DWORD sampleFrequency = 44100, UINT deviceId = WAVE_MAPPER,
				 WORD layer = ACM_MPEG_LAYER3, void (*Callback)(MPEGrecorder&) = NULL) :
	  bufferFullEvent( CreateEvent(NULL, FALSE, FALSE, NULL) ),
      notifyUser( CreateEvent(NULL, FALSE, FALSE, NULL) ),
	  lastResult(NULL),
	  hdev( createWaveHandle( bitrate, sampleFrequency, deviceId, &lastResult, bufferFullEvent, &normalWave, layer ) ),
	  record_block_size( normalWave ? waveBufferSize : mpegBufferSize ),
	  writeToHandle( createNamedFile( filename, normalWave ) ),
	  encoderPipe( createEncoderPipe(normalWave, bitrate, sampleFrequency, &childReadPipe, &child_pi, filename) ),
	  closeOnDestroy( TRUE ),
	  recording(FALSE),
	  callback(Callback),
	  dequeMutex( CreateMutex( NULL, FALSE, NULL ) ) { };

	//Uses an Open Handle:  (API style -- only works with an MP3 capable device -- for now)
	MPEGrecorder(HANDLE writeToHandleIn, bool closeWhenDone = TRUE, DWORD bitrate = 128000, 
		         DWORD sampleFrequency = 44100, UINT deviceId = WAVE_MAPPER,
				 WORD layer = ACM_MPEG_LAYER3, void (*Callback)(MPEGrecorder&) = NULL) :
	  bufferFullEvent( CreateEvent(NULL, FALSE, FALSE, NULL) ),
      notifyUser( CreateEvent(NULL, FALSE, FALSE, NULL) ),
	  lastResult(NULL),
	  hdev( createWaveHandle( bitrate, sampleFrequency, deviceId, &lastResult, bufferFullEvent, &normalWave, layer ) ),
	  record_block_size( normalWave ? waveBufferSize : mpegBufferSize ),
	  writeToHandle( writeToHandleIn ),
	  encoderPipe( createEncoderPipe(normalWave, bitrate, sampleFrequency, &childReadPipe, &child_pi, NULL) ),
	  closeOnDestroy( closeWhenDone ), recording(FALSE),
	  callback(Callback),
	  dequeMutex( CreateMutex( NULL, FALSE, NULL ) ) { };

	virtual ~MPEGrecorder();

	MPEGrecorder& setBlockSize(DWORD blockSize) { record_block_size = blockSize; return *this; };
	DWORD getBlockSize() const                  { return record_block_size; };
	HWAVEIN getDeviceHandle() const             { return hdev; };
	MMRESULT getLastResultCode() const          { return lastResult; };
	bool Recording() const						{ return recording; };
	MPEGrecorder& SetLastResult(MMRESULT r)     { lastResult = r; return *this; };
	MPEGrecorder& SetRecordingFlag(bool v)		{ recording = v;  return *this; };
	HANDLE GetBufferEvent() const				{ return bufferFullEvent; };
	HANDLE GetOutputHandle() const				{ return writeToHandle; };
	HANDLE GetEncoderPipe() const				{ return encoderPipe; };
	MPEGrecorder& shutdownChild();
	int isNormalWave() const					{ return normalWave; };
	MPEGrecorder& AddFullBuffer(WAVEHDR *in);
	WAVEHDR * GetFullBuffer(unsigned long *cnt);
	MPEGrecorder& RunUserCallback()				{ if(callback) SaferUserCallback(); return *this; };
	
	MPEGrecorder& Start();
	MPEGrecorder& Stop();
	HANDLE GetRecordEndEvent() const			{ return notifyUser; };
	MPEGrecorder& WaitForRecordingToEnd()		{ WaitForSingleObject(notifyUser, INFINITE); return *this; };
private:
	MPEGrecorder& SaferUserCallback();
};

typedef struct UserCallBackDataBlockMPG
{
	MPEGrecorder *recorder;
	void (*callback)(MPEGrecorder&);
} UserCallBackDataBlockMPG;


//
// Note: It seems that LAME on NT/Win32 can't really handle stdin
//       quite right.  You have to use -r and doing so makes writing
//		 the WAVE HEADER garbage noise.  SO, use -r -x and lame gets
//		 it right (all except for the shutdown part which might just
//		 be a Windows bug -- Unix works perfectly)
//
#define WAV_ID_RIFF 0x46464952 /* "RIFF" */
#define WAV_ID_WAVE 0x45564157 /* "WAVE" */
#define WAV_ID_FMT  0x20746d66 /* "fmt " */
#define WAV_ID_DATA 0x61746164 /* "data" */
#define WAV_ID_PCM  0x0001

struct WaveHeader
{
	unsigned long riff;
	unsigned long file_length;
	unsigned long wave;
	unsigned long fmt;
	unsigned long fmt_length;
	short fmt_tag;
	short channels;
	unsigned long sample_rate;
	unsigned long bytes_per_second;
	short block_align;
	short bits;
	unsigned long data;
	unsigned long data_length;
};

//44100 bits per second
//128000 bits per second (@16bits = 8000bytes/sec 480000bytes/min * 2 960000bytes/min)
//(144 * BitRate/SampleFreq) = 417.9591836735
//1M / 60 = 17476 bytes per second / 2 = 8738 bytes/sec/channel /16 = 546 bits/sec/channel
//418 * 2channels * 16bits = 13376 bytes/second   + (10*418) = 17556  * 60 sec = ~1M

