/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <setjmp.h>

#include "cmp.h"
#include "kalc.h"


void f_size(void)
{
  /*
   * This function calls the _f_size function through the wrapper.
   */
  
  run1_1_Function(_f_size, "size");
}


void f_TOstr(void)
{
  /*
   * This function calls the _f_TOstr function through the wrapper.
   */
  
  run1_1_Function(_f_TOstr, ">str");
}


void f_strTO(void)
{
  /*
   * This function implements the str> function. It evaluates the
   * string in level one as a series of commands.
   */

  if (enoughArgs(1)) {
    if (type(**tos) == TYPE_STR) {
      /* This is the string the user wants to evaluate */
      char *str = strdup((**tos).value.str);
      /* This is the rest of the command line */
      char *oldstr = strdup(getTillEOL());

      saveLastArgs(1);

      /* Drop string from stack */
      _f_drop();
      /* Evaluate its commands */
      parseLine(str);
      /* And free the space allocated for it */
      free(str);

      /* If the rest of command line isn't empty, execute it */
      if (*oldstr)
	parseLine(oldstr);
      /* Free space allocated for rest of command line */
      if (oldstr)
	free(oldstr);

      getTillEOL(); /* We are finished with this line */
    } else
      doError("str>", ERR_BADARGUMENTTYPE);
  } else
    doError("str>", ERR_TOOFEWARGUMENTS);
}


void f_num(void)
{
  /*
   * This function calls the _f_num function through the wrapper.
   */
  
  run1_1_Function(_f_num, "num");
}


void f_chr(void)
{
  /*
   * This function calls the _f_chr function through the wrapper.
   */
  
  run1_1_Function(_f_chr, "chr");
}


void f_head(void)
{
  /*
   * This function implements the head function. It takes a string and
   * returns a string with just the first character of it.
   */
  
  if (enoughArgs(1)) {
    if (type(**tos) == TYPE_STR) {
      saveLastArgs(1);
      (**tos).value.str[1] = '\0';
    } else
      doError("head", ERR_BADARGUMENTTYPE);
  } else
    doError("head", ERR_TOOFEWARGUMENTS);
}


void f_tail(void)
{
  /*
   * This function implements the tail function. It takes a string and
   * returns the string minus its first character.
   */
  
  if (enoughArgs(1)) {
    if (type(**tos) == TYPE_STR) {
      saveLastArgs(1);
      memmove((**tos).value.str,
	      (**tos).value.str + 1,
	      strlen((**tos).value.str)
	     );
    } else
      doError("tail", ERR_BADARGUMENTTYPE);
  } else
    doError("tail", ERR_TOOFEWARGUMENTS);
}


void f_pos(void)
{
  /*
   * This function calls the _f_pos function through the wrapper.
   */
  
  run2_1_Function(_f_pos, "pos");
}


void f_sub(void)
{
  /*
   * This function calls the _f_sub function through the wrapper.
   */
  
  run3_1_Function(_f_sub, "sub");
}


void f_repl(void)
{
  /*
   * This function calls the _f_repl function through the wrapper.
   */
  
  run3_1_Function(_f_repl, "repl");
}


void f_strTOid(void)
{
  /*
   * This function implements the str>id function (also called $>id).
   * It converts a string into an identifier.
   */
  
  if (enoughArgs(1)) {
    if (type(**tos) == TYPE_STR) {
      saveLastArgs(1);
      (**tos).type = TYPE_ID;
    } else
      doError("str>id", ERR_BADARGUMENTTYPE);
  } else
    doError("str>id", ERR_TOOFEWARGUMENTS);
}


void f_idTOstr(void)
{
  /*
   * This function implements the id>str function (also called id>$).
   * It converts an identifier into a string.
   */
  
  if (enoughArgs(1)) {
    if (type(**tos) == TYPE_ID) {
      saveLastArgs(1);
      (**tos).type = TYPE_STR;
    } else
      doError("id>str", ERR_BADARGUMENTTYPE);
  } else
    doError("id>str", ERR_TOOFEWARGUMENTS);
}

  

Object _f_strcat(Object n, Object p, int *err)
{
  /*
   * This function concatenates the strings n and p, and returns the
   * result, which is malloc()'ed.
   */
  
  char *str;
  
  if (type(n) != TYPE_STR && type(n) != TYPE_ID)
    n = _f_TOstr(n, err);
  if (type(p) != TYPE_STR && type(p) != TYPE_ID)
    p = _f_TOstr(p, err);
  
  str = (char *) malloc(strlen(n.value.str) + strlen(p.value.str) + 1);
  if (!str) {
    *err = ERR_NOTENOUGHMEMORY;
  } else {
    *err = ERR_NOERR;
    strcpy(str, n.value.str);
    strcat(str, p.value.str);
    n.value.str = str;
  }

  n.type = TYPE_STR;
  return n;
}


Object _f_size(Object n, int *err)
{
  /*
   * This function returns the size of the string n.
   *
   * For real and complex numbers and identifiers, returns 1. (HP48
   * behavior).
   */

  switch (type(n)) {
  case TYPE_REAL:
  case TYPE_CMP:
  case TYPE_ID:
    *err = ERR_NOERR;
    n.type = TYPE_REAL;
    n.value.real = 1;
    break;

  case TYPE_STR:
    *err = ERR_NOERR;
    n.type = TYPE_REAL;
    n.value.real = strlen(n.value.str);
    break;

  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


Object _f_TOstr(Object n, int *err)
{
  /*
   * This function returns the string representation of the object n.
   * Strings are not touched.
   */
  
  *err = ERR_NOERR;
  
  if (type(n) != TYPE_STR) {
    n.value.str = stackDecomp(n);
    n.type = TYPE_STR;
  }

  return n;
}


Object _f_num(Object n, int *err)
{
  /*
   * This function returns a real number with the ASCII value of the first
   * character of the string n.
   */
  
  Object result;
  result.type = TYPE_REAL;
  
  if (type(n) == TYPE_STR) {
    *err = ERR_NOERR;
    result.value.real = (double) *n.value.str;
  } else
    *err = ERR_BADARGUMENTTYPE;

  return result;
}


Object _f_chr(Object n, int *err)
{
  /*
   * This function returns a one-character string, the character is
   * the character whose ASCII value is n.
   */
  
  Object result;
  result.type = TYPE_STR;

  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    result.value.str = (char *) malloc(2);
    if (!result.value.str)
      doError("Fatal", ERR_NOTENOUGHMEMORY);

    result.value.str[0] = (int) n.value.real;
    result.value.str[1] = '\0';
  } else
    *err = ERR_BADARGUMENTTYPE;

  return result;
}


Object _f_pos(Object n, Object p, int *err)
{
  /*
   * This function searches the string n for a substring p.
   * It returns the position of the start of the first matched substring,
   * or 0 if not found.
   */
  
  Object result;
  result.type = TYPE_REAL;

  if (type(n) == TYPE_STR && type(p) == TYPE_STR) {
    unsigned char *substringStart;

    *err = ERR_NOERR;

    /* If the search string is empty, the return should be zero. So we
       set substringStart to NULL so that the next test returns zero in
       this case. */
    substringStart = (*p.value.str) ? strstr(n.value.str, p.value.str) : NULL;
    result.value.real = ((substringStart != NULL)
			 ? substringStart - n.value.str + 1: 0);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return result;
}


Object _f_sub(Object n, Object p, Object q, int *err)
{
  /*
   * This function returns a substring of n, going from character p to n
   * (inclusive).
   */
  
  Object result;
  result.type = TYPE_STR;

  if (type(n) == TYPE_STR && type(p) == TYPE_REAL && type(q) == TYPE_REAL) {
    char *substring;
    int length;

    *err = ERR_NOERR;

    /* If the start point is less than one, set it to one */
    if (p.value.real < 1)
      p.value.real = 1;
    
    /* If the end point is less than the start point, adjust the end point
       to be equal to the start point, so that an empty string is returned. */
    if (q.value.real < p.value.real)
      q.value.real = p.value.real - 1;

    /* If the end point is greater than the length of the string, adjust the
       end point to be the last character of the string */
    if (q.value.real > strlen(n.value.str))
      q.value.real = strlen(n.value.str);

    length = q.value.real - p.value.real;

    substring = (char *) malloc(length + 1);
    if (!substring) {
      *err = ERR_NOTENOUGHMEMORY;
      return q;
    }

    memcpy(substring, n.value.str + (int) p.value.real - 1, length + 1);
    *(substring + length + 1) = '\0';  /* Add terminating NULL character */

    result.value.str = substring;
  }

  return result;
}


Object _f_repl(Object n, Object p, Object q, int *err)
{
  /*
   * This function changes the characters of the string n starting at
   * position p with the string q.
   */
  
  Object result;
  result.type = TYPE_STR;

  if (type(n) == TYPE_STR && type(p) == TYPE_REAL && type(q) == TYPE_STR) {
    int start = p.value.real;
    int size = strlen(q.value.str);

    *err = ERR_NOERR;

    /* If the start position is less than one, error */
    if (start < 1) {
      *err = ERR_BADARGUMENTVALUE;
      return p;
    }

    /* If the start position is greater than the string size, concatenate */
    if (start > strlen(n.value.str))
      start = strlen(n.value.str) + 1;
    
    /* If the result string will be the same size as the input string */
    if (size <= strlen(n.value.str) - start + 1) {
      result.value.str = strdup(n.value.str);
      if (!result.value.str) {
	*err = ERR_NOTENOUGHMEMORY;
	return p;
      }

      memcpy(result.value.str + start - 1, q.value.str, size);
    } else {          /* The resulting string will be bigger */
      result.value.str = (char *) malloc(start + size);
      if (!result.value.str) {
	*err = ERR_NOTENOUGHMEMORY;
	return p;
      }

      /* Copy initial part of string */
      memcpy(result.value.str, n.value.str, start - 1);
      /* Copy replaced part */
      memcpy(result.value.str + start - 1, q.value.str, size + 1);
    }
  } else
    *err = ERR_BADARGUMENTTYPE;

  return result;
}
