// C-File for pressure modul
// Programmer: Dr. Ralf Wieland
// Version: 0.1 17.11.2002
// Version: 0.2 15.01.2003

#include "druck.h"
#include <time.h>

/**************************************************************************/
/********************** Interface to the sensor ***************************/
/**************************************************************************/

void SendeBit(int hi)
{
    unsigned char q;
    if (hi)
	q = 0xfb;
    else
	q = 0xff;
    outb(q, m_iPort);
    usleep(25);
    outb(q & 0xfd, m_iPort);
    usleep(25);
    outb(q, m_iPort);
}

// Bit auslesen
int LeseBit()
{
  return(!(inb(m_iPort + 1) & 32));
}


// Reset des Drucksensors ausfhren
void ResetSensor(void)
{
    int i;
    for(i = 0; i < 8; i++)
    {
	SendeBit(1);
	SendeBit(0);
    }
    for(i = 0; i < 5; i++)
	SendeBit(0);
}


// AD-Wandler oder Cal-Daten auslesen
unsigned int AdcLesen()
{
    int i;
    unsigned int wert;
    SendeBit(0);    
    // Wandler starten
    // warten bis Wandlung beendet
    for(i=0; i<25; i++){
	usleep(1);
	if(!LeseBit()) break;
    }
        
    // 16 Bit auslesen
    wert = 0;
    for(i = 0; i < 16; i++)
    {
	SendeBit(0);
	wert <<= 1;
	if (LeseBit())
	    wert |= 1;
    }
    SendeBit(0);
    return(wert);
}



// Auslesen eines Kalibrierwortes
unsigned int CalDatenLesen(unsigned befehl)
{
    int i;
    // Befehl senden  
    for (i = 0; i < 12; i++)
    {
	SendeBit(befehl & 1);
	befehl >>= 1;
    }
    // Wert einlesen
    return(AdcLesen());
}


  
// Auslesen eines AD-Wertes
unsigned int ADWertLesen(unsigned befehl)
{
    int i;
    // Befehl senden  
    for (i = 0; i < 10; i++)
    {
	SendeBit(befehl & 1);
	befehl >>= 1;
    }
    SendeBit(0);
    // Wandler auslesen
    return(AdcLesen());
}

// Auslesen der Kalibrierdaten des Drucksensors
// Speichern der Daten im m_wC1 .. m_wC6


int ParameterLesen()
{
    unsigned int w1,w2,w3,w4;
    
    if (!m_iPort)
	return(false);
    ResetSensor();
    ResetSensor();
    
    w1 = CalDatenLesen(0x1157);		// Wort 1 lesen
    w2 = CalDatenLesen(0x00d7);		// Wort 2 lesen
    w3 = CalDatenLesen(0x1137);		// Wort 3 lesen
    w4 = CalDatenLesen(0x00b7);		// Wort 4 lesen
    
    // Parameter aufbereiten
    parameterf.m_wC1 = w1 >> 1;
    parameterf.m_wC2 = ((w3 & 0x3f) << 6) | (w4 & 0x3f);
    parameterf.m_wC3 = w4 >> 6;
    parameterf.m_wC4 = w3 >> 6;
    parameterf.m_wC5 = ((w1 << 10) & 0x400) + (w2 >> 6);
    parameterf.m_wC6 = w2 & 0x3f;
    ResetSensor();
    return(true);
}


// Druck und Temperatur auslesen
// Der Druck wird in 1/10 hPa 
// und die Temperatur in 1/10 C zurckgegeben
int DruckTempLesen()
{
    int d1, d2, ut1, dut, off, sens, x;
    
    if (!m_iPort)
	return(false);
    d1 = ADWertLesen(0x2f);
    d2 = ADWertLesen(0x4f);

    ut1  = 8 * parameterf.m_wC5 + 20224;
    if((d2-ut1)>0)
	dut = d2 - ut1;
    else
	dut = (d2-ut1)-(int)((d2-ut1)/128)*((d2-ut1)/128)/4;
    off  = (int)(parameterf.m_wC2*4 
		 + (int)((int)((parameterf.m_wC4 - 512) * dut) / 4096));
    sens = parameterf.m_wC1 + ((int)(parameterf.m_wC3 * dut) / 1024) + 24576;
    x    = (int)(sens * (d1 - 7168)) / 16384 - off;
    parameterf.druck = (int)(x * 10 / 32 + 250 * 10);
    if((d2-ut1)>0)
	parameterf.temp = (int)(200 + 
				(int)(dut * (parameterf.m_wC6 + 50)) /1024);
    else
	parameterf.temp = (int)(200 + 
				(int)(dut * (parameterf.m_wC6 + 50)) /1024+
	    dut/256);
     return(true);
}

/**************************************************************************/
/********************** Ringbuffer for the pressure values  ***************/
/**************************************************************************/

void init_ring()
{
    int i;
    if((pressure=malloc(RINGLENGTH*sizeof(int)))==NULL){
	fprintf(stderr,"error(init_ring): not enough memory\n");
	exit(1);
    }
    for(i=0; i<RINGLENGTH; i++)
	pressure[i]=-1;
    first=0;
}

void insert_ring(int val)
{
    static float erg=-1.0;
    static int i=0;
    if(val<=0) return;
    if(++i == 12){
	pressure[first]= (int) erg;
	if(++first == RINGLENGTH){
	    first=0;
	}
	i=0;
    }
    if(erg <= 0){erg = (float) val;}
    else{
	erg = 0.9 * erg + 0.1 * (float) val;
    }
}

/**************************************************************************/
/***************************** The two threads  ****************************/
/**************************************************************************/

void *pthr_getpressure( void *pV_data )
{
   time_t now,t;
   t=time(NULL);
   while ( 1 == 1 ) {
       pthread_mutex_lock(&fastmutex);
       now=time(NULL);
       DruckTempLesen();
       insert_ring(parameterf.druck);
       printf("%d %d %s",parameterf.druck,parameterf.temp,ctime(&now));
       fflush(stdout);
       pthread_mutex_unlock(&fastmutex);
       sleep(1);
       while(difftime((now=time(NULL)),t)<10)usleep(100*1000);
       t=now;
   }
   return( NULL );
}


// including socket

void *pthr_sendpressure( void *pV_data )
{
    int i;
    static int count=0;
    char *buf;
    char *bp;
    // socket structure
    struct sockaddr_in  client1, server1;
    struct servent *serviceInfo;
    struct hostent *hostinfo;
    int s1,s;
    socklen_t length;
    // socket server
    if(serviceInfo = getservbyname("socktest","tcp")){
	server1.sin_port = serviceInfo->s_port;
    }
    server1.sin_family = AF_INET;
    server1.sin_addr.s_addr = INADDR_ANY;
    s = socket(AF_INET,SOCK_STREAM,0);
    if(s<0){
	fprintf(stderr,"Error in socket\n");
	exit(1);
    }
    if(bind(s,(struct sockaddr*)&server1,sizeof(server1))<0){
	fprintf(stderr,"Error in bind\n");
	exit(2);
    }
    if(listen(s,5)<0){
	fprintf(stderr,"Error in listen\n");
	exit(3);
    }
    length = sizeof(client1);
    // transferbuffer
    if((buf=malloc(RINGLENGTH*5))==NULL){
	fprintf(stderr,"error (sendpressure): no space on device\n");
	exit(1);
    }
    fprintf(stderr,"Server ready: %d\n",s);
    while ( 1 == 1 ) {
	if((s1=accept(s,(struct sockaddr*)&client1,&length))<0){
	    fprintf(stderr,"Error in accept: %d\n",s1);
	    exit(3);
	}
	pthread_mutex_lock(&fastmutex);
	bp=buf;
	count = first;  // first stands of i+1
	if(count<0)count=RINGLENGTH-1;
	for(i=0; i<RINGLENGTH; i++){
	    if(count+i>=RINGLENGTH)count=-i;
	    sprintf(bp,"%5d",pressure[count+i]);
	    bp+=5;
	}
	fprintf(stderr,"Connected to client\n");
	write(s1,buf,5*RINGLENGTH);
	pthread_mutex_unlock(&fastmutex);
    }
    return( NULL );
}



int main(int argc, char **argv)
{
    char *pC_port;
    int druck, temp;
    int count=0;
    int i;
    if ( argc != 2 ) {
	printf( "\nusage: %s <LPT1|LPT2>\n\n", argv[0] );
	exit( 1 );
    }
    pC_port    = argv[1];
    if ( strncmp( pC_port, "LPT2", 4 ) == 0 ) {
	m_iPort = D_LPT2_DATA;
    } else {
	m_iPort = D_LPT1_DATA;
    }
    //### Abschnitt 3
    if  ( ioperm( m_iPort, 3, 1) != 0 ) { /* get 10 perm */
	perror( "ERROR: can_t open LPT device for read/write" );
	return(-1);
    }
    //### Abschnitt 4
    printf( "\nPress ^C to stop program.\n");
    init_ring();
    ParameterLesen();
    if ( pthread_create(&pthr_get, NULL, pthr_getpressure, (void *)0) != 0) {
	printf( "Create of task 1 failed");
	exit( -1 );
    }
    if ( pthread_create(&pthr_send, NULL, pthr_sendpressure,(void *)1) != 0) {
	printf( "Create of task 2 failed");
	exit( -1 );
    }
    
    while ( 1 == 1 ) {
	fprintf(stderr, "I'm alive ...\n" ); fflush( stderr );
	sleep( 3600 );
    }
    return( 0 );

}








