#include "calcfloat.h"
#include "progmem.h"
#include "hpglobals.h"


int KeyType::keycount;
KeyType **KeyType::sortedkeys;



KeyType::KeyType(char const fn_keyname[],
		 int hasanarg,
		 int fn_immediate,
		 int fn_program,
		 int (* const fn_parse_arg)(char *rest, char **arg, char **comment),
		 void (* const fn_print_line)(char *buffer, int blen,
					      char const *arg,
					      char const *comment),
		 sp_acts (* const fn_action)(CalcProgMem *prog))
{
  keycount++;
  keycap.keyname = strdup(fn_keyname);
  keycap.can_have_arg = (hasanarg != 0);
  keycap.immediate = fn_immediate;
  keycap.program = fn_program;

  new_parse_arg = fn_parse_arg;
  new_print_line = fn_print_line;
  new_action = fn_action;
}



void KeyType::print_line(char *buffer, int blen, char const *arg,
			 char const *comment) const
{
  if (this->new_print_line != NULL) {
    (*this->new_print_line)(buffer, blen, arg, comment);
    return;
  }

  if (arg == NULL)
    snprintf(buffer, blen, "%s", this->keycap.keyname);
  else 
    snprintf(buffer, blen, "%-s %-s", this->keycap.keyname, arg);

  buffer[blen-1] = 0;

  if (arg != NULL &&
      comment != NULL && 
      comment[0] != 0 &&
      (int) strlen(buffer) < blen - 2) {
    strcat(buffer, "   ");
    append_char(buffer, COMMENT);
    strncat(buffer, comment, blen - 2 - strlen(buffer));
  }
}


sp_acts KeyType::action(CalcProgMem *prog) const
{
  if (this->new_action != NULL) {
    return (*this->new_action)(prog);
  }

  /* No default action */
  return NONE;
}




void CalcProgMem::copycontents(CalcProgMem *dest, CalcProgMem const &src) const
{

  dest->validcell = src.validcell;

  if (dest->value != NULL) delete dest->value;

  if (src.value != NULL) {
    dest->value = new CalcFloat(*src.value);
  } else {
    dest->value = NULL;
  }

  if (src.argument != NULL) {
    if (dest->argument != NULL)
      free(dest->argument);
    dest->argument = (char *) malloc(strlen(src.argument) + strlen(src.comment) + 2);
    CHKMEM(dest->argument);
    strcpy(dest->argument, src.argument);
    strcpy(dest->argument + strlen(src.argument) + 1, src.comment);  
  } else {
    dest->argument = NULL;
  }

  dest->myspace = src.myspace;
  dest->function = src.function;

  if (src.comment != NULL)
    dest->comment = dest->argument + (src.comment - src.argument);

  
  dest->gototarget = NULL;   // Don't count on the address being good
  dest->prevmem = src.prevmem;
  dest->nextmem = src.nextmem;
}


int CalcProgMem::perform_goto(void)
{
  int indirect = 0;

  if (argument[0] == '.') { // Absolute address
    gototarget = (CalcProgMem *) program.findlinenum(gotolinenum = atoi(argument+1));
    if (gototarget == NULL) return -1;
    return 0;
  }

  if (myspace == NULL) {
    gototarget = (CalcProgMem *) program.findlabel(argument, &indirect);
    if (gototarget == NULL) return -1;
    if (indirect) {
      gototarget = NULL;
    } else {
      gotolinenum = program.currentnum;
      if (runmode == RUN_PROG ||
	  runmode == STEPPING)
	program.must_invalidate_gotos = TRUE;
    }
    return 0;

  } else {

    if (gototarget == NULL) {
      gototarget = (CalcProgMem *) myspace->findlabel(argument, &indirect);
      if (gototarget == NULL) return -1;
      if (indirect) {
	gototarget = NULL;
      } else {
	gotolinenum = program.currentnum;
	if (runmode == RUN_PROG ||
	    runmode == STEPPING)
	  program.must_invalidate_gotos = TRUE;
      }
    } else {
      myspace->currentcell = gototarget;
      myspace->currentnum = gotolinenum;
    }
    
    return 0;
  }
}




// The ProgramSpace functions

CalcProgMem const *ProgramSpace::findlinenum(int linenum)
{
  if (linenum > (int) numcells + 1 || linenum < 0) return NULL;
  if (linenum < abs((int) (currentnum - linenum))) {   // Close to beginning
    currentcell = firstmem;
    currentnum = 0;
    *this += linenum;
  } else if ((int) numcells + 1 - linenum < abs((int) (currentnum - linenum))) { // Close to end
    currentcell = lastmem;
    currentnum = numcells + 1;
    *this -= numcells + 1 - linenum;
  } else {   // Close to current position
    *this += linenum - currentnum;
  }
  return currentcell;  
}



CalcProgMem const *ProgramSpace::findlabel(char const *label, int *indirect)
{
  CalcProgMem const *stop;
  int success;

  if (numcells == 0) {
    currentcell = firstmem;
    return NULL;  // No program, so the label's not there
  }

  if (strcmp(label, INDIRECT) == 0) {  /* An indirect subroutine call */
    int ival = I_REGISTER.int_value();

    *indirect = 1;
    if (ival < 0) {   // Negative number, back up that many steps
      *this -= -ival;
      return currentcell;
    } else {          // Positive number, find the appropriate label
      intstring sbuf;  // Sufficient to hold any integer
      
      sprintf(sbuf, "%d", ival);
      return findlabel(sbuf, indirect);    // Recurse in, doing the indirection
    }
  }

  stop = currentcell;
  success = 0;
  
  do {
    if (currentcell->argument != NULL &&  // A quick check, short-circuit
	*currentcell->argument != 0 &&    // the expensive functions
	strcmp(currentcell->function->keycap.keyname, LABELSTR) == 0 &&
	strcmp(currentcell->argument, label) == 0) { /* Found the label */
      success = 1;
      break;
    }
  
    ++(*this);

  } while (currentcell != stop);

  if (success) return currentcell;  // Found it

  return NULL;   // No such label
}



void ProgramSpace::push_retaddr(void)
{
  ReturnAddr *newframe = new ReturnAddr;
  newframe->retaddr = currentcell;
  newframe->retstepnum = currentnum;
  newframe->nextframe = frame0;
  frame0 = newframe;
}


/* Return from last gosub. If there are no more return frames, send back -1 */
int ProgramSpace::pop_retaddr(void)
{
  ReturnAddr *oldframe;

  if (frame0 == NULL) return -1;

  currentcell = frame0->retaddr;
  currentnum = frame0->retstepnum;
  oldframe = frame0;
  frame0 = frame0->nextframe;
  delete oldframe;
  return 0;
}


void ProgramSpace::erase_program(void)
{
  currentcell = lastmem->prevmem;
  currentnum = numcells;

  while (numcells > 0)
    delete_current_cell();
  clear_return_stack();
}


void ProgramSpace::clear_return_stack(void)
{
  ReturnAddr *holdf;

  while (frame0 != NULL) {
    holdf = frame0;
    frame0 = frame0->nextframe;
    delete holdf;
  }
}



void ProgramSpace::invalidate_goto_ptrs(void)
{
  CalcProgMem *pptr = firstmem->nextmem;

  while (pptr != &barrierhigh) {
    pptr->gototarget = NULL;
    pptr = pptr->nextmem;
  }
  must_invalidate_gotos = FALSE;
}


void ProgramSpace::writeoutput(FILE *ostream)
{
  CalcProgMem *pptr = firstmem->nextmem;
  char buffer[1000];

  while (pptr != lastmem) {
    pptr->printcontents(buffer, 1000);
    fputs(buffer, ostream);
    fprintf(ostream, "\n");
    pptr = pptr->nextmem;
  }
}
