/* A template to produce a stack class */

#ifndef STACK_H
#define STACK_H

#include <stdio.h>
#include <stdlib.h>

#define ELEMSPERBLOCK 20


#ifndef CHKMEM
#define CHKMEM(x) do { if ((x) == NULL) { fprintf(stderr, "Memory allocation failure.\n"); abort(); } } while (0)
#endif  /* !CHKMEM */



/* I make an array which grows transparently to the user as the [] 
 * operations are used */
template <class DTYPE>
class GrowingArray {

private: 
  DTYPE **blocks;  // vector of pointers to blocks of stack elements
  int numblocks;   // number of blocks malloc-ed.

private:
  void get_new_block(void);
  void make_first_block(void)
    {
      blocks = (DTYPE **) malloc(sizeof(DTYPE *));
      CHKMEM(blocks);
      blocks[0] = new DTYPE [ELEMSPERBLOCK];
      numblocks = 1;
    }
  void destroy_all_blocks(void)
    {
      for (int i = 0; i < numblocks; i++)
	delete [] blocks[i];
      
      free(blocks);
    }

public:
  GrowingArray(void);
  ~GrowingArray(void);

  DTYPE &operator[] (int index);
  void erase_space(void);
};


template <class DTYPE>
GrowingArray<DTYPE>::GrowingArray(void)
{
  make_first_block();
}

template <class DTYPE>
GrowingArray<DTYPE>::~GrowingArray(void)
{
  destroy_all_blocks();
}


template <class DTYPE>
void GrowingArray<DTYPE>::erase_space(void)
{
  destroy_all_blocks();
  make_first_block();
}


/* If somebody is referencing an element which is not yet allocated, 
 * make new blocks until we have enough. */
template <class DTYPE>
DTYPE &GrowingArray<DTYPE>::operator[] (int index)
{
  while (index >= ELEMSPERBLOCK * numblocks)
    get_new_block();

  return blocks[index / ELEMSPERBLOCK][index % ELEMSPERBLOCK];
}


template <class DTYPE>
void GrowingArray<DTYPE>::get_new_block(void)
{
  numblocks++;
  blocks = (DTYPE **) realloc(blocks, numblocks * sizeof(DTYPE *));
  CHKMEM(blocks);
  blocks[numblocks-1] = new DTYPE [ELEMSPERBLOCK];
}

#undef ELEMSPERBLOCK



/* A pure method used by the IO methods.  Might have to be expanded
 * later, or another pure method inserted between VStack and
 * Stack<DTYPE>. */
class VStack
{
public:
  virtual Arithdatatype const &reveal_value(unsigned int depth) = 0;
  virtual unsigned int get_size(void) const = 0;
  virtual ~VStack(void) {};
};



/* The stack template */
template <class DTYPE>
class Stack : public VStack {

public:
  DTYPE lastx;  // last X value
  DTYPE const *elem_x, *elem_y;  // X and Y registers, for simplicity of notation. Can't be used to change the values, only to read them

private:
  GrowingArray<DTYPE> holdspace;  // the actual elements are held here

  unsigned int index;  /* Position within array of next push */

/* some IO methods may find the depthchanged and stackops values
 * useful, as they can be used to shortcut redrawing. 'stackops'
 * positive means that a net push has occured, while negative means
 * that a net pop has occured. 'depthchanged' holds the number of
 * elements which were at the top of the stack before which are not
 * now present on the stack. So, the technique to using these values
 * efficiently is:

  draw the stack
  reset the depthchanged and stackops variables
  do whatever has to be done, calculate, run programs, and so on, without
    updating the stack display
  when time comes to redraw the stack, do it by:
    scrolling the entire stack up "stackops" lines
    redrawing the bottom "depthchanged - (-stackops)" elements

  A push then a pop results in no action required to redraw.
  A pop then a push results in the requirement simply to redraw the X element
*/
  int depthchanged;   
  int stackops;

/* popping when there's nothing there returns this */
  DTYPE const zeroval;

public:
  Stack(void);
  ~Stack(void) {};

  DTYPE pop(void);
  void push(Arithdatatype &);
  void reset_unbalanced_count(void);
  int get_unbalanced_count(void) const;  
  int get_change_depth(void) const;
  unsigned int get_size(void) const;
  DTYPE const &reveal_value(unsigned int depth);
  inline void exchange_xy(void);
  void roll_down(void);
  void roll_up(void);
  void erase_stack(void) {
    index = 0;
    reset_unbalanced_count();
    holdspace.erase_space();
  }

private:
  void set_elems_xy(void);
};


template <class DTYPE>
Stack<DTYPE>::Stack(void) 
{
  index = 0;
  stackops = depthchanged = 0;
  elem_x = elem_y = &zeroval;
}


template <class DTYPE>
void Stack<DTYPE>::push(Arithdatatype &data)
{
  holdspace[index] = (DTYPE &)data;
  index++;
  stackops++;
  set_elems_xy();
}


template <class DTYPE>
DTYPE Stack<DTYPE>::pop(void)
{
  stackops--;
  if (-stackops > depthchanged)
    depthchanged = -stackops;

  if (index == 0)
    return zeroval;

  index--;
  set_elems_xy();
  return holdspace[index];
}


template <class DTYPE>
void Stack<DTYPE>::reset_unbalanced_count(void)
{
  stackops = depthchanged = 0;
}


template <class DTYPE>
int Stack<DTYPE>::get_unbalanced_count(void) const
{
  return stackops;
}


template <class DTYPE>
int Stack<DTYPE>::get_change_depth(void) const
{
  return depthchanged;
}


template <class DTYPE>
unsigned int Stack<DTYPE>::get_size(void) const
{
  return index;
}


// peak into the stack without popping. Used by output methods which have to draw the stack.
template <class DTYPE>
DTYPE const &Stack<DTYPE>::reveal_value(unsigned int depth)
{
  if (depth >= get_size()) {
    if (get_size() == 0 && depth == 0)
      return zeroval;

    fprintf(stderr, "Error. Tried to reveal a value which isn't there.\n");
    abort();
  }

  return holdspace[((int) index) - depth - 1];
}


// a stack method for exchanging the top two values of the stack
template <class DTYPE>
inline void Stack<DTYPE>::exchange_xy(void)
{
  DTYPE holdx, holdy;
  
  holdx = pop();
  holdy = pop();
  push(holdx);
  push(holdy);
  set_elems_xy();
}


// an efficient roll of the stack. The last element pushed becomes the deepest
// pushed element, all others move one line closer to the top of the stack
template <class DTYPE>
void Stack<DTYPE>::roll_down(void)
{
  DTYPE holdx;
  unsigned int i;

  if (index == 0) return;
  holdx = holdspace[index - 1];
  for (i = index-1; i > 0; i--)
    holdspace[i] = holdspace[i-1];

  holdspace[0] = holdx;
  set_elems_xy();
}


// As above, but rolling the other way
template <class DTYPE>
void Stack<DTYPE>::roll_up(void)
{
  DTYPE holdtop;
  unsigned int i;

  if (index == 0) return;
  holdtop = holdspace[0];
  for (i = 0; i < index-1; i++)
    holdspace[i] = holdspace[i+1];

  holdspace[index-1] = holdtop;
  set_elems_xy();
}


// Set the pointers for X and Y elements.
template <class DTYPE>
inline void Stack<DTYPE>::set_elems_xy(void)
{
  switch(index) {
  case 0:
    elem_x = elem_y = &zeroval;
    break;
  case 1:
    elem_x = &holdspace[0];
    elem_y = &zeroval;
    break;
  default:
    elem_x = &holdspace[index-1];
    elem_y = &holdspace[index-2];
    break;
  }
}


#endif  /* !STACK_H */
