/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.VirtualNumaNodeBuilder;
import org.ovirt.engine.sdk4.types.VirtualNumaNode;

public interface VmNumaNodeService extends Service {
    public interface GetRequest extends Request<GetRequest, GetResponse> {
    }
    
    public interface GetResponse extends Response {
        VirtualNumaNode node();
    }
    
    GetRequest get();
    
    /**
     * Removes a virtual NUMA node.
     * 
     * An example of removing a virtual NUMA node:
     * 
     * [source]
     * ----
     * DELETE /ovirt-engine/api/vms/123/numanodes/456
     * ----
     */
    public interface RemoveRequest extends Request<RemoveRequest, RemoveResponse> {
        /**
         * Indicates if the remove should be performed asynchronously.
         */
        RemoveRequest async(Boolean async);
    }
    
    /**
     * Removes a virtual NUMA node.
     * 
     * An example of removing a virtual NUMA node:
     * 
     * [source]
     * ----
     * DELETE /ovirt-engine/api/vms/123/numanodes/456
     * ----
     */
    public interface RemoveResponse extends Response {
    }
    
    /**
     * Removes a virtual NUMA node.
     * 
     * An example of removing a virtual NUMA node:
     * 
     * [source]
     * ----
     * DELETE /ovirt-engine/api/vms/123/numanodes/456
     * ----
     */
    RemoveRequest remove();
    
    /**
     * Updates a virtual NUMA node.
     * 
     * An example of pinning a virtual NUMA node to a physical NUMA node on the host:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/vms/123/numanodes/456
     * ----
     * 
     * The request body should contain the following:
     * 
     * [source,xml]
     * ----
     * <vm_numa_node>
     *   <numa_node_pins>
     *     <numa_node_pin>
     *       <host_numa_node id="789"/>
     *       <index>0</index>
     *       <pinned>true</pinned>
     *     </numa_node_pin>
     *   </numa_node_pins>
     * </vm_numa_node>
     * ----
     */
    public interface UpdateRequest extends Request<UpdateRequest, UpdateResponse> {
        /**
         * Indicates if the update should be performed asynchronously.
         */
        UpdateRequest async(Boolean async);
        UpdateRequest node(VirtualNumaNode node);
        UpdateRequest node(VirtualNumaNodeBuilder node);
    }
    
    /**
     * Updates a virtual NUMA node.
     * 
     * An example of pinning a virtual NUMA node to a physical NUMA node on the host:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/vms/123/numanodes/456
     * ----
     * 
     * The request body should contain the following:
     * 
     * [source,xml]
     * ----
     * <vm_numa_node>
     *   <numa_node_pins>
     *     <numa_node_pin>
     *       <host_numa_node id="789"/>
     *       <index>0</index>
     *       <pinned>true</pinned>
     *     </numa_node_pin>
     *   </numa_node_pins>
     * </vm_numa_node>
     * ----
     */
    public interface UpdateResponse extends Response {
        VirtualNumaNode node();
    }
    
    /**
     * Updates a virtual NUMA node.
     * 
     * An example of pinning a virtual NUMA node to a physical NUMA node on the host:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/vms/123/numanodes/456
     * ----
     * 
     * The request body should contain the following:
     * 
     * [source,xml]
     * ----
     * <vm_numa_node>
     *   <numa_node_pins>
     *     <numa_node_pin>
     *       <host_numa_node id="789"/>
     *       <index>0</index>
     *       <pinned>true</pinned>
     *     </numa_node_pin>
     *   </numa_node_pins>
     * </vm_numa_node>
     * ----
     */
    UpdateRequest update();
    
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

