/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.internal.xml;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import org.ovirt.api.metamodel.runtime.util.ArrayListWithHref;
import org.ovirt.api.metamodel.runtime.util.ListWithHref;
import org.ovirt.api.metamodel.runtime.xml.XmlReader;
import org.ovirt.engine.sdk4.internal.containers.VmPoolContainer;
import org.ovirt.engine.sdk4.types.VmPool;

public class XmlVmPoolReader {
    
    public static VmPool readOne(XmlReader reader) {
        // Do nothing if there aren't more tags:
        if (!reader.forward()) {
            return null;
        }
        
        // Create the object:
        VmPoolContainer object = new VmPoolContainer();
        
        // Process the attributes:
        for (int i = 0; i < reader.getAttributeCount(); i++) {
            String name = reader.getAttributeLocalName(i);
            String image = reader.getAttributeValue(i);
            switch (name) {
                case "href":
                object.href(image);
                break;
                case "id":
                object.id(image);
                break;
                default:
                break;
            }
        }
        
        // Process the inner elements:
        reader.next();
        while (reader.forward()) {
            String name = reader.getLocalName();
            switch (name) {
                case "cluster":
                object.cluster(XmlClusterReader.readOne(reader));
                break;
                case "comment":
                object.comment(reader.readString());
                break;
                case "description":
                object.description(reader.readString());
                break;
                case "display":
                object.display(XmlDisplayReader.readOne(reader));
                break;
                case "instance_type":
                object.instanceType(XmlInstanceTypeReader.readOne(reader));
                break;
                case "max_user_vms":
                object.maxUserVms(reader.readInteger());
                break;
                case "name":
                object.name(reader.readString());
                break;
                case "permissions":
                object.permissions(XmlPermissionReader.readMany(reader));
                break;
                case "prestarted_vms":
                object.prestartedVms(reader.readInteger());
                break;
                case "rng_device":
                object.rngDevice(XmlRngDeviceReader.readOne(reader));
                break;
                case "size":
                object.size(reader.readInteger());
                break;
                case "soundcard_enabled":
                object.soundcardEnabled(reader.readBoolean());
                break;
                case "stateful":
                object.stateful(reader.readBoolean());
                break;
                case "template":
                object.template(XmlTemplateReader.readOne(reader));
                break;
                case "type":
                object.type(XmlVmPoolTypeReader.readOne(reader));
                break;
                case "use_latest_template_version":
                object.useLatestTemplateVersion(reader.readBoolean());
                break;
                case "vm":
                object.vm(XmlVmReader.readOne(reader));
                break;
                case "link":
                readLink(reader, object);
                break;
                default:
                reader.skip();
                break;
            }
        }
        
        // Discard the end tag:
        reader.next();
        
        return object;
    }
    
    public static Iterator<VmPool> iterateMany(final XmlReader reader) {
        return new Iterator<VmPool>() {
            private boolean first = true;
            
            @Override
            public void remove() {
                // Empty on purpose
            }
            
            @Override
            public boolean hasNext() {
                if (first) {
                    if (!reader.forward()) {
                        return false;
                    }
                    reader.next();
                    first = false;
                }
                if (!reader.forward()) {
                    reader.next();
                    return false;
                }
                return true;
            }
            
            @Override
            public VmPool next() {
                VmPool next = readOne(reader);
                if (next == null) {
                    throw new NoSuchElementException();
                }
                return next;
            }
        };
    }
    
    public static List<VmPool> readMany(XmlReader reader) {
        List<VmPool> list = new ArrayList<>();
        Iterator<VmPool> iterator = iterateMany(reader);
        while (iterator.hasNext()) {
            list.add(iterator.next());
        }
        return list;
    }
    
    private static void readLink(XmlReader reader, VmPoolContainer object) {
        // Process the attributes:
        String rel = null;
        String href = null;
        for (int i = 0; i < reader.getAttributeCount(); i++) {
            String attrName = reader.getAttributeLocalName(i);
            String attrVal = reader.getAttributeValue(i);
            switch (attrName) {
                case "href":
                href = attrVal;
                break;
                case "rel":
                rel = attrVal;
                break;
                default:
                reader.skip();
                break;
            }
        }
        if (href != null) {
            ListWithHref list = new ArrayListWithHref();
            list.href(href);
            switch (rel) {
                case "permissions":
                object.permissions(list);
                break;
                default:
                break;
            }
        }
        reader.skip();
    }
}

