/*	$NetBSD: hash.c,v 1.1.2.2 2023/08/01 17:03:53 martin Exp $	 */

/*
 * Copyright 1996 John D. Polstra.
 * Copyright 1996 Matt Thomas <matt@3am-software.com>
 * Copyright 2002 Charles M. Hannum <root@ihack.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by John Polstra.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Dynamic linker for ELF.
 *
 * John Polstra <jdp@polstra.com>.
 */

#include <sys/cdefs.h>
#ifndef lint
__RCSID("$NetBSD: hash.c,v 1.1.2.2 2023/08/01 17:03:53 martin Exp $");
#endif /* not lint */

#include <stdint.h>

#include "hash.h"

/*
 * SysV hash function for symbol table lookup.  It is a slightly optimized
 * version of the hash specified by the System V ABI.
 */
Elf32_Word
_rtld_sysv_hash(const char *name)
{
	const unsigned char *p = (const unsigned char *) name;
	Elf32_Word h = 0;

	while (__predict_true(*p != '\0')) {
		h = (h << 4) + *p++;
		h ^= (h >> 24) & 0xf0;
	}
	return (h & 0x0fffffff);
}

/*
 * Hash function for symbol table lookup.  Don't even think about changing
 * this.  It is specified by the GNU toolchain ABI.
 */
Elf32_Word
_rtld_gnu_hash(const char *name)
{
	const unsigned char *p = (const unsigned char *) name;
	uint_fast32_t h = 5381;
	unsigned char c;

	for (c = *p; c != '\0'; c = *++p)
		h = h * 33 + c;
	return (h & 0xffffffff);
}
