#if !defined(lint) && !defined(__INSIGHT__)
static char sos__rcsid[] = "$Id$";
static char sos__copyright[] = "Copyright (c) 1994, 1995, 1996 SOS Corporation";
static char sos__contact[] = "SOS Corporation <sos-info@soscorp.com> +1 800 SOS UNIX";
#endif /* not lint */

/*
 * ++Copyright Released Product++
 *
 * Copyright (c) 1994, 1995, 1996 Sources of Supply Corporation ("SOS").
 * All rights reserved.
 *
 * The SOS Released Product License Agreement specifies the terms and
 * conditions for redistribution.  You may find the License Agreement
 * in the file LICENSE.
 *
 * SOS Corporation
 * 461 5th Ave.; 16th floor
 * New York, NY 10017
 *
 * +1 800 SOS UNIX
 * <sos-info@soscorp.com>
 *
 * --Copyright Released Product--
 */

#include "sos.h"
#include "md5.h"

/* 
 * Debug levels
 * 8 Protocol Tracing
 * 32 NBIO test (will force a return at state changes).
 * 64 General Info
 * 128 Extra Info
 */


/* 
 * Randomly determine an identity. Quite useful when you want to use a
 * cryptographic protocols which are not inherently symetric, but which would
 * you would like to make so.  Most protocols have two peers, so this 
 * routine simply generates a random number, sends it, reads the peer's,
 * compares the two, and assigns an identity according to the ordering of the
 * numbers. Now obviously the peer can cheat the routine with ease, but 
 * the assumption is that you know what your doing and cheating (ie allowing
 * you peer to choose ALICE or BOB) will not by her/him anything.
 *
 * input: readpeer, writepeer: file descriptors open on peer connection
 * 	  rfun, wfun: read and write functions to use on connection.
 *
 * output: SOS_IDENT_ALICE, SOS_IDENT_BOB, or -1
 */
int
sos_alice_bob(int readpeer,
	      int (*rfun) (int, caddr_t,  __SIZE_TYPE__ ),
	      int writepeer,
	      int (*wfun) (int, caddr_t, __SIZE_TYPE__ ) )
{
  SOS_ENTRY("sos_alice_bob","sos_alice_bob","(%d,%x,%d,%x)",readpeer,rfun,writepeer,wfun);
  long  localidentrand;		/* Random number used for identity  */
  long  peeridentrand;		/* Random number used for identity  */

  /* Exchange numbers until the numbers are different */
  do
    {
      if (sos_get_rand((unsigned char *)&localidentrand, sizeof(localidentrand)) < 1)
	{
	  sos_error_printf("random number generation failed\n");
	  SOS_RETURN(-1);
	}
      
      if ( sos_xdr_wencode (writepeer, wfun, "l",  localidentrand) <=0 )
	{
	  sos_error_printf("XDR wire encode failed\n");
	  SOS_RETURN(-1);
	}
      
      if ( sos_xdr_wdecode (readpeer, rfun, "l", &peeridentrand) != 1 )
	{
	  sos_error_printf("XDR wire decode failed\n");
	  SOS_RETURN(-1);
	}
    } while (localidentrand == peeridentrand );

  /* Determine Alice/Bob Roles */
  if ( localidentrand < peeridentrand )
    SOS_RETURN(SOS_IDENT_ALICE);
  else 
    SOS_RETURN(SOS_IDENT_BOB);
}


/*
 * NBIO states
 */
#define SOS_NBIO_ALICE_BOB_START 0
#define SOS_NBIO_ALICE_BOB_WRITE 1
#define SOS_NBIO_ALICE_BOB_READ 2

/*
 * Same as above but supports nbio rfun and wfun functions
 * Accepts an sos_alice_bob_h handle and jumps back into the appropriate
 * place in the protocol. For this function there is only 1 place we can block
 * (sos_xdr_decode), so this handle is pretty darn simple
 * Returns same values as above as well as either SOS_NBIO_ALICE_BOB_PROGRESS
 * or  SOS_NBIO_ALICE_BOB_NO_PROGRESS (which indicate if the protocol is
 * moving forward or not. Some callers might want to time out if enough
 * NO_PROGRESS's are returned).
 * We assume that you are using nbio routines with interfaces consistant with
 * the sos_nbio package. Sorry for any inconvenence.
 */
int
sos_nbio_alice_bob(sos_nbio_handle *readpeer,
		   int (*rfun) (sos_nbio_handle *, sos_string *),
		   sos_nbio_handle *writepeer,
		   int (*wfun) (sos_nbio_handle *, sos_string *, int, fd_set *, u_int *),
		   fd_set *fds,
		   u_int *byteswritten,
		   sos_alice_bob_handle *abhp)
{
  SOS_ENTRY("sos_alice_bob","sos_nbio_alice_bob", NULL);
  long  localidentrand;		/* Random number used for identity  */
  long  peeridentrand;		/* Random number used for identity  */
  sos_alice_bob_handle abh;
  int ret;
  sos_string lbuffer;		/* Read in from peer */

  lbuffer.str=NULL;
  lbuffer.len=0;

  if ( !(*abhp) )
    {
      sos_debug_printf_and(128, "Allocating new protocol handle\n");
      if ( (abh=(sos_alice_bob_handle)malloc(sizeof(abh))) == NULL )
	{
	  sos_error_printf("Allocating new alice_bob handle: %s", strerror(errno));
	  SOS_RETURN(-1);
	}

      abh->state=SOS_NBIO_ALICE_BOB_START;		/* Pendantic... */
      abh->buffer=NULL;
      *abhp=abh;
    }
  else
    {
      abh=*abhp;
      sos_debug_printf_and(64, "Reentering (client handle is non-null)\n");
    }

  

  switch (abh->state)
    {
    case SOS_NBIO_ALICE_BOB_START:
      goto sos_nbio_alice_bob_start;
      break; 

    case SOS_NBIO_ALICE_BOB_WRITE:
      sos_debug_printf_and(64, "Cutting straight down to write\n");
      goto sos_nbio_alice_bob_write;
      break;

    case SOS_NBIO_ALICE_BOB_READ:
      sos_debug_printf_and(64, "Cutting straight down to read\n");
      goto sos_nbio_alice_bob_read;
      break;

    default:
      sos_error_printf("Client handle indicates unknown state: %d", abh->state);
      sos_debug_printf_and(128, "Deallocating protocol handle\n");
      free(abh); *abhp=NULL;
      SOS_RETURN(-1);
      break;
    }

  /* Exchange numbers until the numbers are different */
 sos_nbio_alice_bob_start:
  abh->state=SOS_NBIO_ALICE_BOB_START;		/* Pendantic... */
  sos_debug_printf_and(8,"Alice/Bob protocol START\n");

  if (sos_get_rand((unsigned char *)&localidentrand, sizeof(localidentrand)) < 1)
    {
      sos_error_printf("random number generation failed\n");
      sos_debug_printf_and(128, "Deallocating protocol handle\n");
      free(abh); *abhp=NULL;
      SOS_RETURN(-1);
    }
      
  if ( (abh->buffer = sos_xdr_sencode("l", localidentrand)) == NULL )
    {
      sos_error_printf("XDR encoding failed\n");
      sos_debug_printf_and(128, "Deallocating protocol handle\n");
      free(abh); *abhp=NULL;
      SOS_RETURN(-1);
    }
      
  abh->state =  SOS_NBIO_ALICE_BOB_WRITE;
 sos_nbio_alice_bob_write:
  sos_debug_printf_and(8,"Alice/Bob protocol WRITE\n");

  /* 
   * Don't worry about copying the return from sos_xdr_sencode (staticspace).
   * nbio will copying. Outside folks are not permitted to modify this 
   */
  ret = (*wfun)(writepeer, abh->buffer, 0, fds, byteswritten);

  switch (ret)
    {
    
    case SOS_NBIO_ADMIN_ERROR:
      sos_debug_printf_and(8, "NBIO write returned ADMIN ERROR\n");
    case SOS_NBIO_IOFUN_ERROR:
      sos_debug_printf_and(8, "NBIO write returned IOFUN ERROR\n");
      sos_error_printf("NBIO write failed\n");
      sos_debug_printf_and(128, "Deallocating protocol handle\n");
      free(abh); *abhp=NULL;
      SOS_RETURN(-1);
      break;

    case SOS_NBIO_EOF: 
      sos_debug_printf_and(8, "NBIO write returned EOF\n");
      sos_error_printf("NBIO write produced EOF\n");
      /*
       * Just return -1. Caller will either be fooled into attempting 
       * to read again or may just close the channel. In the first case 
       * she will generate EOF and figure out what's going on. In the 
       * second she has just completed the close handshake.
       */
      sos_debug_printf_and(128, "Deallocating protocol handle\n");
      free(abh); *abhp=NULL;
      SOS_RETURN(-1);
      break;
      
    case SOS_NBIO_PROGRESS:
      sos_debug_printf_and(8, "NBIO write returned PROGRESS\n");
      return (SOS_NBIO_ALICE_BOB_PROGRESS);
      break;

    case SOS_NBIO_NOPROGRESS:
      sos_debug_printf_and(8, "NBIO write returned NO PROGRESS\n");
      return (SOS_NBIO_ALICE_BOB_NOPROGRESS);
      break;

    case SOS_NBIO_BUFCOMPLETE: 
      sos_debug_printf_and(8, "NBIO write returned BUFCOMPLETE\n");
      break;
      
    default:
      sos_debug_printf_and(8, "NBIO write returned UNKNOWN: %d\n", ret);
      sos_error_printf("Unexpected return value from nbio read\n");
      sos_debug_printf_and(128, "Deallocating protocol handle\n");
      free(abh); *abhp=NULL;
      SOS_RETURN (-1);
      break;
    }

  /* Update our position in the state engine */
  abh->state=SOS_NBIO_ALICE_BOB_READ;	

  if ( sos_debug_and (32) )
    {
      if  ( sos_debug_and(8) || sos_debug_and(64) )
	{
	  sos_debug_printf("Faking a return at the state change\n");
	}
      SOS_RETURN(SOS_NBIO_ALICE_BOB_NOPROGRESS);
    }

sos_nbio_alice_bob_read:  
  sos_debug_printf_and(8,"Alice/Bob protocol READ\n");

  ret = sos_nbio_read(readpeer, &lbuffer);
  switch (ret)
    {
    case SOS_NBIO_ADMIN_ERROR:
      sos_debug_printf_and(8, "NBIO read returned ADMIN ERROR\n");
    case SOS_NBIO_IOFUN_ERROR:
      sos_debug_printf_and(8, "NBIO read returned IOFUN ERROR\n");
      /* 
       * Since this is a hard failure, any malloced space should already
       * be cleaned up
       */
      sos_error_printf("NBIO read failed\n");
      sos_debug_printf_and(128, "Deallocating protocol handle\n");
      free(abh); *abhp=NULL;
      SOS_RETURN(-1);
      break;
      
    case SOS_NBIO_EOF: 
      sos_debug_printf_and(8, "NBIO read returned EOF\n");
      sos_error_printf("NBIO read produced EOF\n");
      /*
       * Just return -1. Caller will either be fooled into attempting 
       * to read again or may just close the channel. In the first case 
       * she will generate EOF and figure out what's going on. In the 
       * second she has just completed the close handshake.
       */
      sos_debug_printf_and(128, "Deallocating protocol handle\n");
      free(abh); *abhp=NULL;
      SOS_RETURN(-1);
      break;
      
    case SOS_NBIO_PROGRESS:
      sos_debug_printf_and(8, "NBIO read returned PROGRESS\n");
      return (SOS_NBIO_ALICE_BOB_PROGRESS);
      break;

    case SOS_NBIO_NOPROGRESS:
      sos_debug_printf_and(8, "NBIO read returned NO PROGRESS\n");
      return (SOS_NBIO_ALICE_BOB_NOPROGRESS);
      break;

    case SOS_NBIO_BUFCOMPLETE: 
      sos_debug_printf_and(8, "NBIO read returned BUFCOMPLETE\n");
      break;
      
    default:
      sos_debug_printf_and(8, "NBIO read returned UNKNOWN: %d\n", ret);
      sos_error_printf("Unexpected return value from nbio read\n");
      sos_debug_printf_and(128, "Deallocating protocol handle\n");
      free(abh); *abhp=NULL;
      SOS_RETURN (-1);
      break;
    }
  
  
  
  /* We've got sdecode this */
  if ( sos_xdr_sdecode (lbuffer.str, "l", &peeridentrand) != 1 )
    {
      free (lbuffer.str);
/*      sos_freestr(lbuffer);*/
      sos_error_printf("XDR buffer decode failed\n");
      sos_debug_printf_and(128, "Deallocating protocol handle\n");
      free(abh); *abhp=NULL;
      SOS_RETURN(-1);
    }
  
  if (localidentrand == peeridentrand ) 
    {
      sos_debug_printf_and(8,"Alice/Bob protocol: restarting\n");
      goto sos_nbio_alice_bob_start;
    }

  sos_debug_printf_and(128, "Deallocating protocol handle\n");
  free(abh); *abhp=NULL;

  /* Determine Alice/Bob Roles */
  if ( localidentrand < peeridentrand )
    SOS_RETURN(SOS_IDENT_ALICE);
  else 
    SOS_RETURN(SOS_IDENT_BOB);
}
