#if !defined(lint) && !defined(__INSIGHT__)
static char sos__rcsid[] = "$Id$";
static char sos__copyright[] = "Copyright (c) 1994, 1995, 1996 SOS Corporation";
static char sos__contact[] = "SOS Corporation <sos-info@soscorp.com> +1 800 SOS UNIX";
#endif /* not lint */

/*
 * ++Copyright Released Product++
 *
 * Copyright (c) 1994, 1995, 1996 Sources of Supply Corporation ("SOS").
 * All rights reserved.
 *
 * The SOS Released Product License Agreement specifies the terms and
 * conditions for redistribution.  You may find the License Agreement
 * in the file LICENSE.
 *
 * SOS Corporation
 * 461 5th Ave.; 16th floor
 * New York, NY 10017
 *
 * +1 800 SOS UNIX
 * <sos-info@soscorp.com>
 *
 * --Copyright Released Product--
 */

/*
 * SOS debugging functions
 *
 * Routines to help manage debugging with large projects.  Have specific
 * debugging levels associated with given packages or functions which
 * do not effect other packages or functions.
 */

#include "sos.h"


#ifndef FAKE_DEBUGGING

#define SOS_FILEDEBUG 1
#define SOS_SYSLOGDEBUG 2
#define SOS_BUFSIZE 2048

struct debuginfo
{
  char *name;
  u_int level;
};

static ht_val phashfstr(char **a);
static ht_val hashfstr(char *a);
static int mystrcmp(char **a, char **b);
static int mykeycmp(char *a, char **b);
static int sos_debug_vprintf(char *fmt, va_list args);
static struct ht_args DebugSize = { 64, 1, phashfstr, hashfstr };


dict_h sos_debug_db = NULL;
int sos_debug_on = 0;		/* 1 file 2 syslog */
FILE *sos_debug_out = NULL;



/*
 * Find out the current debugging level.  If package/function NULL, use
 * current package/function from sos_fun_*()
 */
u_int sos_debug_query(char *package, char *function)
{
  struct debuginfo *found;

  if (!sos_debug_db || !sos_debug_on)
    return(0);

  if (function && (found = (struct debuginfo *)ht_search(sos_debug_db, function)))
    return(found->level);

  if (package && (found = (struct debuginfo *)ht_search(sos_debug_db, package)))
    return(found->level);

  return(0);
}



/*
 * Set debugging level for symbolic name to specified level.
 */
int sos_debug_set(char *name, u_int level)
{
  struct debuginfo *new = malloc(sizeof(struct debuginfo));
  struct debuginfo *orig;

  if (!sos_debug_db)
    sos_debug_db = ht_create(mystrcmp, mykeycmp, DICT_UNIQUE_KEYS, NULL, &DebugSize);

  if (!name || !new || !sos_debug_db)
    {
      if (new) free(new);
#ifdef BAD_IDEA
      sos_error_printf("Can not malloc %d bytes\n",sizeof(struct debuginfo));
#endif /* BAD_IDEA */
      return(-1);
    }

  if (!(new->name = strdup(name)))
    {
      free(new);
#ifdef BAD_IDEA
      sos_error_printf("Can not malloc %d bytes\n",strlen(name));
#endif /* BAD_IDEA */
      return(-1);
    }
  new->level = level;

  /* Try to insert new debug record */
  if (ht_insert_uniq(sos_debug_db, (dict_obj)new, &orig) != DICT_OK)
    {
      if (orig && orig != new)
	{			/* Clash with previous entry */
	  orig->level = level;	/* Override current level */
	  free(new->name);
	  free(new);
	  return(1);
	}

#ifdef BAD_IDEA
      sos_error_printf("ht_insert_uniq insertion error inserting %s\n",name);
#endif /* BAD_IDEA */
      return(-1);
    }

  return(0);
}



/*
 * Read debugging names and level from a SOS config file
 * key is the index to grab them from the file
 */
int sos_debug_set_config(sos_config_t config, char *key)
{
  int tmp, ret = 0;
  sos_config_value value;
  char *name, *clevel;
  u_int level;
  char *line;

  while ((value = (char *)sos_config_getnext(config, key, SOS_CONFIG_FORWARD, SOS_CONFIG_LINEAR)) != NULL)
    {
      line = value;
      line += strspn(line,SOS_WHITESPACE);

      /* Working storage so we don't mess up config file */
      if (!(line = strdup(line)))
	{
#ifdef BAD_IDEA
	  sos_error_printf("Can not malloc %d bytes\n",strlen(value));
#endif /* BAD_IDEA */
	  return(-1);
	}

      /* Parse line */
      name = strtok(line, SOS_WHITESPACE);
      clevel = strtok(NULL, SOS_WHITESPACE);

      if (!name || !clevel)
	{
#ifdef BAD_IDEA
	  sos_error_printf("Syntax error on %s\n",level);
#endif /* BAD_IDEA */
	  free(line);
	  return(-1);
	}

      level = atoi(clevel);	/* strtol instead for error checking?? */

      /* Set debug level */
      tmp = sos_debug_set(name, level);
      free(line);

      if (tmp < 0)
	return(tmp);

      if (tmp)
	ret |= ret;
    }

  return(ret);
}



/*
 * Initialization routine.  Enable debugging and specify
 * where such debugging will go (file or soslog).
 */
void sos_debug_info(FILE *debug_file, int want_soslog)
{
  if (debug_file)
    {
      sos_debug_on |= SOS_FILEDEBUG;
      sos_debug_out = debug_file;
    }
  else
    {
      sos_debug_on &= (~SOS_FILEDEBUG);
      sos_debug_out = NULL;
    }

  if (want_soslog)
    sos_debug_on |= SOS_SYSLOGDEBUG;
  else
    sos_debug_on &= (~SOS_SYSLOGDEBUG);

  return;
}



/*
 * Print the message to the debugging output device
 * if the supplied level is EQUAL to the current debugging level
 * (as set by sos_debug_query)
 */
int sos_debug_printf_eq(u_int level, char *fmt, ...)
{
  va_list args;
  int ret;

  if (!sos_debug_db || !sos_debug_on || !sos_cur_fun || !sos_cur_fun->debuglevel || !fmt)
    return(0);

  va_start(args, fmt);
  ret = sos_debug_vprintf_ceq(level, sos_cur_fun->debuglevel, fmt, args);
  va_end(args);

  return(ret);
}



/*
 *
 * Print the message to the debugging output device
 * if the supplied level is GREATER THAN to the current debugging level
 * (as set by sos_debug_query)
 */
int sos_debug_printf_gt(u_int level, char *fmt, ...)
{
  int ret;
  va_list args;

  if (!sos_debug_db || !sos_debug_on || !sos_cur_fun || !sos_cur_fun->debuglevel || !fmt)
    return(0);

  va_start(args, fmt);
  ret = sos_debug_vprintf_cgt(level, sos_cur_fun->debuglevel, fmt, args);
  va_end(args);

  return(ret);
}



/*
 * Print the message to the debugging output device
 * if (the supplied level & the current debugging level) is true
 * (as set by sos_debug_query)
 */
int sos_debug_printf_and(u_int level, char *fmt, ...)
{
  int ret;
  va_list args;

  if (!sos_debug_db || !sos_debug_on || !sos_cur_fun || !sos_cur_fun->debuglevel || !fmt)
    return(0);

  va_start(args, fmt);
  ret = sos_debug_vprintf_cand(level, sos_cur_fun->debuglevel, fmt, args);
  va_end(args);

  return(ret);
}



/*
 * Print the message to the debugging output device regardless of level.
 * Assume that caller had already tested the level or doesn't care.
 */
int sos_debug_printf(char *fmt, ...)
{
  va_list args;
  int ret;

  if (!sos_debug_db || !sos_debug_on || !sos_cur_fun || !fmt)
    return(0);

  va_start(args, fmt);
  ret = sos_debug_vprintf(fmt, args);
  va_end(args);

  return(ret);
}


/*
 * Print the message to the debugging output device
 * if the supplied level is EQUAL supplied test level
 */
int sos_debug_printf_ceq(u_int level, u_int test, char *fmt, ...)
{
  int ret;
  va_list args;

  if (!sos_debug_db || !sos_debug_on || !test || !fmt)
    return(0);

  va_start(args, fmt);
  ret = sos_debug_vprintf_ceq(level, test, fmt, args);
  va_end(args);

  return(ret);
}



/*
 *
 * Print the message to the debugging output device
 * if the supplied level is GREATER THAN supplied test level
 */
int sos_debug_printf_cgt(u_int level, u_int test, char *fmt, ...)
{
  int ret;
  va_list args;

  if (!sos_debug_db || !sos_debug_on || !test || !fmt)
    return(0);

  va_start(args, fmt);
  ret = sos_debug_vprintf_cgt(level, test, fmt, args);
  va_end(args);

  return(ret);
}



/*
 * Print the message to the debugging output device
 * if (the supplied level & the current debugging level) is true
 * (as set by sos_debug_query)
 */
int sos_debug_printf_cand(u_int level, u_int test, char *fmt, ...)
{
  int ret;
  va_list args;

  if (!sos_debug_db || !sos_debug_on || !test || !fmt)
    return(0);

  va_start(args, fmt);
  ret = sos_debug_vprintf_cand(level, test, fmt, args);
  va_end(args);

  return(ret);
}



/*
 * Print the message to the debugging output device
 * if the supplied level is EQUAL supplied test level
 */
int sos_debug_vprintf_ceq(u_int level, u_int test, char *fmt, va_list args)
{
  if (!sos_debug_db || !sos_debug_on || !test || !fmt)
    return(0);

  if (test != level)
    return(0);

  return(sos_debug_vprintf(fmt,args));
}



/*
 *
 * Print the message to the debugging output device
 * if the supplied level is GREATER THAN supplied test level
 */
int sos_debug_vprintf_cgt(u_int level, u_int test, char *fmt, va_list args)
{
  if (!sos_debug_db || !sos_debug_on || !test || !fmt)
    return(0);

  if (test <= level)
    return(0);

  return(sos_debug_vprintf(fmt,args));
}



/*
 * Print the message to the debugging output device
 * if (the supplied level & the current debugging level) is true
 * (as set by sos_debug_query)
 */
int sos_debug_vprintf_cand(u_int level, u_int test, char *fmt, va_list args)
{
  if (!sos_debug_db || !sos_debug_on || !test || !fmt)
    return(0);

  if ((test & level) != level)
    return(0);

  return(sos_debug_vprintf(fmt,args));
}



/*
 * Test for debug level
 */
int sos_debug_eq(u_int level)
{
  if (!sos_debug_db || !sos_debug_on || !sos_cur_fun || !sos_cur_fun->debuglevel)
    return(0);

  return(sos_cur_fun->debuglevel == level);
}



/*
 * Test for debug level
 */
int sos_debug_gt(u_int level)
{
  if (!sos_debug_db || !sos_debug_on || !sos_cur_fun || !sos_cur_fun->debuglevel)
    return(0);

  return(sos_cur_fun->debuglevel > level);
}



/*
 * Test for debug level
 */
int sos_debug_and(u_int level)
{
  if (!sos_debug_db || !sos_debug_on || !sos_cur_fun || !sos_cur_fun->debuglevel)
    return(0);

  return((level & sos_cur_fun->debuglevel) == level);
}



/*
 * Internal print routine (prepends last function name--better be right!!)
 */
static int sos_debug_vprintf(char *fmt, va_list args)
{
  char buf1[SOS_BUFSIZE];
  char buf2[SOS_BUFSIZE];
  int ret = 0;

  if (sos_cur_fun && sos_cur_fun->funname)
    {
      vsnprintf(buf1,SOS_BUFSIZE,fmt,args);
      snprintf(buf2,SOS_BUFSIZE,"%s: %s",sos_cur_fun->funname,buf1);
    }
  else
    {
      vsnprintf(buf2,SOS_BUFSIZE,fmt,args);
    }

  if (sos_debug_on & SOS_SYSLOGDEBUG)
    soslog(SOSLOG_DEBUG, SOSLOG_TYPE_CONDITION, 0, "%s", buf2);

  if (sos_debug_on & SOS_FILEDEBUG && sos_debug_out)
    {
      int blen = strlen(buf2);

      if ((blen > 0) && (buf2[blen-1] == '\n'))
	ret = fprintf(sos_debug_out, "%s", buf2);
      else
	ret = fprintf(sos_debug_out, "%s\n", buf2);
    }

  return(ret);
}



/*
 * Use Sedgewick simple hash function
 *
 * (for CLC sorting)
 */
static ht_val phashfstr(char **a) { return(hashfstr(*a)); }
static ht_val hashfstr(char *a)
{
  ht_val ret;
  const unsigned int M = (unsigned)2147486459U; /* Arbitrary large prime */
  int h;

  for(h = 0; *a; a++)
    h = ( h + *a ) % M;

  memcpy((void *)&ret,(void *)&h,sizeof(ret));

  return(ret);
}



/*
 * Compare two pointers to strings
 *
 * Function for CLC sorting
 */
static int
mystrcmp(char **a, char **b)
{
  return(strcmp(*a,*b));
}



/*
 * Compare a string with a pointer to a string
 *
 * Function for CLC sorting
 */
static int
mykeycmp(char *a, char **b)
{
  return(strcmp(a,*b));
}

#else /* FAKE_DEBUGGING */
inline
u_int sos_debug_query(char *package, char *function)
{
  return(0);
}
inline
int sos_debug_set(char *name, u_int level)
{
  return(0);
}
inline
int sos_debug_set_config(sos_config_t config, char *key)
{
  return(0);
}
inline
void sos_debug_info(FILE *debug_file, int want_soslog)
{
  return;
}
/*inline /* Cannot have varags with inline (sigh) */
int sos_debug_printf_eq(u_int level, char *fmt, ...)
{
  return(0);
}
/*inline /* Cannot have varags with inline (sigh) */
int sos_debug_printf_gt(u_int level, char *fmt, ...)
{
  return(0);
}
/*inline /* Cannot have varags with inline (sigh) */
int sos_debug_printf_and(u_int level, char *fmt, ...)
{
  return(0);
}
/*inline /* Cannot have varags with inline (sigh) */
int sos_debug_printf_ceq(u_int level, char *fmt, ...)
{
  return(0);
}
/*inline /* Cannot have varags with inline (sigh) */
int sos_debug_printf_cgt(u_int level, char *fmt, ...)
{
  return(0);
}
/*inline /* Cannot have varags with inline (sigh) */
int sos_debug_printf_cand(u_int level, char *fmt, ...)
{
  return(0);
}
inline
int sos_debug_vprintf_ceq(u_int level, u_int test, char *fmt, va_list args)
{
  return(0);
}
inline
int sos_debug_vprintf_cgt(u_int level, u_int test, char *fmt, va_list args)
{
  return(0);
}
inline
int sos_debug_vprintf_cand(u_int level, u_int test, char *fmt, va_list args)
{
  return(0);
}
inline
int sos_debug_eq(u_int level)
{
  return(0);
}
inline
int sos_debug_gt(u_int level)
{
  return(0);
}
inline
int sos_debug_and(u_int level)
{
  return(0);
}
#endif /* FAKE_DEBUGGING */
