#if !defined(lint) && !defined(__INSIGHT__)
static char sos__rcsid[] = "$Id$";
static char sos__copyright[] = "Copyright (c) 1994, 1995, 1996 SOS Corporation";
static char sos__contact[] = "SOS Corporation <sos-info@soscorp.com> +1 800 SOS UNIX";
#endif /* not lint */

/*
 * ++Copyright Released Product++
 *
 * Copyright (c) 1994, 1995, 1996 Sources of Supply Corporation ("SOS").
 * All rights reserved.
 *
 * The SOS Released Product License Agreement specifies the terms and
 * conditions for redistribution.  You may find the License Agreement
 * in the file LICENSE.
 *
 * SOS Corporation
 * 461 5th Ave.; 16th floor
 * New York, NY 10017
 *
 * +1 800 SOS UNIX
 * <sos-info@soscorp.com>
 *
 * --Copyright Released Product--
 */

/*
 * Generic event queue--events occur at or after their scheduled times
 * (granularity depends on granularity of caller)
 *
 * Debug levels:
 *
 * 1 - Trace scan of event queue for stuff to do
 * 2 - Trace stuff that should never happen
 */

#include "sos.h"

#include <pq.h>


struct seq_event
{
  time_t when;
  void (*event)(void *args);
  void *args;
};


static int seq_compare(struct seq_event *a, struct seq_event *b);



/*
 * Add an event to the event queue
 * return number of second to next event
 *
 * -1 on error
 */
int sos_event_enqueue(pq_h equeue, time_t when, void (*event)(void *args), void *args, void **handle)
{
  SOS_ENTRY("sos_eventq","sos_event_enqueue","(%x, %d, %x, %x, %x)\n",equeue,when,event,args,handle);
  int diff;
  struct seq_event *new = malloc(sizeof(struct seq_event));
  struct seq_event *head;

  if (!new || !equeue)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  new->when = when;
  new->event = event;
  new->args = args;

  if (pq_insert(equeue, new) < 0)
    {
      sos_error_printf("Could not insert into priority queue: %s\n",strerror(errno));
      SOS_RETURN(-1);
    }

  head = pq_head(equeue);
  if (!head)
    {
      sos_error_printf("Something I inserted into event queue disappeared!!\n");
      SOS_RETURN(-1);
    }

  if (handle)
    *handle = new;

  diff = head->when - time(NULL);
  if (diff < 0)
    diff = 0;

  SOS_RETURN(diff);
}



/*
 * Check to see if any events need to be fired
 * (fire all those up for consideration)
 * return number of second to next event
 *
 * -1 on error
 * 0 if no more events
 */
int sos_event_check(pq_h equeue)
{
  SOS_ENTRY("sos_eventq","sos_event_check",NULL);
  struct seq_event *head, *tmp;

  if (!equeue)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  while ((head = pq_head(equeue)))
    {
      int diff = head->when - time(NULL);

      if (diff > 0)
	SOS_RETURN(diff);

      sos_debug_printf_and(1,"@%d *(%x)(%x)\n",head->when,head->event,head->args);

      /* This had better be the same as head */
      if ((tmp = pq_extract_head(equeue)) != head)
	{
	  sos_debug_printf_and(2,"sos_event_check: head (%x) != extract (%x)\n",head,tmp);
	}

      (*head->event)(head->args);

      free(head);
    }

  SOS_RETURN(0);		/* No more events */
}



/*
 * Check to see if any events need to be fired
 * return number of second to next event
 *
 * -1 on error
 * 0 if no more events
 */
pq_h sos_event_createq(void)
{
  SOS_ENTRY("sos_eventq","sos_event_createq",NULL);
  SOS_RETURN(pq_create(seq_compare,PQ_RETURN_ERROR,NULL));
}



/*
 * Delete an event from the event queue
 *
 * Returns 0, -1
 */
int sos_event_dequeue(pq_h equeue, void *handle)
{
  SOS_ENTRY("sos_eventq","sos_event_dequeue",NULL);
  int ret;

  if (!equeue || !handle)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  if ((ret = pq_delete(equeue,handle)) == DICT_ERR)
    {
      sos_error_printf("Could not find/delete event\n");
      SOS_RETURN(-1);
    }

  free(handle);

  SOS_RETURN(ret);
}



/*
 * Destroy event queue
 */
int sos_event_destroy(pq_h equeue)
{
  SOS_ENTRY("sos_eventq","sos_event_destroy",NULL);
  struct sos_event *head;

  if (!equeue)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  while (head = pq_extract_head(equeue))
    {
      free(head);
    }

  pq_destroy(equeue);
  SOS_RETURN(0);
}



/*
 * Function for event queue comparisons
 */
static int seq_compare(struct seq_event *a, struct seq_event *b)
{
  /* We will not trace this because it is used by CLC and called very frequently */
  return(a->when < b->when);
}
