#if !defined(lint) && !defined(__INSIGHT__)
static char sos__rcsid[] = "$Id$";
static char sos__copyright[] = "Copyright (c) 1994, 1995, 1996 SOS Corporation";
static char sos__contact[] = "SOS Corporation <sos-info@soscorp.com> +1 800 SOS UNIX";
#endif /* not lint */

/*
 * ++Copyright Released Product++
 *
 * Copyright (c) 1994, 1995, 1996 Sources of Supply Corporation ("SOS").
 * All rights reserved.
 *
 * The SOS Released Product License Agreement specifies the terms and
 * conditions for redistribution.  You may find the License Agreement
 * in the file LICENSE.
 *
 * SOS Corporation
 * 461 5th Ave.; 16th floor
 * New York, NY 10017
 *
 * +1 800 SOS UNIX
 * <sos-info@soscorp.com>
 *
 * --Copyright Released Product--
 */

/*
 * Connect to authentication server
 */

#include "sos.h"

static void sos_Connection_timeout(void);

static struct sigaction old_action;	/* Caches the old sigvec.  */
static struct sigaction timeout = SIGACTIONDEFINE( (void *)sos_Connection_timeout, 0, _INTERRUPT_SYSCALLS_ );

static int TIMEDOUT = 0;

/* forward definitions */
int sos_amake_conn(char *hoststring, char *port, int timeout_val, int socopts,  int fdflags, int *isconn, FILE *verbose);
int sos_nmake_conn(char *hoststring, u_short port, int timeout_val, int socopts, FILE *verbose);
int sos_anmake_conn(struct in_addr *addr, u_short port, int socopts, int fdflags, int *isconn, FILE *verbose);

/*
 * Trivial timeout handler so I can catch SIGALRM without aborting
 * If connect really timesout, the system call will return interupted
 */
static void
sos_Connection_timeout(void)
{
  TIMEDOUT = 1;

  /*
   * Reset your old action vector to whatever it was
   */
  sigaction(SIGALRM, (struct sigaction *)&old_action,
	    (struct sigaction *)NULL);
  return;
}



/*
 * Attempt to establish a TCP connection to the port service on 
 * hoststring. Allow both numeric and name representations of ports.
 * Try to connect to all known addresses for the host.
 *
 * Return system call result. Will be socket of laundered connection if all
 * goes according to plan.
 */
int
sos_make_conn(char *hoststring, char *port, int timeout_val, int socopts, FILE *verbose)
{
  SOS_ENTRY("sos_make_conn","sos_make_conn",NULL);
  int inport;
  u_short nport;

  inport = sos_getsbyfoo(port);
  nport = inport;

  if (inport < 0)
    {
      sos_error_printf("Invalid port %s: %s\n",port,strerror(errno));
      SOS_RETURN(-1);
    }

  SOS_RETURN(sos_nmake_conn(hoststring, nport, timeout_val, socopts, verbose));
}


/* 
 * Asynchronous connection (wrapper)
 * Does DNS and port string -> number mapping. Now of course any caller who
 * is *really* doing DNS is asking for big trouble. It assumed that name 
 * lookups will be either quite local (hence fast) or dotted quad conversions
 * (which in the soslib world do *not* resolve as reverse lookups, but rather
 * are just straight conversions to a strict addr_in). Nevertheless blocks
 * *can* occur here, so timeouts are a problem. It assumed that 
 * sos_anmake_conn()  cannot block for too long (see below).
 */
int
sos_amake_conn(char *hoststring, char *port, int timeout_val, int socopts,  int fdflags, int *isconn, FILE *verbose)
{
  SOS_ENTRY("sos_make_conn","sos_make_conn",NULL);
  int inport;
  u_short nport;
  struct hostent *hostp;
  struct in_addr addr;
  struct itimerval newit,oldit;
  int sock;

  /*
   * We need to sneak our timeout values
   * without disturbing everything else
   */
  if (timeout_val)
    {
      newit.it_interval.tv_sec = newit.it_interval.tv_usec = timeout_val;
      newit.it_value.tv_sec = timeout_val; newit.it_value.tv_usec = 0;
      setitimer(ITIMER_REAL, &newit, &oldit);
      sigaction(SIGALRM, (struct sigaction *)&timeout,
		(struct sigaction *)&old_action);
    }
  TIMEDOUT = 0;

  inport = sos_getsbyfoo(port);
  nport = inport;

  if (inport < 0)
    {
      sos_error_printf("Invalid port %s: %s\n",port,strerror(errno));
      SOS_RETURN(-1);
    }

  /*
   * Get the hostent containing all addresses
   *
   * (Note: if hoststring is an dotted quad
   * it will NOT be looked up in DNS -- This is really what the caller 
   * probably wants to do. 
   */
  memset(&addr, (char)0, sizeof(addr));
  if ( sos_getabyfoo(hoststring, &addr) )
    {
      sos_error_printf("Failed host lookup of \"%s\"\n", hoststring);
      SOS_RETURN(-1);
    }

  sock = sos_anmake_conn(&addr, nport, socopts, fdflags, isconn, verbose);
  
  if (TIMEDOUT) 
    { 
      if (verbose) 
	fprintf(verbose, "  Timed out.\r\n");
      close(sock);
      sock = -1;
    }

  if (timeout_val)
    {
      sigaction(SIGALRM, (struct sigaction *)&old_action,
		(struct sigaction *)NULL);
      setitimer(ITIMER_REAL, &oldit, NULL);
    }



  SOS_RETURN(sock);
}



/*
 * Numeric form
 */
int
sos_nmake_conn(char *hoststring, u_short port, int timeout_val, int socopts, FILE *verbose)
{
  SOS_ENTRY("sos_make_conn","sos_nmake_conn",NULL);
  struct hostent *hostp;	/* Library return value */
  struct sockaddr_in svc_addr;	/* Name of the service on host */
  int sock = -1;		/* Connection_socket */
  int one = 1;
  char **haddrs;
  struct itimerval newit,oldit;

  /*
   * We need to sneak our timeout values
   * without disturbing everything else
   */
  if (timeout_val)
    {
      newit.it_interval.tv_sec = newit.it_interval.tv_usec = 0;
      newit.it_value.tv_sec = timeout_val; newit.it_value.tv_usec = 0;
      setitimer(ITIMER_REAL, &newit, &oldit);
      sigaction(SIGALRM, (struct sigaction *)&timeout,
		(struct sigaction *)&old_action);
    }

  /*
   * Set the association structure.
   */
  memset((caddr_t) & svc_addr, 0, sizeof(svc_addr));
  svc_addr.sin_family = AF_INET;

  svc_addr.sin_port = SOS_HTONS(port);

  /*
   * Get the hostent containing all addresses
   *
   * (Note: if hoststring is an dotted quad
   * it will NOT be looked up in DNS)
   */
  if ( (hostp=sos_getmcbyfoo(hoststring)) == NULL )
    {
      sock = -2;
      goto exit;
    }

  /* Try to connect to all A records */
  for (haddrs = hostp->h_addr_list; *haddrs; haddrs++)
    {
      memcpy((caddr_t)&svc_addr.sin_addr, *haddrs, sizeof(struct in_addr));
      TIMEDOUT = 0;

      if (verbose)
	{
	  fprintf(verbose, "Trying... %s port %d.",
		  inet_ntoa(svc_addr.sin_addr), port);
	  fflush(verbose);
	}

      /* Creat a new socket */
      if (timeout_val)
	{
	  newit.it_value.tv_sec = timeout_val; newit.it_value.tv_usec = 0;
	  setitimer(ITIMER_REAL, &newit, NULL);
	}
      if ((sock = socket(svc_addr.sin_family, SOCK_STREAM, 0)) < 0)
	{
	  if (TIMEDOUT) { continue; }
	  sos_error_printf("Could not create socket: %s\n",strerror(errno));
	  goto exit;
	}
      if (TIMEDOUT) { if (verbose) fprintf(verbose, "  Timed out.\r\n"); close(sock); sock = -1; continue; }

      /* Set the options on this socket */
      if (timeout_val)
	{
	  newit.it_value.tv_sec = timeout_val; newit.it_value.tv_usec = 0;
	  setitimer(ITIMER_REAL, &newit, NULL);
	}
      if (socopts)
	if (setsockopt(sock, SOL_SOCKET, socopts, (char *)&one, sizeof(one)) < 0)
	  {
	    if (TIMEDOUT) { if (verbose) fprintf(verbose, "  Timed out.\r\n"); close(sock); sock = -1; continue; }
	    close(sock);
	    sock = -1;
	    sos_error_printf("Could not set socket options (%d): %s\n",socopts,strerror(errno));
	    goto exit;
	  }

      /* Try to connect to this new address */
      if (timeout_val)
	{
	  newit.it_value.tv_sec = timeout_val; newit.it_value.tv_usec = 0;
	  setitimer(ITIMER_REAL, &newit, NULL);
	}
      if ((connect(sock, (struct sockaddr *)&svc_addr, sizeof(svc_addr))) < 0)
	{
	  sos_error_printf("Could not connect to remote address: <%s, %d>: %s\n",inet_ntoa(svc_addr.sin_addr), port, strerror(errno));
	  if (TIMEDOUT)
	    {
	      if (verbose) fprintf(verbose,"  Connection timed out.\r\n");
	    }
	  else
	    {
	      if (verbose) fprintf(verbose,"  Connection refused.\r\n");
	    }
	    
	  close(sock);
	  sock = -1;
	  continue;
	}

      /* WE WON!!! */
      if (verbose) 	
	{
	  fprintf(verbose, "  Accepted.\r\n");
	  fflush(verbose);
	}
      break;
    }

  if (sock < 0)
    sos_error_printf("Could not connect: %s\n",strerror(errno));

 exit:
  if (timeout_val)
    {
      sigaction(SIGALRM, (struct sigaction *)&old_action,
		(struct sigaction *)NULL);
      setitimer(ITIMER_REAL, &oldit, NULL);
    }
  
  SOS_RETURN (sock);
}



/*
 * Numeric form
 */
int
sos_nnmake_conn(struct in_addr *addr, u_short port, int timeout_val, int socopts, FILE *verbose)
{
  SOS_ENTRY("sos_make_conn","sos_nmake_conn",NULL);
  struct sockaddr_in svc_addr;	/* Name of the service on host */
  int sock = -1;		/* Connection_socket */
  int one = 1;
  struct itimerval newit,oldit;

  /*
   * We need to sneak our timeout values
   * without disturbing everything else
   */
  if (timeout_val)
    {
      newit.it_interval.tv_sec = newit.it_interval.tv_usec = 0;
      newit.it_value.tv_sec = timeout_val; newit.it_value.tv_usec = 0;
      setitimer(ITIMER_REAL, &newit, &oldit);
      sigaction(SIGALRM, (struct sigaction *)&timeout,
		(struct sigaction *)&old_action);
    }

  /*
   * Set the association structure.
   */
  memset((caddr_t) & svc_addr, 0, sizeof(svc_addr));
  svc_addr.sin_family = AF_INET;

  svc_addr.sin_port = SOS_HTONS(port);

  memcpy((caddr_t)&svc_addr.sin_addr, addr, sizeof(struct in_addr));
   TIMEDOUT = 0;

  if (verbose)
    {
      fprintf(verbose, "Trying... %s port %d.",
	      inet_ntoa(svc_addr.sin_addr), port);
      fflush(verbose);
    }

  /* Creat a new socket */
  if (timeout_val)
    {
      newit.it_value.tv_sec = timeout_val; newit.it_value.tv_usec = 0;
      setitimer(ITIMER_REAL, &newit, NULL);
    }
  if ((sock = socket(svc_addr.sin_family, SOCK_STREAM, 0)) < 0)
    {
      if (TIMEDOUT) { goto oldloopend; }
      sos_error_printf("Could not create socket: %s\n",strerror(errno));
      goto exit;
    }
  if (TIMEDOUT) { if (verbose) fprintf(verbose, "  Timed out.\r\n"); close(sock); sock = -1; goto oldloopend; }

  /* Set the options on this socket */
  if (timeout_val)
    {
      newit.it_value.tv_sec = timeout_val; newit.it_value.tv_usec = 0;
      setitimer(ITIMER_REAL, &newit, NULL);
    }
  if (socopts)
    if (setsockopt(sock, SOL_SOCKET, socopts, (char *)&one, sizeof(one)) < 0)
      {
	if (TIMEDOUT) { if (verbose) fprintf(verbose, "  Timed out.\r\n"); close(sock); sock = -1; goto oldloopend; }
	close(sock);
	sock = -1;
	sos_error_printf("Could not set socket options (%d): %s\n",socopts,strerror(errno));
	goto exit;
      }

  /* Try to connect to this new address */
  if (timeout_val)
    {
      newit.it_value.tv_sec = timeout_val; newit.it_value.tv_usec = 0;
      setitimer(ITIMER_REAL, &newit, NULL);
    }
  if ((connect(sock, (struct sockaddr *)&svc_addr, sizeof(svc_addr))) < 0)
    {
      sos_error_printf("Could not connect to remote address: <%s, %d>: %s\n",inet_ntoa(svc_addr.sin_addr), port, strerror(errno));
      if (TIMEDOUT)
	{
	  if (verbose) fprintf(verbose,"  Connection timed out.\r\n");
	}
      else
	{
	  if (verbose) fprintf(verbose,"  Connection refused.\r\n");
	}
	    
      close(sock);
      sock = -1;
      goto oldloopend;
    }

  /* WE WON!!! */
  if (verbose) 	
    {
      fprintf(verbose, "  Accepted.\r\n");
      fflush(verbose);
    }

oldloopend:
  if (sock < 0)
    sos_error_printf("Could not connect: %s\n",strerror(errno));

 exit:
  if (timeout_val)
    {
      sigaction(SIGALRM, (struct sigaction *)&old_action,
		(struct sigaction *)NULL);
      setitimer(ITIMER_REAL, &oldit, NULL);
    }

  SOS_RETURN (sock);
}

/*
 * Asynchronous connect. We're going to postulate that *this* routine cannot
 * block (no system call except connect() even has "would block" error 
 * indicators, so timeout stuff is *not* included in this level of the 
 * interface (but see sos_amake_conn() above). It is assumed that callers 
 * will check for timeout immediately before and after calling. 
 */
int
sos_anmake_conn(struct in_addr *addr, u_short port, int socopts, int fdflags, int *isconn, FILE *verbose)
{
  SOS_ENTRY("sos_make_conn","sos_amake_conn",NULL);
  struct sockaddr_in svc_addr;	/* Name of the service on host */
  int sock = -1;		/* Connection_socket */
  int one = 1;
  int flags;
  int lisconn=-1;

  if ( !addr )
    {
      sos_error_printf("Illegal Arguments");
      goto exit;
    }

  lisconn=-1;
  /*
   * Set the association structure.
   */
  memset((caddr_t) & svc_addr, 0, sizeof(svc_addr));
  svc_addr.sin_family = AF_INET;

  svc_addr.sin_port = SOS_HTONS(port);

  memcpy((caddr_t)&svc_addr.sin_addr, addr, sizeof(struct in_addr));
  

  if (verbose)
    {
      fprintf(verbose, "Trying... %s port %d.",
	      inet_ntoa(svc_addr.sin_addr), port);
      fflush(verbose);
    }

  if ((sock = socket(svc_addr.sin_family, SOCK_STREAM, 0)) < 0)
    {
      sos_error_printf("Could not create socket: %s\n",strerror(errno));
      goto exit;
    }

  if ( (flags = fcntl(sock, F_GETFL, (FCNTL_ARG_T)0)) < 0 ||
       (fcntl(sock, F_SETFL, (FCNTL_ARG_T)(flags|fdflags))) < 0 )
    {
      sos_error_printf("Failed to set (nonblocking) descriptor flags on fd: %d\n", sock);
      close(sock);
      sock=-1;
      goto exit;
    }

  if (socopts)
    if (setsockopt(sock, SOL_SOCKET, socopts, (char *)&one, sizeof(one)) < 0)
      {
	close(sock);
	sock = -1;
	sos_error_printf("Could not set socket options (%d): %s\n",socopts,strerror(errno));
	goto exit;
      }

  if ((connect(sock, (struct sockaddr *)&svc_addr, sizeof(svc_addr))) < 0)
    {
      if ( errno == EINPROGRESS || errno == EALREADY )
	{
	  lisconn=0;
	  sos_debug_printf_and(1,"Connection to [%s] pending: %s", 
			       inet_ntoa(*addr), strerror(errno));
	  goto exit;
	}
      sos_error_printf("Could not connect to remote address: <%s, %d>: %s\n",inet_ntoa(svc_addr.sin_addr), port, strerror(errno));
      close(sock);
      sock = -1;
      goto oldloopend;
    }

  /* WE WON!!! */
  lisconn=1;
  if (verbose) 	
    {
      fprintf(verbose, "  Accepted.\r\n");
      fflush(verbose);
    }

oldloopend:
  if (sock < 0)
    sos_error_printf("Could not connect: %s\n",strerror(errno));

 exit:
  if ( isconn )
    *isconn = lisconn;
  SOS_RETURN (sock);
}

/*
 * Attempt to establish a UDP connection to the port service on 
 * hoststring. Allow both numeric and name representations of ports.
 * Try to connect to all known addresses for the host.
 *
 * Return system call result. Will be socket of laundered connection if all
 * goes according to plan.
 */
int
sos_make_uconn(char *hoststring, char *port, int timeout_val, int socopts, FILE *verbose)
{
  SOS_ENTRY("sos_make_conn","sos_make_uconn",NULL);
  int nport;
  struct hostent *hostp;	/* Library return value */
  struct sockaddr_in svc_addr;	/* Name of the service on host */
  int sock = -1;		/* Connection_socket */
  int one = 1;
  char **haddrs;
  struct itimerval newit,oldit;

  nport = sos_getsbyfoo(port);

  if (nport < 0)
    {
      sos_error_printf("Invalid port %s: %s\n",port,strerror(errno));
      SOS_RETURN(-1);
    }

  /*
   * We need to sneak our timeout values
   * without disturbing everything else
   */
  if (timeout_val)
    {
      newit.it_interval.tv_sec = newit.it_interval.tv_usec = 0;
      newit.it_value.tv_sec = timeout_val; newit.it_value.tv_usec = 0;
      setitimer(ITIMER_REAL, &newit, &oldit);
      sigaction(SIGALRM, (struct sigaction *)&timeout,
		(struct sigaction *)&old_action);
    }

  /*
   * Set the association structure.
   */
  memset((caddr_t) & svc_addr, 0, sizeof(svc_addr));
  svc_addr.sin_family = AF_INET;

  svc_addr.sin_port = SOS_HTONS(nport);

  /*
   * Get the hostent containing all addresses
   *
   * (Note: if hoststring is an dotted quad
   * it will NOT be looked up in DNS)
   */
  if ( (hostp=sos_getmcbyfoo(hoststring)) == NULL )
    {
      sock = -2;
      goto exit;
    }

  /* Try to connect to all A records */
  for (haddrs = hostp->h_addr_list; *haddrs; haddrs++)
    {
      memcpy((caddr_t)&svc_addr.sin_addr, *haddrs, sizeof(struct in_addr));
      TIMEDOUT = 0;

      if (verbose)
	{
	  fprintf(verbose, "Trying... %s port %d.",
		  inet_ntoa(svc_addr.sin_addr), port);
	  fflush(verbose);
	}

      /* Creat a new socket */
      if (timeout_val)
	{
	  newit.it_value.tv_sec = timeout_val; newit.it_value.tv_usec = 0;
	  setitimer(ITIMER_REAL, &newit, NULL);
	}
      if ((sock = socket(svc_addr.sin_family, SOCK_DGRAM, 0)) < 0)
	{
	  if (TIMEDOUT) { continue; }
	  sos_error_printf("Could not create socket: %s\n",strerror(errno));
	  goto exit;
	}
      if (TIMEDOUT) { if (verbose) fprintf(verbose, "  Timed out.\r\n"); close(sock); sock = -1; continue; }

      /* Set the options on this socket */
      if (timeout_val)
	{
	  newit.it_value.tv_sec = timeout_val; newit.it_value.tv_usec = 0;
	  setitimer(ITIMER_REAL, &newit, NULL);
	}
      if (socopts)
	if (setsockopt(sock, SOL_SOCKET, socopts, (char *)&one, sizeof(one)) < 0)
	  {
	    if (TIMEDOUT) { if (verbose) fprintf(verbose, "  Timed out.\r\n"); close(sock); sock = -1; continue; }
	    close(sock);
	    sock = -1;
	    sos_error_printf("Could not set socket options (%d): %s\n",socopts,strerror(errno));
	    goto exit;
	  }

      /* Try to connect to this new address */
      if (timeout_val)
	{
	  newit.it_value.tv_sec = timeout_val; newit.it_value.tv_usec = 0;
	  setitimer(ITIMER_REAL, &newit, NULL);
	}
      if ((connect(sock, (struct sockaddr *)&svc_addr, sizeof(svc_addr))) < 0)
	{
	  sos_error_printf("Could not connect to remote address: <%s, %d>: %s\n",inet_ntoa(svc_addr.sin_addr), port, strerror(errno));
	  if (TIMEDOUT)
	    {
	      if (verbose) fprintf(verbose,"  Connection timed out.\r\n");
	    }
	  else
	    {
	      if (verbose) fprintf(verbose,"  Connection refused.\r\n");
	    }
	    
	  close(sock);
	  sock = -1;
	  continue;
	}

      /* WE WON!!! */
      if (verbose) 	
	{
	  fprintf(verbose, "  Accepted.\r\n");
	  fflush(verbose);
	}
      break;
    }

  if (sock < 0)
    sos_error_printf("Could not connect: %s\n",strerror(errno));

 exit:
  if (timeout_val)
    {
      sigaction(SIGALRM, (struct sigaction *)&old_action,
		(struct sigaction *)NULL);
      setitimer(ITIMER_REAL, &oldit, NULL);
    }

  SOS_RETURN (sock);
}



/*
 * Attempt to establish a UNIX connection to the unix socket.
 *
 * Return system call result. Will be socket of laundered connection if all
 * goes according to plan.
 */
int
sos_make_unixconn(char *filename, int timeout_val, int socopts, FILE *verbose)
{
  SOS_ENTRY("sos_make_conn","sos_make_unixconn",NULL);
  int nport;
  struct sockaddr_un svc_addr;	/* Name of the service on host */
  int sock = -1;		/* Connection_socket */
  int one = 1;
  struct itimerval newit,oldit;

  if (!filename)
    {
      sos_error_printf("Invalid argument\n");
      SOS_RETURN(-1);
    }

  /*
   * We need to sneak our timeout values
   * without disturbing everything else
   */
  if (timeout_val)
    {
      newit.it_interval.tv_sec = newit.it_interval.tv_usec = 0;
      newit.it_value.tv_sec = timeout_val; newit.it_value.tv_usec = 0;
      setitimer(ITIMER_REAL, &newit, &oldit);
      sigaction(SIGALRM, (struct sigaction *)&timeout,
		(struct sigaction *)&old_action);
    }

  /*
   * Set the association structure.
   */
  memset((caddr_t) & svc_addr, 0, sizeof(svc_addr));
  svc_addr.sun_family = AF_UNIX;
  strncpy(svc_addr.sun_path,filename,sizeof(svc_addr.sun_path));

  TIMEDOUT = 0;

  if (verbose)
    {
      fprintf(verbose, "Trying... %s.",filename);
      fflush(verbose);
    }

  /* Creat a new socket */
  if (timeout_val)
    {
      newit.it_value.tv_sec = timeout_val; newit.it_value.tv_usec = 0;
      setitimer(ITIMER_REAL, &newit, NULL);
    }
  if ((sock = socket(svc_addr.sun_family, SOCK_STREAM, 0)) < 0)
    {
      if (TIMEDOUT) { goto breakpoint; }
      sos_error_printf("Could not create socket: %s\n",strerror(errno));
      goto exit;
    }
  if (TIMEDOUT) { if (verbose) fprintf(verbose, "  Timed out.\r\n"); close(sock); sock = -1; goto breakpoint; }

  /* Set the options on this socket */
  if (timeout_val)
    {
      newit.it_value.tv_sec = timeout_val; newit.it_value.tv_usec = 0;
      setitimer(ITIMER_REAL, &newit, NULL);
    }
  if (socopts)
    if (setsockopt(sock, SOL_SOCKET, socopts, (char *)&one, sizeof(one)) < 0)
      {
	if (TIMEDOUT) { if (verbose) fprintf(verbose, "  Timed out.\r\n"); close(sock); sock = -1; goto breakpoint; }
	close(sock);
	sock = -1;
	sos_error_printf("Could not set socket options (%d): %s\n",socopts,strerror(errno));
	goto exit;
      }

  sos_debug_printf_and(1,"Unix domain socket: %s\n",svc_addr.sun_path);

  /* Try to connect to this new address */
  if (timeout_val)
    {
      newit.it_value.tv_sec = timeout_val; newit.it_value.tv_usec = 0;
      setitimer(ITIMER_REAL, &newit, NULL);
    }
  if ((connect(sock, (struct sockaddr *)&svc_addr, sizeof(svc_addr))) < 0)
    {
      sos_error_printf("Could not connect to remote part: <%s>: %s\n",svc_addr.sun_path, strerror(errno));
      if (TIMEDOUT)
	{
	  if (verbose) fprintf(verbose,"  Connection timed out.\r\n");
	}
      else
	{
	  if (verbose) fprintf(verbose,"  Connection refused.\r\n");
	}
	    
      close(sock);
      sock = -1;
      goto breakpoint;
    }

  /* WE WON!!! */
  if (verbose) 	
    {
      fprintf(verbose, "  Accepted.\r\n");
      fflush(verbose);
    }

breakpoint:
  if (sock < 0)
    sos_error_printf("Could not connect: %s\n",strerror(errno));

 exit:
  if (timeout_val)
    {
      sigaction(SIGALRM, (struct sigaction *)&old_action,
		(struct sigaction *)NULL);
      setitimer(ITIMER_REAL, &oldit, NULL);
    }

  SOS_RETURN (sock);
}
