#if !defined(lint) && !defined(__INSIGHT__)
static char sos__rcsid[] = "$Id$";
static char sos__copyright[] = "Copyright (c) 1994, 1995, 1996 SOS Corporation";
static char sos__contact[] = "SOS Corporation <sos-info@soscorp.com> +1 800 SOS UNIX";
#endif /* not lint */

/*
 * ++Copyright Released Product++
 *
 * Copyright (c) 1994, 1995, 1996 Sources of Supply Corporation ("SOS").
 * All rights reserved.
 *
 * The SOS Released Product License Agreement specifies the terms and
 * conditions for redistribution.  You may find the License Agreement
 * in the file LICENSE.
 *
 * SOS Corporation
 * 461 5th Ave.; 16th floor
 * New York, NY 10017
 *
 * +1 800 SOS UNIX
 * <sos-info@soscorp.com>
 *
 * --Copyright Released Product--
 */

/*
 * Bounded message queue functions
 *
 * Have a queue of arbitrary strings which is bounded
 * (after n entries have been added to the queue, the
 * n+1th entry will cause the 1st entry to fall off).
 *
 * Note that the strings are strdupped and then freed.
 *
 * Sort of like a ring buffer.
 *
 * Debug levels:
 *
 * 1 - debug messages added
 * 2 - error messages (we cannot use sos_error)
 */

#include "sos.h"



/*
 * Create a message queue and perform initialization
 */
sos_mq_t sos_mq_create(int queue_len)
{
  SOS_ENTRY("sos_mq","sos_mq_create",NULL);
  struct sos_mq_type *new = malloc(sizeof(struct sos_mq_type));

  if (!new)
    {
      sos_debug_printf_and(2,"Can not malloc %d bytes\n",sizeof(struct sos_mq_type));
      SOS_RETURN(NULL);
    }

  if (queue_len < 1)
    SOS_RETURN(NULL);

  new->queue = dll_create(NULL, NULL, DICT_UNORDERED, NULL);

  if (!new->queue)
    {
      sos_debug_printf_and(2,"Can not create dll\n");
      free(new);
      SOS_RETURN(NULL);
    }

  new->curlen = 0;
  new->maxlen = queue_len;

  SOS_RETURN(new);
}


/*
 * Add a string to the message queue
 * (possibly causing an older one to fall off the end)
 */
int sos_mq_add(sos_mq_t mq, char *str)
{
  SOS_ENTRY("sos_mq","sos_mq_add",NULL);
  char *dup;

  if (!mq || !str)
    SOS_RETURN(-1);

  if (!(dup = strdup(str)))
    {
      sos_debug_printf_and(2,"could not duplicate %s\n",str);
      SOS_RETURN(-1);
    }

  /* Nuke all elements over bounded queue length */
  while (mq->curlen >= mq->maxlen)
    {
      dict_obj foo;

      /* FIFO */
      if (!(foo = dll_maximum(mq->queue)))
	{
	  sos_debug_printf_and(2,"message queue totally fucked--curlen does not agree with queue");
	  SOS_RETURN(-1);
	}
      dll_delete(mq->queue,foo);
      free(foo);		/* Free strduped string */
      mq->curlen--;
    }

  /* Insert new element */
  if (dll_insert(mq->queue, dup) != DICT_OK)
    {
      free(dup);
      sos_debug_printf_and(2,"could not insert into mq dll\n");
      SOS_RETURN(-1);
    }

  mq->curlen++;
  sos_debug_printf_and(1,"adding message %s\n",dup);

  SOS_RETURN(0);
}


/*
 * Add a string generated by sprintf to the message queue
 * (possibly causing an older one to fall off the end)
 *
 */
int sos_mq_printf(sos_mq_t mq, char *fmt, ...)
{
  SOS_ENTRY("sos_mq","sos_mq_printf",NULL);
  va_list args;
  char outbuf[1024];

  if (!mq || !fmt)
    SOS_RETURN(0);

  va_start(args, fmt);
  vsnprintf(outbuf, 1024, fmt, args);
  va_end(args);

  SOS_RETURN(sos_mq_add(mq, outbuf));
}


/*
 * Print, non-destructively, the current strings in the queue
 * to the supplied file descriptor
 */
int sos_mq_fdout(sos_mq_t mq, int fd)
{
  SOS_ENTRY("sos_mq","sos_mq_fdout",NULL);
  dict_obj foo;
  int tmp, ret = 0;

  if (!mq || fd < 0)
    SOS_RETURN(0);

  for(foo=dll_maximum(mq->queue);foo;foo=dll_predecessor(mq->queue, foo))
    {
      if ((tmp = write(fd,foo,strlen(foo))) < 0)
	{
	  sos_debug_printf_and(2,"could not write mq");
	  SOS_RETURN(-1);
	}
      ret += tmp;
    }

  SOS_RETURN(ret);
}


/*
 * Print, non-destructively, the current strings in the queue
 * to the supplied file handle
 */
int sos_mq_stdout(sos_mq_t mq, FILE *out)
{
  SOS_ENTRY("sos_mq","sos_mq_stdout",NULL);
  dict_obj foo;
  int tmp, ret = 0;

  if (!mq || !out)
    SOS_RETURN(0);

  for(foo=dll_maximum(mq->queue);foo;foo=dll_predecessor(mq->queue, foo))
    {
      if ((tmp = fputs(foo,out)) < 0)
	{
	  sos_debug_printf_and(2,"could not fputs mq");
	  SOS_RETURN(-1);
	}
      ret += tmp;
    }

  SOS_RETURN(ret);
}


/*
 * Print, non-destructively, the current strings in the queue
 * via soslog at the supplied priority
 */
int sos_mq_soslog(sos_mq_t mq, int priority)
{
  SOS_ENTRY("sos_mq","sos_mq_soslog",NULL);
  dict_obj foo;
  int tmp, ret = 0;

  if (!mq)
    SOS_RETURN(0);

  for(foo=dll_maximum(mq->queue);foo;foo=dll_predecessor(mq->queue, foo))
    {
      soslog(priority, SOSLOG_TYPE_CONDITION, 0, "%s", foo);
    }

  SOS_RETURN(ret);
}


/*
 * Peek at the depth string from the top (0 is top)
 */
char *sos_mq_peek(sos_mq_t mq, int depth)
{
  SOS_ENTRY("sos_mq","sos_mq_peek",NULL);
  dict_obj foo;
  int cntr;

  if (!mq || depth < 0)
    SOS_RETURN(0);

  for(cntr = 0, foo=dll_maximum(mq->queue);
      foo && (cntr<depth);
      foo=dll_predecessor(mq->queue, foo),cntr++)
    ;

  SOS_RETURN(foo);
}


/*
 * Pop number of strings from the queue (1 will pop the top--0 will pop all)
 */
int sos_mq_pop(sos_mq_t mq, int number)
{
  SOS_ENTRY("sos_mq","sos_mq_pop",NULL);
  dict_obj foo;
  int cntr = 0;

  if (!mq || number < 0)
    SOS_RETURN(0);

  if (number == 0)
    number = INT_MAX;

  for(foo=dll_maximum(mq->queue);foo && number;foo=dll_maximum(mq->queue),number--)
    {
      mq->curlen--;
      dll_delete(mq->queue,foo);
      free(foo);
      cntr++;
    }

  SOS_RETURN(cntr);
}


/*
 * Destroy the queue and all messages in it
 */
int sos_mq_destroy(sos_mq_t mq)
{
  SOS_ENTRY("sos_mq","sos_mq_destroy",NULL);

  if (!mq)
    SOS_RETURN(0);

  sos_mq_pop(mq, 0);
  dll_destroy(mq->queue);
  free(mq);

  SOS_RETURN(0);
}
