#if !defined(lint) && !defined(__INSIGHT__)
static char sos__rcsid[] = "$Id$";
static char sos__copyright[] = "Copyright (c) 1994, 1995, 1996 SOS Corporation";
static char sos__contact[] = "SOS Corporation <sos-info@soscorp.com> +1 800 SOS UNIX";
#endif /* not lint */

/*
 * ++Copyright Released Product++
 *
 * Copyright (c) 1994, 1995, 1996 Sources of Supply Corporation ("SOS").
 * All rights reserved.
 *
 * The SOS Released Product License Agreement specifies the terms and
 * conditions for redistribution.  You may find the License Agreement
 * in the file LICENSE.
 *
 * SOS Corporation
 * 461 5th Ave.; 16th floor
 * New York, NY 10017
 *
 * +1 800 SOS UNIX
 * <sos-info@soscorp.com>
 *
 * --Copyright Released Product--
 */

/*
 * Bounded message queue functions
 *
 * Have a queue of arbitrary strings which is bounded
 * (after n entries have been added to the queue, the
 * n+1th entry will cause the 1st entry to fall off).
 *
 * Note that the strings are strdupped and then freed.
 *
 * Sort of like a ring buffer.
 *
 * Debug levels:
 *
 * 1 - debug messages added
 * 2 - error messages (we cannot use sos_error)
 */

#include "sos.h"

/*
 * Create a sos queue and perform initialization
 * NB This code was stolen directly from sos_mq stuff, so a many `mq' symbols
 * remain which really don't point at message queues. Just stay on your toes.
 */


sos_mq_t sos_q_create(int queue_len)
{
  SOS_ENTRY("sos_q","sos_mq_create",NULL);
  struct sos_mq_type *new = malloc(sizeof(struct sos_mq_type));

  if (!new)
    {
      sos_debug_printf_and(2,"Can not malloc %d bytes\n",sizeof(struct sos_mq_type));
      SOS_RETURN(NULL);
    }

  if (queue_len < 1)
    SOS_RETURN(NULL);

  new->queue = dll_create(NULL, NULL, DICT_UNORDERED, NULL);

  if (!new->queue)
    {
      sos_debug_printf_and(2,"Can not create dll\n");
      free(new);
      SOS_RETURN(NULL);
    }

  new->curlen = 0;
  new->maxlen = queue_len;

  SOS_RETURN(new);
}


/*
 * Add a string to the message queue
 * (possibly causing an older one to fall off the end)
 */
int sos_q_add(sos_mq_t mq, void *data, void(*destr)(void *data))
{
  SOS_ENTRY("sos_q","sos_q_add",NULL);
  sos_q_elem_t elem;

  /* DATA may me null I suppose, though I can't imagine why */
  if (!mq )
    SOS_RETURN(-1);

  /* Nuke all elements over bounded queue length */
  while (mq->curlen >= mq->maxlen)
    {
      dict_obj foo;
      sos_q_elem_t qe;

      /* FIFO */
      if (!(foo = dll_maximum(mq->queue)))
	{
	  sos_debug_printf_and(2,"generic queue encountered internal inconsistancies");
	  SOS_RETURN(-1);
	}
      
      qe=(sos_q_elem_t)foo;

      if ( qe->destr)		/* This should clean up both data and args. */
	{
	  (*(qe->destr))(qe->data);
	}

      dll_delete(mq->queue,foo);
      free(foo);		/* Foo is our responsibility */
      mq->curlen--;
    }


  if ( !(elem=malloc(sizeof *elem)) )
    {
      sos_error_printf("Error allocating Q data element: %s", strerror(errno));
      SOS_RETURN(-1);
    }

  elem->data = data;
  elem->destr = destr;

  /* Insert new element */
  if (dll_insert(mq->queue, elem) != DICT_OK)
    {
      sos_debug_printf_and(2,"could not insert into mq dll\n");
      free(elem);
      SOS_RETURN(-1);
    }

  mq->curlen++;

  SOS_RETURN(0);
}


/*
 * Peek at the depth string from the top (0 is top)
 */
void *sos_q_peek(sos_mq_t mq, int depth)
{
  SOS_ENTRY("sos_q","sos_q_peek",NULL);
  dict_obj foo;
  int cntr;

  if (!mq )
    SOS_RETURN(0);

  /*
   * NB!!!
   * depth = 0 means top of the Queue.
   * depth = -1 means bottom
   * Would probably be nicer if 1=>top, -1=>bottom, but we have legacy 
   * code to manage.
   */
  if ( depth >= 0 )
    {
      for(cntr = 0, foo=dll_maximum(mq->queue);
	  foo && (cntr<depth);
	  foo=dll_predecessor(mq->queue, foo),cntr++)
	;
    }
  else
    {
      depth=-depth;
      for(cntr = 1, foo=dll_minimum(mq->queue);
	  foo && (cntr<depth);
	  foo=dll_successor(mq->queue, foo),cntr++)
	;
    }

  if ( foo)
    SOS_RETURN( ((sos_q_elem_t)foo)->data  );
  else
    SOS_RETURN(NULL);
}


/*
 * Pop number of strings from the queue (1 will pop the top--0 will pop all)
 */
int sos_q_pop(sos_mq_t mq, int number)
{
  SOS_ENTRY("sos_q","sos_mq_pop",NULL);
  dict_obj foo;
  int cntr = 0;
  sos_q_elem_t qe;

  if (!mq || number < 0)
    SOS_RETURN(0);

  if (number == 0)
    number = INT_MAX;

  for(foo=dll_maximum(mq->queue);foo && number;foo=dll_maximum(mq->queue),number--)
    {
      mq->curlen--;
      qe=(sos_q_elem_t)foo;

      if ( qe->destr)
	(*(qe->destr))(qe->data);

      dll_delete(mq->queue,foo);
      free(foo);
      cntr++;
    }

  SOS_RETURN(cntr);
}



/*
 * Destroy the queue and all messages in it
 */
int sos_q_destroy(sos_mq_t mq)
{
  SOS_ENTRY("sos_q","sos_q_destroy",NULL);

  if (!mq)
    SOS_RETURN(0);

  sos_mq_pop(mq, 0);
  dll_destroy(mq->queue);
  free(mq);

  SOS_RETURN(0);
}



/*
 * Return length and maximum length of the suppied Q
 */
int sos_q_length(sos_mq_t mq, int *curlen, int *maxlen)
{
  SOS_ENTRY("sos_q","sos_mq_create",NULL);
  int ret=0;

  if ( !mq )
    {
      sos_error_printf("Illegal Arguments\n");
      SOS_RETURN(-1);
    }

  if ( curlen )
    *curlen = mq->curlen;

  if ( maxlen )
    *maxlen = mq->maxlen;

  SOS_RETURN(ret);
}
