#if !defined(lint) && !defined(__INSIGHT__)
static char sos__rcsid[] = "$Id$";
static char sos__copyright[] = "Copyright (c) 1994, 1995, 1996 SOS Corporation";
static char sos__contact[] = "SOS Corporation <sos-info@soscorp.com> +1 800 SOS UNIX";
#endif /* not lint */

/*
 * ++Copyright Released Product++
 *
 * Copyright (c) 1994, 1995, 1996 Sources of Supply Corporation ("SOS").
 * All rights reserved.
 *
 * The SOS Released Product License Agreement specifies the terms and
 * conditions for redistribution.  You may find the License Agreement
 * in the file LICENSE.
 *
 * SOS Corporation
 * 461 5th Ave.; 16th floor
 * New York, NY 10017
 *
 * +1 800 SOS UNIX
 * <sos-info@soscorp.com>
 *
 * --Copyright Released Product--
 */

/*
 *			      S K I D 3
 *
 * SKID3 -- secret-key identification protocol developed for RACE's
 * RIPE project It uses a keyed one-way hash function (a MAC) to
 * provide security, and assumes that both Alice and Bob share a
 * secret key, K.
 *
 * SKID - Secret Key IDentification
 * RACE - Research and development in Advanced Communication technologies in Europe
 * RIPE - RACE Integrity Primitives Evaluation
 * 
 * Implemented from a description of SKID3 in:
 *
 * B. Schneier, _Applied_cryptography_, New York: John Wiley & Sons, 1994.
 *
 * SKID3 source document:
 *
 * RACE, _RIPE_Integrity_Primitives:_Final_Report_of_RACE_Integrity_Primitives_Evaluation
 * (R1040), June 1992.
 *
 *
 * Protocol (as it appears in Schneier--we perform very minor reordering in the code):
 *
 * Step		ALICE		BOB
 * 
 * A 	       	Ra	-->
 * 
 *		Alice generates 64 bit random
 *		number (Ra) and sends to Bob
 * 
 * B			<--	Rb H(Ra,Rb,Bob,K)
 *
 *		Bob generates 64 bit random number
 *		(Rb), computes keyed hash. Bob is Bob's
 *		name (we use source IP address).
 *		K is shared secret key.
 *
 * C		Verify
 *
 *		Alice verifies that Bob knows the secret
 *		key (K) by computing the hash herself.
 *
 * D		H(Rb,Alice,K)	-->
 *
 *		Alice computes keyed hash.  Alice is
 *		Alice's name.
 *
 * E				Verify
 *
 *		Bob verifies that Alice knows the secret
 *		key (K) by computing the hash himself.
 *
 *
 * At step C, Alice knows that Bob knows K.
 * At step E, Bob knows that Alice knows K.
 *
 *
 * Debugging levels:
 *
 * 1 - general tracing
 * 
 * if using noio skid (will hopefully become the default. 
 * 1 -- State entry
 * 2 -- State changes
 * 4 -- i/o requests
 * 32 -- other info		       
 * 64 -- verbosity
 * 128 -- Obnoxious verbosity (may not be used).
 */

#include "sos.h"
#include "md5.h"


static char *skid_name[2] = { "Alice", "Bob" };

static int skid_find_my_name(int fd, int mode, 
			     struct sos_skid_protocol *skidInfo);
static int skid_find_his_name(int fd, int mode, 
			      struct sos_skid_protocol *skidInfo);
static void skid_hash(char *hash, char *R1, char *R2, sos_string *Name,
		      char *Key);



/*
 * sos_skid
 *
 * Routine to perform SKID3 authentication.
 *
 * Returns -1 on error
 * Returns 1 on success
 * Returns 0 on failure
 */
int
sos_skid_with_ident_skidinfo(int fd, int mode, char *key,
			int (*readf)(int, caddr_t, __SIZE_TYPE__),
			int (*writef)(int, caddr_t, __SIZE_TYPE__),
			struct sos_skid_protocol *skidInfo)
{
  SOS_ENTRY("sos_skid","sos_skid_with_ident_skidinfo",NULL);
  unsigned char Remote_Hash[SOS_HASH_LEN];
  int retval = -1;

  /*
   * This function rather 'grew in telling'. Originally named simply 
   * sos_skid(), we have, through daily use, discovered functional 
   * deficiencies in that model which force us to pass in increasing amounts
   * of information from above.  With the advent of random peer `identity'
   * routines (ie who is `Alice' and whois `Bob', we had to `mode' variable. 
   * Later we discoveredd that ofter authentication must occur between hosts 
   * which cannot mutually look up each other in DNS (firewalls getting in 
   * the way, that sort of thing). Therefor we were forced to relegate naming
   * to a higher routine as well.
   */

  if (!skidInfo || fd < 0 || !readf || !writef || !key ||
      ( mode != SOS_IDENT_ALICE && mode != SOS_IDENT_BOB ))
    {
      sos_error_printf("SKID: Invalid argument\n");
      SOS_RETURN(-1);
    }

  sos_debug_printf_and(1,"Welcome, %s, to the SKID3 protocol\n",
		   skid_name[(mode==SOS_IDENT_ALICE)?0:1]);


  /* Fill out shared secret key */
  skidInfo->Key = key;


  /* Generate Random Numbers */
  if (mode == SOS_IDENT_ALICE)
    {
      if (sos_get_rand((unsigned char *)&(skidInfo->Ra), SOS_RAN_LEN) < 1)
	{
	  sos_error_printf("Random numbers failed\n");
	  SOS_RETURN(-1);
	}
    }
  else /* BOB */
    {
      if (sos_get_rand((unsigned char *)&(skidInfo->Rb), SOS_RAN_LEN) < 1)
	{
	  sos_error_printf("Random numbers failed\n");
	  SOS_RETURN(-1);
	}
    }

  /* Alice sends Ra to Bob */
  if (mode == SOS_IDENT_ALICE)
    {				/* Alice write Ra */
      if ( sos_xdr_wencode (fd, writef, "cccccccc", 
			    skidInfo->Ra[0],
			    skidInfo->Ra[1],
			    skidInfo->Ra[2],
			    skidInfo->Ra[3],
			    skidInfo->Ra[4],
			    skidInfo->Ra[5],
			    skidInfo->Ra[6],
			    skidInfo->Ra[7] ) <= 0 )
	{
	  sos_error_printf("Wire encode of Ra failed\n");
	  SOS_RETURN(-1);
	}
    }
  else /* BOB */
    {				/* Bob reads Ra */
      if ( sos_xdr_wdecode (fd, readf, "cccccccc", 
			    &skidInfo->Ra[0],
			    &skidInfo->Ra[1],
			    &skidInfo->Ra[2],
			    &skidInfo->Ra[3],
			    &skidInfo->Ra[4],
			    &skidInfo->Ra[5],
			    &skidInfo->Ra[6],
			    &skidInfo->Ra[7] ) != 8 )
	{
	  sos_error_printf("Wire decode of Ra failed\n");
	  SOS_RETURN(-1);
	}
    }

  sos_debug_printf_and(1,"Ra has been exchanged\n");

  /* Bob sends Rb to Alice */
  if (mode == SOS_IDENT_ALICE)
    {				/* Alice reads Rb */
      if ( sos_xdr_wdecode (fd, readf, "cccccccc", 
			    &skidInfo->Rb[0],
			    &skidInfo->Rb[1],
			    &skidInfo->Rb[2],
			    &skidInfo->Rb[3],
			    &skidInfo->Rb[4],
			    &skidInfo->Rb[5],
			    &skidInfo->Rb[6],
			    &skidInfo->Rb[7] ) != 8 )
	{
	  sos_error_printf("Wire encode of Rb failed\n");
	  SOS_RETURN(-1);
	}
    }
  else /* BOB */
    {				/* Bob writes Rb */
      if ( sos_xdr_wencode (fd, writef, "cccccccc", 
			    skidInfo->Rb[0],
			    skidInfo->Rb[1],
			    skidInfo->Rb[2],
			    skidInfo->Rb[3],
			    skidInfo->Rb[4],
			    skidInfo->Rb[5],
			    skidInfo->Rb[6],
			    skidInfo->Rb[7] ) <= 0 )
	{
	  sos_error_printf("Wire encode of Rb failed\n");
	  SOS_RETURN(-1);
	}
    }

  sos_debug_printf_and(1,"Rb has been exchanged\n");

  /* Compute Hb - Bob's Keyed hash */
  skid_hash(skidInfo->Hb, skidInfo->Ra, skidInfo->Rb, skidInfo->Bob, skidInfo->Key);

  /* Compute Ha - Alice's Keyed hash */
  skid_hash(skidInfo->Ha, skidInfo->Rb, NULL, skidInfo->Alice, skidInfo->Key);

  sos_debug_printf_and(1,"SKID:\tKey: %s, Alice: %s, Bob: %s\n\tRa: %08x %08x, Rb: %08x %08x\n",
		   skidInfo->Key, skidInfo->Alice->str, skidInfo->Bob->str,
		   *((int *)(&(skidInfo->Ra[0]))),
		   *((int *)(&(skidInfo->Ra[4]))),
		   *((int *)(&(skidInfo->Rb[0]))),
		   *((int *)(&(skidInfo->Rb[4]))));
  sos_debug_printf_and(1,"\tHa: %08x %08x %08x %08x\n",
		   *((int *)(&(skidInfo->Ha[0]))),
		   *((int *)(&(skidInfo->Ha[4]))),
		   *((int *)(&(skidInfo->Ha[8]))),
		   *((int *)(&(skidInfo->Ha[12]))));
  sos_debug_printf_and(1,"\tHb: %08x %08x %08x %08x\n\n",
		   *((int *)(&(skidInfo->Hb[0]))),
		   *((int *)(&(skidInfo->Hb[4]))),
		   *((int *)(&(skidInfo->Hb[8]))),
		   *((int *)(&(skidInfo->Hb[12]))));


  /* Bob sends Hb to Alice */
  if (mode == SOS_IDENT_ALICE)
    {				/* Alice reads Hb */
      if ( sos_xdr_wdecode(fd, readf, "cccccccccccccccc",
			   &Remote_Hash[0],
			   &Remote_Hash[1],
			   &Remote_Hash[2],
			   &Remote_Hash[3],
			   &Remote_Hash[4],
			   &Remote_Hash[5],
			   &Remote_Hash[6],
			   &Remote_Hash[7],
			   &Remote_Hash[8],
			   &Remote_Hash[9],
			   &Remote_Hash[10],
			   &Remote_Hash[11],
			   &Remote_Hash[12],
			   &Remote_Hash[13],
			   &Remote_Hash[14],
			   &Remote_Hash[15] ) != 16 )
	{
	  sos_error_printf("Wire decode of Hb failed\n");
	  SOS_RETURN(-1);
	}
    }
  else /* BOB */
    {				/* Bob writes Hb */
      if ( sos_xdr_wencode(fd, writef, "cccccccccccccccc",
			   skidInfo->Hb[0],
			   skidInfo->Hb[1],
			   skidInfo->Hb[2],
			   skidInfo->Hb[3],
			   skidInfo->Hb[4],
			   skidInfo->Hb[5],
			   skidInfo->Hb[6],
			   skidInfo->Hb[7],
			   skidInfo->Hb[8],
			   skidInfo->Hb[9],
			   skidInfo->Hb[10],
			   skidInfo->Hb[11],
			   skidInfo->Hb[12],
			   skidInfo->Hb[13],
			   skidInfo->Hb[14],
			   skidInfo->Hb[15] ) <= 0 )
	{
	  sos_error_printf("Wire encode of Hb failed\n");
	  SOS_RETURN(-1);
	}
    }

  /* Alice verifies that Remote hash is equal to Hb */
  if (mode == SOS_IDENT_ALICE)
    {
      sos_debug_printf_and(1,"SKID:\tRemote hash: %08x %08x %08x %08x\n",
		       *((int *)(&(Remote_Hash[0]))),
		       *((int *)(&(Remote_Hash[4]))),
		       *((int *)(&(Remote_Hash[8]))),
		       *((int *)(&(Remote_Hash[12]))));

      if (memcmp(Remote_Hash,skidInfo->Hb,SOS_HASH_LEN))
	{
	  sos_debug_printf_and(1,"SKID: Failed--Hb != RemoteHash\n");

	  /* Mess up hash value to fool Ben (Bob's imposter) */
	  sos_get_rand((unsigned char *)&(skidInfo->Ha), SOS_HASH_LEN);

	  retval = 0;
	}
      else
	{
	  sos_debug_printf_and(1,"SKID: SUCCEEDED\n");

	  retval = 1;
	}
    }


  /* Alice sends Ha to Bob */
  if (mode == SOS_IDENT_ALICE)
    {				/* Alice writes Ha */
      if ( sos_xdr_wencode(fd, writef, "cccccccccccccccc",
			   skidInfo->Ha[0],
			   skidInfo->Ha[1],
			   skidInfo->Ha[2],
			   skidInfo->Ha[3],
			   skidInfo->Ha[4],
			   skidInfo->Ha[5],
			   skidInfo->Ha[6],
			   skidInfo->Ha[7],
			   skidInfo->Ha[8],
			   skidInfo->Ha[9],
			   skidInfo->Ha[10],
			   skidInfo->Ha[11],
			   skidInfo->Ha[12],
			   skidInfo->Ha[13],
			   skidInfo->Ha[14],
			   skidInfo->Ha[15] ) <= 0 )
	{
	  sos_error_printf("Wire encode of Ha failed\n");
	  SOS_RETURN(-1);
	}
    }
  else /* BOB */
    {				/* Bob reads Ha */
      if ( sos_xdr_wdecode(fd, readf, "cccccccccccccccc",
			   &Remote_Hash[0],
			   &Remote_Hash[1],
			   &Remote_Hash[2],
			   &Remote_Hash[3],
			   &Remote_Hash[4],
			   &Remote_Hash[5],
			   &Remote_Hash[6],
			   &Remote_Hash[7],
			   &Remote_Hash[8],
			   &Remote_Hash[9],
			   &Remote_Hash[10],
			   &Remote_Hash[11],
			   &Remote_Hash[12],
			   &Remote_Hash[13],
			   &Remote_Hash[14],
			   &Remote_Hash[15] ) != 16 )
	{
	  sos_error_printf("Wire decode of Ha failed\n");
	  SOS_RETURN(-1);
	}
    }

  /* Bob verifies that Remote hash is equal to Hb */
  if (mode == SOS_IDENT_BOB)
    {
      sos_debug_printf_and(1,"SKID:\tRemote hash: %08x %08x %08x %08x\n",
		       *((int *)(&(Remote_Hash[0]))),
		       *((int *)(&(Remote_Hash[4]))),
		       *((int *)(&(Remote_Hash[8]))),
		       *((int *)(&(Remote_Hash[12]))));

      if (memcmp(Remote_Hash,skidInfo->Ha,SOS_HASH_LEN))
	{
	  sos_debug_printf_and(1,"SKID: Failed--Ha != RemoteHash\n");

	  /* Hmm.  We have already proven who we are */
	  retval = 0;
	}
      else
	{
	  sos_debug_printf_and(1,"SKID: SUCCEEDED\n");

	  retval = 1;
	}
    }

  if (skidInfo->allocated_names) 
    {
      sos_debug_printf_and(1,"Freeing up allocated space\n");

      sos_freestr(skidInfo->Alice);
      sos_freestr(skidInfo->Alice);
      free(skidInfo->Alice);
      free(skidInfo->Alice);
      skidInfo->allocated_names=0; /* Paranoia */
    }


  sos_debug_printf_and(1,"SKID: Returning %d\n\n",retval);

  SOS_RETURN(retval);
}



const char neterrorstr[]="NON-NETWORKED CONNECTION";
/*
 * skid_find_my_name
 *
 * Find out local user's name via getsockname()
 */
static int
skid_find_my_name(int fd, int mode, 
		  struct sos_skid_protocol *skidInfo)
{
  SOS_ENTRY("sos_skid","skid_find_my_name",NULL);
  struct sockaddr foo;
  int len = sizeof(struct sockaddr);
  sos_string *buf;
  int buflen;

  if (getsockname(fd, &foo, &len) < 0)
    {
      sos_debug_printf_and(1,"getsockname: %s\n",strerror(errno));

      buflen = strlen(neterrorstr);
      buf = sos_allocstr(NULL, buflen);
      if ( !buf )
	{
	  sos_error_printf("Could not allocate %d bytes for name: %s\n",buflen,strerror(errno));
	  SOS_RETURN(-1);
	}
      strncpy(buf->str,neterrorstr,buflen);
    }
  else
    {
      buflen = MIN(sizeof(foo.sa_data),SOS_NAME_SZ);
      buf = sos_allocstr(NULL, buflen);
      if ( !buf )
	{
	  sos_error_printf("Could not allocate %d bytes for name: %s\n",buflen,strerror(errno));
	  SOS_RETURN(-1);
	}
      memcpy(buf->str,foo.sa_data,buflen);
    }

  if (mode == SOS_IDENT_ALICE )
    {
      skidInfo->Alice = buf;
    }
  else
    {
      skidInfo->Bob = buf;
    }
  skidInfo->allocated_names = 1;

  SOS_RETURN(0);
}



/*
 * skid_find_his_name
 *
 * Find out remote user's name via getpeername()
 */
static int
skid_find_his_name(int fd, int mode, 
		   struct sos_skid_protocol *skidInfo)
{
  SOS_ENTRY("sos_skid","skid_find_his_name",NULL);
  struct sockaddr foo;
  int len = sizeof(struct sockaddr);
  sos_string *buf;
  int buflen;

  if (getpeername(fd, &foo, &len) < 0)
    {
      sos_debug_printf_and(1,"getpeername: %s\n",strerror(errno));

      buflen=strlen(neterrorstr);
      buf = sos_allocstr(NULL, buflen);

      if ( !buf )
	{
	  sos_error_printf("Could not allocate %d bytes for name: %s\n",buflen,strerror(errno));
	  SOS_RETURN(-1);
	}

      strncpy(buf->str,neterrorstr,buflen);
    }
  else
    {
      buflen =sizeof(foo.sa_data);
      buf = sos_allocstr(NULL, buflen);

      if ( !buf )
	{
	  sos_error_printf("Could not allocate %d bytes for name: %s\n",buflen,strerror(errno));
	  SOS_RETURN(-1);
	}

      memcpy(buf->str,foo.sa_data,buflen);
    }

  if (mode == SOS_IDENT_ALICE )
    {
      skidInfo->Bob = buf;
    }
  else
    {
      skidInfo->Alice = buf;
    }
  skidInfo->allocated_names = 1;

  SOS_RETURN(0);
}



/*
 * Create a keyed hash
 */
static void 
skid_hash(char *hash, char *R1, char *R2, sos_string *Name, char *Key)
{
  SOS_ENTRY("sos_skid","skid_hash",NULL);
  MD5_CTX mdContext;

  if (!hash || !R1 || !Name || !Key)
    SOS_VRETURN();

  MD5Init(&mdContext);
  MD5Update(&mdContext, R1, SOS_RAN_LEN);
  if (R2)
    MD5Update(&mdContext, R2, SOS_RAN_LEN);
  MD5Update(&mdContext, Name->str, Name->len);
  MD5Update(&mdContext, Key, strlen(Key));
  MD5Final(&mdContext);

  memcpy(hash, mdContext.digest, SOS_HASH_LEN);

  SOS_VRETURN();
}



/*
 * Run skid without having to know which peer you'll play.  We'll figure
 * that out for you.
 */
int
sos_skid(int fd, char *key,
	 int (*readf)(int, caddr_t, __SIZE_TYPE__),
	 int (*writef)(int, caddr_t, __SIZE_TYPE__))

{
  SOS_ENTRY("sos_skid","sos_skid",NULL);
  int ident;			/* Which peer will I play */

  if (!key || !readf || !writef)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  if ( (ident = sos_alice_bob(fd, readf, fd, writef)) < 0 )
    {
      sos_error_printf("Could not determine gender\n");
      SOS_RETURN(-1);
    }

  sos_debug_printf_and(1,"I am playing the role of %s\n", 
		   (ident==SOS_IDENT_ALICE)?"Alice":"Bob");

  SOS_RETURN(sos_skid_with_ident(fd, ident, key, readf, writef));
}



/*
 * This routine assumes that a higher routine has determined the role we 
 * play in SKID (ie are we `Alice' or `Bod', but that we are able to use
 * the connection to automatically determine both our own and our peer's
 * name
 */
int
sos_skid_with_ident(int fd, int mode, char *key,
		    int (*readf)(int, caddr_t, __SIZE_TYPE__),
		    int (*writef)(int, caddr_t, __SIZE_TYPE__))
{
  SOS_ENTRY("sos_skid","sos_skid_with_ident",NULL);
  struct sos_skid_protocol skidInfo;

  if (!key || !readf || !writef)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  /* Initialize structure */
  memset((char *)&skidInfo, 0, sizeof(skidInfo));

  /* Fill out name information */
  if ( skid_find_my_name(fd, mode, &skidInfo) < 0 ||
      skid_find_his_name(fd, mode, &skidInfo) < 0  )
    {
      sos_error_printf("A name function failed\n");
      SOS_RETURN(-1);
    }

  SOS_RETURN(sos_skid_with_ident_skidinfo(fd, mode, key, readf, writef,&skidInfo));
}



/*
 * Authenticate with strings for names
 *
 * If you use this function you are responsible for cleaning up your own
 * sos_strings.
 */
int
sos_skid_ident_usenames(int fd, int mode, char *key,
		 int (*readf)(int, caddr_t, __SIZE_TYPE__),
		 int (*writef)(int, caddr_t, __SIZE_TYPE__),
		 sos_string *localname, sos_string *peername)
{
  SOS_ENTRY("sos_skid","sos_skid_ident_usenames",NULL);
  struct sos_skid_protocol skidInfo;

  if (!localname || !peername || !key || !readf || !writef)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  /* Initialize structure */
  memset((char *)&skidInfo, 0, sizeof(skidInfo));

  if ( mode==SOS_IDENT_ALICE )
    {
      skidInfo.Alice = localname;
      skidInfo.Bob = peername;
    }
  else
    {
      skidInfo.Alice = peername;
      skidInfo.Bob = localname;
    }

  SOS_RETURN(sos_skid_with_ident_skidinfo(fd, mode, key, readf, writef,&skidInfo)); 
}



/* 
 * Asumes "Alice", "Bob" identify chosen (mode)
 * Builds sos_strings from passed in NULL terminated strings, and passes
 * info down to general SKID routined.
 */
int
sos_skid_ident_stringname(int fd, int mode, char *key,
		 int (*readf)(int, caddr_t, __SIZE_TYPE__),
		 int (*writef)(int, caddr_t, __SIZE_TYPE__),
		 char *localname, char *peername)
{
  SOS_ENTRY("sos_skid","sos_skid_ident_stringname",NULL);
  sos_string localn, peern;
  int ret_val;

  if (!localname || !peername || !key || !readf || !writef)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  localn.str = localname;
  localn.len = strlen(localname);
  peern.str = peername;
  peern.len = strlen(peername);

  ret_val = sos_skid_ident_usenames(fd, mode,key, readf, writef, &localn, &peern);

  sos_debug_printf_and(1,"SKID returing with %d\n", ret_val);

  SOS_RETURN(ret_val);
}



/*
 * Run skid without having to know which peer you'll play.  We'll figure
 * that out for you. You supply NULL terminated SKID names.
 */
int 
sos_skid_stringname(int fd, char *key,
		    int (*readf)(int, caddr_t, __SIZE_TYPE__),
		    int (*writef)(int, caddr_t, __SIZE_TYPE__),
		    char *localname, char *peername)
{
  SOS_ENTRY("sos_skid","sos_skid_stringname",NULL);
  int ident;			/* Which peer will I play */

  if (!localname || !peername || !key || !readf || !writef)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  if ( (ident = sos_alice_bob(fd, readf, fd, writef)) < 0 )
    {
      sos_error_printf("Could not determine gender\n");
      SOS_RETURN(-1);
    }

  sos_debug_printf_and(1,"I am playing the role of %s\n", 
		   (ident==SOS_IDENT_ALICE)?"Alice":"Bob");
  
  SOS_RETURN (sos_skid_ident_stringname(fd, ident, key, readf, writef, localname, peername));
}


#ifdef WANT_NOIO_SKID 

/*
 * I/O-less skid. All I/O  left to the callee.
 * This protocol is slightly modified from the RIPE specifications.
 * Returns 1 on complete success.
 * 	   0 on protocol failure.
 * 	   -1 on system or programmer failure
 */

int sos_skid_state(struct sos_protocol *proto)
{
  SOS_ENTRY("sos_skid", __FUNCTION__,NULL);
  struct sos_skid_protocol *skidInfo;
  int retval=1;
  unsigned char Remote_Hash[SOS_HASH_LEN];
  
  if ( !proto )
    {
      sos_error_printf("Invaild Argument");
      SOS_RETURN(-1);
    }

  
  if ( !proto->private_data )
    {
      sos_error_printf("No skidInfo?? Exiting\n");
      SOS_RETURN(-1);
    };

  skidInfo = (struct sos_skid_protocol *)(proto->private_data);

  if (proto->flags & (SOS_PROT_START) )
    {
      skidInfo->state = SOS_SKID_START;
      SOS_PROT_RESET_START(proto);
      sos_debug_printf_and(2,"Moving to skid start");
    }

  switch (skidInfo->state)
    {

    case SOS_SKID_START:
      sos_debug_printf_and(1,"Entering start");
      if (sos_get_rand((unsigned char *)&(skidInfo->Ra), SOS_RAN_LEN) < 1)
	{
	  sos_error_printf("Random numbers failed\n");
	  retval=-1;
	  break;
	}
      
      if ( (proto->buf = sos_xdr_sencode ("cccccccc",
					 skidInfo->Ra[0],
					 skidInfo->Ra[1],
					 skidInfo->Ra[2],
					 skidInfo->Ra[3],
					 skidInfo->Ra[4],
					 skidInfo->Ra[5],
					 skidInfo->Ra[6],
					 skidInfo->Ra[7] )) <= 0 )
	{
	  sos_error_printf("XDR encoding of local random number failed");
	  retval=-1;
	  break;
	}

      sos_debug_printf_and(64, "Local random number: %08x %08x\n", 
			   *((int *)(&(skidInfo->Ra[0]))),
			   *((int *)(&(skidInfo->Ra[4]))));

      /* Set output request */
      SOS_PROT_SET_OUTPUT_REQ(proto);
      sos_debug_printf_and(4,"Requesting local RN send");
      /* Move to new state */
      skidInfo->state=SOS_SKID_READ_PEER_RN;
      sos_debug_printf_and(2,"Moving to read peer RN state");
      break;


    case SOS_SKID_READ_PEER_RN:
      sos_debug_printf_and(1,"Entering read peer RN state");
      if ( ! (proto->flags & SOS_PROT_OUTPUT_REQ) )
	{
	  sos_error_printf("Bad protocol flag value: %d", proto->flags);
	  sos_debug_printf_and(1,"Expected to see output request, got: %d", proto->flags);
	  retval=-1;
	  break;
	}

      SOS_PROT_RESET_OUTPUT_REQ(proto);
      SOS_PROT_SET_INPUT_REQ(proto);
      sos_debug_printf_and(4,"Requesting peer random number read");

      skidInfo->state=SOS_SKID_LOCAL_HASH;
      sos_debug_printf_and(2,"Moving to local hash computation");
      break;


    case SOS_SKID_LOCAL_HASH:
      sos_debug_printf_and(1,"Entering local hash computation");
      
      if ( ! (proto->flags &= SOS_PROT_INPUT_REQ) )
	{
	  sos_error_printf("Bad protocol flag value: %d", proto->flags);
	  sos_debug_printf_and(1,"Expected to see input request, got: %d", proto->flags);
	  retval=-1;
	  break;
	}
      
      SOS_PROT_RESET_INPUT_REQ(proto);
      /* Decode the peer's random number and store it away */
      if ( sos_xdr_sdecode (proto->buf->str, "cccccccc", 
			    &skidInfo->Rb[0],
			    &skidInfo->Rb[1],
			    &skidInfo->Rb[2],
			    &skidInfo->Rb[3],
			    &skidInfo->Rb[4],
			    &skidInfo->Rb[5],
			    &skidInfo->Rb[6],
			    &skidInfo->Rb[7] ) != 8 )
	{
	  sos_error_printf("XDR decoding failed\n");
	  retval=-1;
	  break;
	}
      
      sos_debug_printf_and(64, "Remote random number: %08x %08x\n", 
			   *((int *)(&(skidInfo->Rb[0]))),
			   *((int *)(&(skidInfo->Rb[4]))));

      /* Compute Hb -- Peer's keyed hash */
      skid_hash(skidInfo->Hb, skidInfo->Rb, skidInfo->Ra, skidInfo->Bob, skidInfo->Key);

      /* Compute Ha - Local keyed hash */
      skid_hash(skidInfo->Ha, skidInfo->Ra, skidInfo->Rb, skidInfo->Alice, skidInfo->Key);

  sos_debug_printf_and(64,"SKID:\tKey: %s, Alice: %s, Bob: %s\n\tRa: %08x %08x, Rb: %08x %08x\n",
		   skidInfo->Key, skidInfo->Alice->str, skidInfo->Bob->str,
		   *((int *)(&(skidInfo->Ra[0]))),
		   *((int *)(&(skidInfo->Ra[4]))),
		   *((int *)(&(skidInfo->Rb[0]))),
		   *((int *)(&(skidInfo->Rb[4]))));
  sos_debug_printf_and(64,"\tHa: %08x %08x %08x %08x\n",
		   *((int *)(&(skidInfo->Ha[0]))),
		   *((int *)(&(skidInfo->Ha[4]))),
		   *((int *)(&(skidInfo->Ha[8]))),
		   *((int *)(&(skidInfo->Ha[12]))));
  sos_debug_printf_and(64,"\tHb: %08x %08x %08x %08x\n\n",
		   *((int *)(&(skidInfo->Hb[0]))),
		   *((int *)(&(skidInfo->Hb[4]))),
		   *((int *)(&(skidInfo->Hb[8]))),
		   *((int *)(&(skidInfo->Hb[12]))));


      if ( (proto->buf= sos_xdr_sencode("cccccccccccccccc",
					skidInfo->Ha[0],
					skidInfo->Ha[1],
					skidInfo->Ha[2],
					skidInfo->Ha[3],
					skidInfo->Ha[4],
					skidInfo->Ha[5],
					skidInfo->Ha[6],
					skidInfo->Ha[7],
					skidInfo->Ha[8],
					skidInfo->Ha[9],
					skidInfo->Ha[10],
					skidInfo->Ha[11],
					skidInfo->Ha[12],
					skidInfo->Ha[13],
					skidInfo->Ha[14],
					skidInfo->Ha[15] )) <= 0 )
	{
	  sos_error_printf("Encode of local keyed hash failed\n");
	  SOS_RETURN(-1);
	}

      SOS_PROT_SET_OUTPUT_REQ(proto);
      sos_debug_printf_and(4,"Sending local keyed hash");
      
      skidInfo->state = SOS_SKID_REMOTE_HASH;
      sos_debug_printf_and(2,"Moving to receiving remote keyed hash");
      break;


    case SOS_SKID_REMOTE_HASH:
      sos_debug_printf_and(1,"Entering receiving remote keyed hash");

      if ( ! (proto->flags &= SOS_PROT_OUTPUT_REQ) )
	{
	  sos_error_printf("Bad protocol flag value: %d\n", proto->flags);
	  sos_debug_printf_and(1,"Expected to see output request, got: %d", proto->flags);
	  retval=-1;
	  break;
	}

      SOS_PROT_RESET_OUTPUT_REQ(proto);
      SOS_PROT_SET_INPUT_REQ(proto);
      sos_debug_printf_and(4,"Requesting remote keyed hash");

      skidInfo->state = SOS_SKID_HASH_COMPARE;
      sos_debug_printf_and(2,"Moving to hash compare");
      break;



    case SOS_SKID_HASH_COMPARE:
      sos_debug_printf_and(1,"Entering hash compare");
      if ( ! (proto->flags &= SOS_PROT_INPUT_REQ) )
	{
	  sos_error_printf("Bad protocol flag value: %d", proto->flags);
	  sos_debug_printf_and(1,"Expected to see input request, got: %d", proto->flags);
	  retval=-1;
	  break;
	}

      SOS_PROT_RESET_INPUT_REQ(proto);
      
      if ( sos_xdr_sdecode(proto->buf->str, "cccccccccccccccc",
			   &Remote_Hash[0],
			   &Remote_Hash[1],
			   &Remote_Hash[2],
			   &Remote_Hash[3],
			   &Remote_Hash[4],
			   &Remote_Hash[5],
			   &Remote_Hash[6],
			   &Remote_Hash[7],
			   &Remote_Hash[8],
			   &Remote_Hash[9],
			   &Remote_Hash[10],
			   &Remote_Hash[11],
			   &Remote_Hash[12],
			   &Remote_Hash[13],
			   &Remote_Hash[14],
			   &Remote_Hash[15] ) != 16 )
	{
	  sos_error_printf("Wire decode of Ha failed\n");
	  retval=-1;
	  break;
	}

      /* Compare received value to computed remote value */
      if (memcmp(Remote_Hash,skidInfo->Hb,SOS_HASH_LEN))
	{
	  sos_debug_printf_and(64,"SKID: Failed--Hb != RemoteHash\n");
	  retval = 0;
	  break;
	}

      sos_debug_printf_and(64,"SKID: Succeeded!");
      SOS_PROT_SET_FINISHED(proto);
      sos_debug_printf_and(2,"Moving to finished state");
      break;



    default:
      sos_error_printf("SIKD in unknown state: %d\n", skidInfo->state);
      retval=-1;
      break;

    } /* End protocol states */

  SOS_RETURN(retval);
}

/*
 * Run sos_skid_state as blocking protocol 
 * With luck this will ultimately replace the current blocking skid engine.
 */  
int sos_BIO_skid_state(int fd, 
		       int (*readf)(int, caddr_t, __SIZE_TYPE__),
		       int (*writef)(int, caddr_t, __SIZE_TYPE__),
		       struct sos_protocol *proto)
{
  SOS_ENTRY("sos_skid", __FUNCTION__, NULL);
  int ret_val = 0;

  while ( !(proto->flags & SOS_PROT_FINISHED) )
    {
      ret_val=sos_skid_state (proto);
      if ( ret_val != 1 )
	break;

      if ( proto->flags & SOS_PROT_INPUT_REQ )
	{
	  sos_debug_printf_and(64, "reading in some data");
	  if ( sos_xdr_sget(fd, proto->buf, readf) < 0 )
	    {
	      ret_val = -1;
	      break;
	    }
	}
      else if ( proto->flags & SOS_PROT_OUTPUT_REQ )
	{
	  sos_debug_printf_and(64, "writing out some data");
	  if ( sos_xdr_sput(fd, proto->buf, writef) < 0 )
	    {
	      ret_val = -1;
	      break;
	    }
	}
    }
  
  SOS_RETURN(ret_val);
}

#endif /* WANT_NOIO_SKID */


