#if !defined(lint) && !defined(__INSIGHT__)
static char sos__rcsid[] = "$Id$";
static char sos__copyright[] = "Copyright (c) 1994, 1995, 1996 SOS Corporation";
static char sos__contact[] = "SOS Corporation <sos-info@soscorp.com> +1 800 SOS UNIX";
#endif /* not lint */

/*
 * ++Copyright Released Product++
 *
 * Copyright (c) 1994, 1995, 1996 Sources of Supply Corporation ("SOS").
 * All rights reserved.
 *
 * The SOS Released Product License Agreement specifies the terms and
 * conditions for redistribution.  You may find the License Agreement
 * in the file LICENSE.
 *
 * SOS Corporation
 * 461 5th Ave.; 16th floor
 * New York, NY 10017
 *
 * +1 800 SOS UNIX
 * <sos-info@soscorp.com>
 *
 * --Copyright Released Product--
 */

/*
 * Terminal handling stuff
 */

#include "sos.h"
#include <termios.h>


/*
 * Turn off echoing
 *
 * We turn off ECHO, but other flags, like ECHOE, ECHOK, and ECHONL, may be important
 *
 * Returns:	0  - not a tty
 *		1  - success
 *		-1 - error
 */
int
sos_tty_echooff(int fd, struct termios *termsave)
{
  SOS_ENTRY("sos_tty","sos_tty_echoon",NULL);
  struct termios old;
  int ret;

  if ( (ret=sos_tty_get(fd, &old) ) != 1 )
    SOS_RETURN(ret);

  old.c_lflag &= ~(ECHO);

  SOS_ORETURN(sos_tty_set(0, &old, termsave));

}



/*
 * Turn on echoing
 *
 * We turn on ECHO, but other flags, like ECHOE, ECHOK, and ECHONL, may be important
 *
 * Returns:	0  - not a tty
 *		1  - success
 *		-1 - error
 */
int 
sos_tty_echoon(int fd, struct termios *termsave)
{
  SOS_ENTRY("sos_tty","sos_tty_echoon",NULL);
  struct termios old;
  int ret;

  if ( (ret=sos_tty_get(fd, &old) ) != 1 )
    SOS_RETURN(ret);

  old.c_lflag |= (ECHO);

  SOS_ORETURN(sos_tty_set(0, &old, termsave));

}


/*
 * Enter 7 bit mode. Use other tty functions as aids. 
 * Returns:	0  - not a tty
 *		1  - success
 *		-1 - error
 */
int 
sos_tty_7bit(int fd, struct termios *termsave)
{
  SOS_ENTRY("sos_tty",__FUNCTION__,NULL);
  struct termios old;
  int ret;

  if ( (ret=sos_tty_get(fd, &old) ) != 1 )
    SOS_RETURN(ret);

  old.c_cflag &= ~CSIZE;
  old.c_cflag |= CS7;

  SOS_ORETURN(sos_tty_set(0, &old, termsave));
}

/*
 * Enter 7 bit mode. Use other tty functions as aids. 
 * Returns:	0  - not a tty
 *		1  - success
 *		-1 - error
 */
int 
sos_tty_8bit(int fd, struct termios *termsave)
{
  SOS_ENTRY("sos_tty",__FUNCTION__,NULL);
  struct termios old;
  int ret;

  if ( (ret=sos_tty_get(fd, &old) ) != 1 )
    SOS_RETURN(ret);

  old.c_cflag &= ~CSIZE;
  old.c_cflag |= CS8;

  SOS_ORETURN(sos_tty_set(0, &old, termsave));
}



/*
 * Turn off raw mode
 *
 * We turn off RAW
 *
 * Returns:	0  - not a tty
 *		1  - success
 *		-1 - error
 */
int 
sos_tty_rawoff(int fd, struct termios *termsave)
{
  SOS_ENTRY("sos_tty","sos_tty_rawoff",NULL);
  struct termios old;
  int ret;

  if ( (ret=sos_tty_get(fd, &old) ) != 1 )
    SOS_RETURN(ret);

  old.c_iflag |= (BRKINT|IGNPAR|ISTRIP|ICRNL|IXON);
  old.c_oflag |= (OPOST);
  old.c_lflag |= (ISIG|ICANON);
  if (VEOF == VMIN)
    old.c_cc[VEOF] = 4;
  if (VEOL == VTIME)
    old.c_cc[VEOL] = 0;

  SOS_ORETURN(sos_tty_set(0, &old, termsave));
}


/* 
 * Turn on ICANON (cooked mode).
 *
 * Returns:	0  - not a tty
 *		1  - success
 *		-1 - error
 */
int 
sos_tty_icanonon(int fd, struct termios *termsave)
{
  SOS_ENTRY("sos_tty",__FUNCTION__,NULL);
  struct termios old;
  int ret;

  if ( (ret=sos_tty_get(fd, &old) ) != 1 )
    SOS_RETURN(ret);

  old.c_lflag |= ICANON;

  SOS_ORETURN(sos_tty_set(0, &old, termsave));
}


/* 
 * Turn on ICANON (cooked mode).
 *
 * Returns:	0  - not a tty
 *		1  - success
 *		-1 - error
 */
int 
sos_tty_icanonoff(int fd, struct termios *termsave)
{
  SOS_ENTRY("sos_tty",__FUNCTION__,NULL);
  struct termios old;
  int ret;

  if ( (ret=sos_tty_get(fd, &old) ) != 1 )
    SOS_RETURN(ret);

  old.c_lflag &= ~(ICANON);

  SOS_ORETURN(sos_tty_set(0, &old, termsave));
}



/*
 * Turn on raw mode
 *
 * We turn on RAW
 *
 * Returns:	0  - not a tty
 *		1  - success
 *		-1 - error
 */
int 
sos_tty_rawon(int fd, struct termios *termsave)
{
  SOS_ENTRY("sos_tty","sos_tty_rawon",NULL);
  struct termios old;
  int ret;

  if ( (ret=sos_tty_get(fd, &old) ) != 1 )
    SOS_RETURN(ret);

  old.c_iflag &= ~(SOS_ANTIRAW);
  old.c_oflag &= ~(OPOST);
  old.c_lflag &= ~(ISIG|ICANON);
  old.c_cc[VMIN] = 1;
  old.c_cc[VTIME] = 0;

  SOS_ORETURN(sos_tty_set(0, &old, termsave));
}



/*
 * Set the termios to a particular value
 *
 * Returns:	0  - not a tty
 *		1  - success
 *		-1 - error
 */
int 
sos_tty_set(int fd, struct termios *termnew, struct termios *termsave)
{
  SOS_ENTRY("sos_tty","sos_tty_set",NULL);
  struct termios old;

  /* It legal for *either* termnew or termsave to be null, but both is odd. */
  if ( !termnew && !termsave )
    {
      sos_error_printf("Unusual arguments (not illegal)\n");
      /* So just go on */
    }
  
  if (termsave)
    {
      memset((void *)termsave, 0, sizeof(struct termios));
    }

  if (!isatty(fd))
    {
      SOS_RETURN(0);
    }

  if (tcgetattr(fd, &old) < 0)
    {
      sos_error_printf("Could not tcgetaddr(%d): %s\n",fd,strerror(errno));
      SOS_RETURN(-1);
    }

  if (termsave)
    {
      memcpy((void *)termsave, (void *)&old, sizeof(old));
    }

  if ( termnew && tcsetattr(fd, TCSANOW, termnew) < 0)
    {
      sos_error_printf("Could not tcsetaddr(%d): %s\n",fd,strerror(errno));
      SOS_RETURN(-1);
    }

  SOS_RETURN(1);
}
